#!/usr/bin/python2
# -----------------------------------------
# Sextant
# Copyright 2014, Ensoft Ltd.
# Author: Patrick Stevens, James Harkin
# -----------------------------------------
# Note: this must be run in Python 2.

from twisted.web.server import Site, NOT_DONE_YET
from twisted.web.resource import Resource
from twisted.web.static import File
from twisted.internet import reactor
from twisted.internet.threads import deferToThread
from twisted.internet import defer

import logging
import os
import json
import requests

import sextant.db_api as db_api
import sextant.export as export
import tempfile
import subprocess

from cgi import escape  # deprecated in Python 3 in favour of html.escape, but we're stuck on Python 2

database_url = None  # the URL to access the database instance

class Echoer(Resource):
    # designed to take one name argument

    def render_GET(self, request):
        if "name" not in request.args:
            return '<html><body>Greetings, unnamed stranger.</body></html>'

        arg = escape(request.args["name"][0])
        return '<html><body>Hello %s!</body></html>' % arg


class SVGRenderer(Resource):

    def error_creating_neo4j_connection(self, failure):
        self.write("Error creating Neo4J connection: %s\n") % failure.getErrorMessage()

    @staticmethod
    def create_neo4j_connection():
        return db_api.SextantConnection(database_url)

    @staticmethod
    def check_program_exists(connection, name):
        return connection.check_program_exists(name)

    @staticmethod
    def get_whole_program(connection, name):
        return connection.get_whole_program(name)

    @staticmethod
    def get_functions_calling(connection, progname, funcname):
        return connection.get_all_functions_calling(progname, funcname)

    @staticmethod
    def get_plot(program, suppress_common_functions=False):
        graph_dot = export.ProgramConverter.to_dot(program, suppress_common_functions)

        file_written_to = tempfile.NamedTemporaryFile(delete=False)
        file_out = tempfile.NamedTemporaryFile(delete=False)
        file_written_to.write(graph_dot)
        file_written_to.close()
        subprocess.call(['dot', '-Tsvg', '-Kdot', '-o', file_out.name, file_written_to.name])

        output = file_out.read().encode()
        file_out.close()
        return output

    @defer.inlineCallbacks
    def _render_plot(self, request):
        if "program_name" not in request.args:
            request.setResponseCode(400)
            request.write("Supply 'program_name' parameter.")
            request.finish()
            defer.returnValue(None)

        logging.info('enter')
        name = request.args["program_name"][0]

        try:
            suppress_common = request.args["suppress_common"][0]
        except KeyError:
            suppress_common = False

        if suppress_common == 'null' or suppress_common == 'true':
            suppress_common = True
        else:
            suppress_common = False

        try:
            neo4jconnection = yield deferToThread(self.create_neo4j_connection)
        except requests.exceptions.ConnectionError:
            request.setResponseCode(502)  # Bad Gateway
            request.write("Could not reach Neo4j server at {}".format(database_url))
            request.finish()
            defer.returnValue(None)
            neo4jconnection = None  # to silence the "referenced before assignment" warnings later

        logging.info('created')
        exists = yield deferToThread(self.check_program_exists, neo4jconnection, name)
        if not exists:
            request.setResponseCode(404)
            logging.info('returning nonexistent')
            request.write("Name %s not found." % (escape(name)))
            request.finish()
            defer.returnValue(None)

        logging.info('done created')
        allowed_queries = ("whole_program", "functions_calling", "functions_called_by", "call_paths", "shortest_path")

        if "query" not in request.args:
            query = "whole_program"
        else:
            query = request.args["query"][0]

        if query not in allowed_queries:
            # raise 400 Bad Request error
            request.setResponseCode(400)
            request.write("Supply 'query' parameter, default is whole_program, allowed %s." % str(allowed_queries))
            request.finish()
            defer.returnValue(None)

        if query == 'whole_program':
            program = yield deferToThread(self.get_whole_program, neo4jconnection, name)
        elif query == 'functions_calling':
            if 'func1' not in request.args:
                # raise 400 Bad Request error
                request.setResponseCode(400)
                request.write("Supply 'func1' parameter to functions_calling.")
                request.finish()
                defer.returnValue(None)
            func1 = request.args['func1'][0]
            program = yield deferToThread(self.get_functions_calling, neo4jconnection, name, func1)
        elif query == 'functions_called_by':
            if 'func1' not in request.args:
                # raise 400 Bad Request error
                request.setResponseCode(400)
                request.write("Supply 'func1' parameter to functions_called_by.")
                request.finish()
                defer.returnValue(None)
            func1 = request.args['func1'][0]
            program = yield deferToThread(neo4jconnection.get_all_functions_called, name, func1)
        elif query == 'call_paths':
            if 'func1' not in request.args:
                # raise 400 Bad Request error
                request.setResponseCode(400)
                request.write("Supply 'func1' parameter to call_paths.")
                request.finish()
                defer.returnValue(None)
            if 'func2' not in request.args:
                # raise 400 Bad Request error
                request.setResponseCode(400)
                request.write("Supply 'func2' parameter to call_paths.")
                request.finish()
                defer.returnValue(None)

            func1 = request.args['func1'][0]
            func2 = request.args['func2'][0]
            program = yield deferToThread(neo4jconnection.get_call_paths, name, func1, func2)
        elif query == 'shortest_path':
            if 'func1' not in request.args:
                # raise 400 Bad Request error
                request.setResponseCode(400)
                request.write("Supply 'func1' parameter to shortest_path.")
                request.finish()
                defer.returnValue(None)
            if 'func2' not in request.args:
                # raise 400 Bad Request error
                request.setResponseCode(400)
                request.write("Supply 'func2' parameter to shortest_path.")
                request.finish()
                defer.returnValue(None)

            func1 = request.args['func1'][0]
            func2 = request.args['func2'][0]
            program = yield deferToThread(neo4jconnection.get_shortest_path_between_functions, name, func1, func2)

        else:  # unrecognised query, so we need to raise a Bad Request response
            request.setResponseCode(400)
            request.write("Query %s not recognised." % escape(query))
            request.finish()
            defer.returnValue(None)
            program = None  # silences the "referenced before assignment" warnings

        if program is None:
            request.setResponseCode(404)
            request.write("At least one of the input functions was not found in program %s." % (escape(name)))
            request.finish()
            defer.returnValue(None)

        logging.info('getting plot')
        logging.info(program)
        if not program.functions:  # we got an empty program back:  the program is in the Sextant but has no functions
            request.setResponseCode(204)
            request.finish()
            defer.returnValue(None)

        output = yield deferToThread(self.get_plot, program, suppress_common)
        request.setHeader("content-type", "image/svg+xml")

        logging.info('SVG: return')
        request.write(output)
        request.finish()

    def render_GET(self, request):
        self._render_plot(request)
        return NOT_DONE_YET


class GraphProperties(Resource):

    @staticmethod
    def _get_connection():
        return db_api.SextantConnection(database_url)

    @staticmethod
    def _get_program_names(connection):
        return connection.get_program_names()

    @staticmethod
    def _get_function_names(connection, program_name):
        return connection.get_function_names(program_name)

    @defer.inlineCallbacks
    def _render_GET(self, request):
        if "query" not in request.args:
            request.setResponseCode(400)
            request.setHeader("content-type", "text/plain")
            request.write("Supply 'query' parameter of 'programs' or 'functions'.")
            request.finish()
            defer.returnValue(None)

        query = request.args['query'][0]

        logging.info('Properties: about to get_connection')

        try:
            neo4j_connection = yield deferToThread(self._get_connection)
        except Exception:
            request.setResponseCode(502)  # Bad Gateway
            request.write("Could not reach Neo4j server at {}.".format(database_url))
            request.finish()
            defer.returnValue(None)
            neo4j_connection = None  # just to silence the "referenced before assignment" warnings

        logging.info('got connection')

        if query == 'programs':
            request.setHeader("content-type", "application/json")
            prognames = yield deferToThread(self._get_program_names, neo4j_connection)
            request.write(json.dumps(list(prognames)))
            request.finish()
            defer.returnValue(None)

        elif query == 'functions':
            if "program_name" not in request.args:
                request.setResponseCode(400)
                request.setHeader("content-type", "text/plain")
                request.write("Supply 'program_name' parameter to ?query=functions.")
                request.finish()
                defer.returnValue(None)
            program_name = request.args['program_name'][0]

            funcnames = yield deferToThread(self._get_function_names, neo4j_connection, program_name)
            if funcnames is None:
                request.setResponseCode(404)
                request.setHeader("content-type", "text/plain")
                request.write("No program with name %s was found in the Sextant." % escape(program_name))
                request.finish()
                defer.returnValue(None)

            request.setHeader("content-type", "application/json")
            request.write(json.dumps(list(funcnames)))
            request.finish()
            defer.returnValue(None)

        else:
            request.setResponseCode(400)
            request.setHeader("content-type", "text/plain")
            request.write("'Query' parameter should be 'programs' or 'functions'.")
            request.finish()
            defer.returnValue(None)

    def render_GET(self, request):
        self._render_GET(request)
        return NOT_DONE_YET


def serve_site(input_database_url='http://localhost:7474', port=2905):

    global database_url
    database_url = input_database_url
    # serve static directory at root
    root = File(os.path.join(
        os.path.dirname(os.path.abspath(__file__)),
        "..", "..", "resources", "sextant", "web"))

    # serve a dynamic Echoer webpage at /echoer.html
    root.putChild("echoer.html", Echoer())

    # serve a dynamic webpage at /Sextant_properties to return graph properties
    root.putChild("database_properties", GraphProperties())

    # serve a generated SVG at /output_graph.svg
    root.putChild('output_graph.svg', SVGRenderer())

    factory = Site(root)
    reactor.listenTCP(port, factory)
    reactor.run()

if __name__ == '__main__':
    serve_site()
