# -----------------------------------------
# Sextant
# Copyright 2014, Ensoft Ltd.
# Author: Patrick Stevens
# -----------------------------------------
# Define 'environment' variables for 'where we are located'.

"""
Defines standard Sextant locations.
"""

__all__ = ("RESOURCES_DIR", "DEFAULT_CONFIG_FILE", "ENV_CONFIG_FILE", 
           "HOME_CONFIG_FILE", "load_config")

import os
import logging
import logging.config
import collections

try:
    import ConfigParser
except ImportError:
    import configparser as ConfigParser


RESOURCES_DIR = ""         # where the resources folder is located
DEFAULT_CONFIG_FILE = ""  # where the bundled config file is located
ENV_CONFIG_FILE = "SEXTANT_CONFIG"  # name of the config-file env var to seek
# path to user-specific configuration file to seek
HOME_CONFIG_FILE = os.path.expanduser(os.path.join("~", ".sextant.conf"))


def _find_folder(name):
    """
    Given a folder name, returns the absolute path to that folder.
    If the folder could not be found in the standard installation places,
    raises IOError.
    """
    path = os.path.abspath(os.path.dirname(__file__))
    trial = path
    for level in range(2):  # search both .. and ../..
        trial = os.path.join(trial, '..')
        if os.path.isdir(os.path.join(trial, name)):
            return os.path.abspath(os.path.join(trial, name))

    raise IOError("Could not find {} folder.".format(name))


RESOURCES_DIR = _find_folder('resources')
_conf_path = _find_folder('etc')
DEFAULT_CONFIG_FILE = os.path.join(_conf_path, 'sextant.conf')


Config = collections.namedtuple('Config', ['web_port', 'remote_neo4j',
                                           'common_cutoff', 'use_ssh_tunnel',
                                           'ssh_user'])


def load_config(check_here=''):
    """
    Loads in the config file (checking in the various locations).

    If check_here is supplied as a location for the config file, it is used
    preferentially. Then, in order, we use HOME_CONFIG_FILE, the contents of
    the SEXTANT_CONFIG environment variable, and finally the bundled
    etc/sextant.conf if no other config file was found.

    :return: configuration object with properties for each config option
    """

    # @@@ Logging level should be configurable (with some structure to setting up
    # logging).
    logging.config.dictConfig({
        "version": 1,
        "handlers": {
            "console": {
                "class": "logging.StreamHandler",
                "level": logging.INFO,
                "stream": "ext://sys.stderr",
            },
        },
        "root": {
            "level": logging.DEBUG,
            "handlers": ["console"],
        },
    })
    log = logging.getLogger()

    def get_config_file(check=''):
        env_config = os.environ.get(ENV_CONFIG_FILE, "")
        try:
            conffile = next(p
                            for p in (check, HOME_CONFIG_FILE,
                                      env_config, DEFAULT_CONFIG_FILE)
                            if os.path.exists(p))
        except StopIteration:
            # No config files accessible
            if ENV_CONFIG_FILE in os.environ:
                # SEXTANT_CONFIG environment variable is set
                log.error("{} file %r doesn't exist.".format(ENV_CONFIG_FILE), env_config)
            raise Exception("Sextant requires a configuration file.")

        log.info("Sextant is using config file {}".format(conffile))
        return conffile

    conf_file = get_config_file(check_here)

    conf = ConfigParser.ConfigParser({'remote_neo4j': 'http://localhost:7474',
                                      'port': '2905',
                                      'common_function_calls': '10',
                                      'use_ssh_tunnel': 'True',
                                      'ssh_username': ''})
    # It turns out that getint, getboolean etc want the defaults to be strings.
    # When getboolean is called and the default is not a string, and the
    # default is used, it complains that it can't work with non-strings.
    # Ultimately, however, we output an object with int common_function_calls
    # and port.

    conf.read(conf_file)

    remote_neo4j = ''
    web_port = 0
    common_def = 0  # definition of a 'common' node
    use_ssh_tunnel = True
    ssh_user = ''
    try:
        conf.options('Neo4j')
    except ConfigParser.NoSectionError:
        pass
    else:
        remote_neo4j = conf.get('Neo4j', 'remote_neo4j')
        web_port = conf.getint('Neo4j', 'port')
        common_def = conf.getint('Neo4j', 'common_function_calls')
        use_ssh_tunnel = conf.getboolean('Neo4j', 'use_ssh_tunnel')
        ssh_user = conf.get('Neo4j', 'ssh_username')

    return Config(remote_neo4j=remote_neo4j, web_port=web_port,
                  common_cutoff=common_def, use_ssh_tunnel=use_ssh_tunnel,
                  ssh_user=ssh_user)

