# -----------------------------------------
# Sextant
# Copyright 2014, Ensoft Ltd.
# Author: James Harkin, using work from Patrick Stevens and James Harkin
# -----------------------------------------
#invokes Sextant and argparse

from __future__ import absolute_import, print_function

import argparse
try:
    import ConfigParser
except ImportError:
    import configparser as ConfigParser
import logging
import logging.config
import os
import sys

from . import update_db
from . import query
from . import db_api


# @@@ Logging level should be configurable (with some structure to setting up
# logging).
logging.config.dictConfig({
    "version": 1,
    "handlers": {
        "console": {
            "class": "logging.StreamHandler",
            "level": logging.INFO,
            "stream": "ext://sys.stderr",
        },
    },
    "root": {
        "level": logging.DEBUG,
        "handlers": ["console"],
    },
})
log = logging.getLogger()


def get_config_file():
    # get the config file option for neo4j server location and web port number
    _ROOT = os.path.abspath(os.path.dirname(__file__))
    def get_data(path, file_name):
        return os.path.join(_ROOT, path, file_name)

    home_config = os.path.expanduser(os.path.join("~", ".sextant.conf"))
    env_config = os.environ.get("SEXTANT_CONFIG", "")
    example_config = get_data('../etc', 'sextant.conf')

    try:
        conffile = next(p
                        for p in (home_config, env_config, example_config)
                        if os.path.exists(p))
    except StopIteration:
        #No config files accessable
        if "SEXTANT_CONFIG" in os.environ:
            #SEXTANT_CONFIG environment variable is set
            log.error("SEXTANT_CONFIG file %r doesn't exist.", env_config)
        log.error("Sextant requires a configuration file.")
        sys.exit(1)

    log.info("Sextant is using config file %s", conffile)
    return conffile

conffile = get_config_file()

conf = ConfigParser.ConfigParser()
conf.read(conffile)

#remote_neo4j = 'http://localhost:7474'
#web_port = 2905
#common_def = 10  # definition of a 'common' node
try:
    options = conf.options('Neo4j')
except ConfigParser.NoSectionError:
    pass
else:
    try:
        remote_neo4j = conf.get('Neo4j', 'remote_neo4j')
    except ConfigParser.NoOptionError:
        pass

    try:
        web_port = conf.get('Neo4j', 'port')
    except ConfigParser.NoOptionError:
        pass

    try:
        common_def = conf.get('Neo4j', 'common_function_calls')
    except ConfigParser.NoOptionError:
        common_def = 10

argumentparser = argparse.ArgumentParser(description="Invoke part of the SEXTANT program")
subparsers = argumentparser.add_subparsers(title="subcommands")

#set what will be defined as a "common function"
db_api.set_common_cutoff(common_def)

parsers = dict()

# add each subparser in turn to the parsers dictionary

parsers['add'] = subparsers.add_parser('add_program', help="add a program to the database")
parsers['add'].add_argument('--input-file', required=True, metavar="FILE_NAME",
                            help="name of file to be put into database",
                            type=str, nargs=1)
parsers['add'].add_argument('--set-file-name', metavar="FILE_NAME",
                            help="string to store this program under", type=str,
                            nargs=1)
parsers['add'].add_argument('--not-object-file',
                            help='default False, if the input file is an object to be disassembled',
                            action='store_true')

parsers['delete'] = subparsers.add_parser('delete_program',
                                          help="delete a program from the database")
parsers['delete'].add_argument('--program-name', required=True, metavar="PROG_NAME",
                               help="name of program as stored in the database",
                               type=str, nargs=1)

parsers['query'] = subparsers.add_parser('query',
                                         help="make a query of the database")
parsers['query'].add_argument('--program-name', metavar="PROG_NAME",
                              help="name of program as stored in the database",
                              type=str, nargs=1)
parsers['query'].add_argument('--query', required=True, metavar="QUERY",
                              help="functions-calling, functions-called-by, "
                                   "calls-between, whole-graph, shortest-path, "
                                   "return-all-program-names or "
                                   "return-all-function-names; if the latter, "
                                   "supply argument --program-name",
                              type=str, nargs=1)
parsers['query'].add_argument('--funcs', metavar='FUNCS',
                              help='functions to pass to the query',
                              type=str, nargs='+')
parsers['query'].add_argument('--suppress-common', metavar='BOOL',
                              help='suppress commonly called functions (True or False)',
                              type=str, nargs=1)

parsers['web'] = subparsers.add_parser('web', help="start the web server")
parsers['web'].add_argument('--port', metavar='PORT', type=int,
                            help='port on which to serve Sextant Web',
                            default=web_port)

for parser_key in parsers:
    parsers[parser_key].add_argument('--remote-neo4j', metavar="URL", nargs=1,
                                     help="URL of neo4j server", type=str,
                                     default=remote_neo4j)

def _start_web(args):
    # Don't import at top level -- this makes twisted dependency semi-optional,
    # allowing non-web functionality to work with Python 3.
    from .web import server
    log.info("Serving site on port %s", args.port)
    server.serve_site(input_database_url=args.remote_neo4j, port=args.port)

parsers['web'].set_defaults(func=_start_web)

def add_file(namespace):

    try:
        alternative_name = namespace.set_file_name[0]
    except TypeError:
        alternative_name = None

    not_object_file = namespace.not_object_file
    # the default is "yes, this is an object file" if not-object-file was
    # unsupplied
    
    update_db.upload_program(namespace.input_file[0],
                             namespace.remote_neo4j,
                             alternative_name, not_object_file)


def delete_file(namespace):
    update_db.delete_program(namespace.program_name[0],
                             namespace.remote_neo4j)

parsers['add'].set_defaults(func=add_file)
parsers['delete'].set_defaults(func=delete_file)


def make_query(namespace):

    arg1 = None
    arg2 = None
    try:
        arg1 = namespace.funcs[0]
        arg2 = namespace.funcs[1]
    except TypeError:
        pass

    try:
        program_name = namespace.program_name[0]
    except TypeError:
        program_name = None

    try:
        suppress_common = namespace.suppress_common[0]
    except TypeError:
        suppress_common = False

    query.query(remote_neo4j=namespace.remote_neo4j,
                input_query=namespace.query[0],
                program_name=program_name, argument_1=arg1, argument_2=arg2,
                suppress_common=suppress_common)

parsers['query'].set_defaults(func=make_query)

# parse the arguments

parsed = argumentparser.parse_args()

# call the appropriate function

parsed.func(parsed)

