# -----------------------------------------
# Sextant
# Copyright 2014, Ensoft Ltd.
# Author: Patrick Stevens,  James Harkin
# -----------------------------------------
# Convert a program from internal Python representation to various output formats

__all__ = "ProgramConverter"


class ProgramConverter:
    # Given our internal program representation, converts it to output formats.
    # Currently supported: GraphML

    @staticmethod
    def get_supported_outputs():
        return ['graphml', 'yed_graphml', 'dot']

    @staticmethod
    def get_display_name(function, suppress_common_nodes=False):
        """
        Given a Function object, retrieve the label we attach to it.
        For instance, function-pointers are labelled "(function pointer)",
        while the `main` function is usually labelled "main".
        """
        if function.type == "function_pointer":
            name = "(function pointer)"
        else:
            name = function.name
        
        if suppress_common_nodes and function.is_common:
            name += ' (common)'
        return name

    @staticmethod
    def to_dot(program, suppress_common_nodes=False):
        """
        Convert the program to DOT output format.
        """
        output_str = 'digraph "{}" {{\n '.format(program.program_name)
        output_str += 'overlap=false; \n'

        font_name = "helvetica"

        for func in program.get_functions():
            if func.type == "plt_stub":
                output_str += ' "{}" [fillcolor=pink, style=filled]\n'.format(func.name)
            elif func.type == "function_pointer":
                output_str += ' "{}" [fillcolor=yellow, style=filled]\n'.format(func.name)

            # in all cases, even if we've specified that we want a filled-in
            # node already, DOT lets us add more information about that node
            # so we can insist on turning that same node into a box-shape
            # and changing its font.
            output_str += ' "{}" [label="{}", fontname="{}", shape=box]\n'.format(func.name,
                                                                                  ProgramConverter.get_display_name(func, suppress_common_nodes),
                                                                                  font_name)
            if func.is_common:
                output_str += ' "{}" [fillcolor=lightgreen, style=filled]\n'.format(func.name)

            for func_called in func.functions_i_call:
                if not (suppress_common_nodes and func_called.is_common):
                    output_str += ' "{}" -> "{}"\n'.format(func.name, func_called.name)

        output_str += '}'
        return output_str



    @staticmethod
    def to_yed_graphml(program, suppress_common_nodes=False):
        commonly_called = []
        output_str = """<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<graphml xmlns="http://graphml.graphdrawing.org/xmlns" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" \
xmlns:y="http://www.yworks.com/xml/graphml" xmlns:yed="http://www.yworks.com/xml/yed/3" \
xsi:schemaLocation="http://graphml.graphdrawing.org/xmlns http://www.yworks.com/xml/schema/graphml/1.1/ygraphml.xsd">
"""
        output_str += """<key for="graphml" id="d0" yfiles.type="resources"/>
                        <key for="port" id="d1" yfiles.type="portgraphics"/>
                        <key for="port" id="d2" yfiles.type="portgeometry"/>
                        <key for="port" id="d3" yfiles.type="portuserdata"/>
                        <key attr.name="url" attr.type="string" for="node" id="d4"/>
                        <key attr.name="description" attr.type="string" for="node" id="d5"/>
                        <key for="node" id="d6" yfiles.type="nodegraphics"/>
                        <key attr.name="url" attr.type="string" for="edge" id="d7"/>
                        <key attr.name="description" attr.type="string" for="edge" id="d8"/>
                        <key for="edge" id="d9" yfiles.type="edgegraphics"/>\n"""

        output_str += """<graph id="{}" edgedefault="directed">\n""".format(program.program_name)

        for func in program.get_functions():
            display_func = ProgramConverter.get_display_name(func)
            if func.type == "plt_stub":
                colour = "#ff00ff"
            elif func.type == "function_pointer":
                colour = "#99ffff"
            elif func.is_common:
                colour = "#00FF00"
            else:
                colour = "#ffcc00"
            output_str += """<node id="{}">
                                <data key="d6">
                                    <y:ShapeNode>
                                        <y:Geometry height="{}" width="{}" x="60.0" y="0.0"/>
                                      <y:Fill color="{}" transparent="false"/>
                                      <y:BorderStyle color="#000000" type="line" width="1.0"/>
                                      <y:NodeLabel alignment="center" autoSizePolicy="content" fontFamily="Dialog"
                                      fontSize="12" fontStyle="plain" hasBackgroundColor="false" hasLineColor="false"
                                      height="18.701171875" modelName="custom" textColor="#000000" visible="true"
                                      width="36.6953125" x="-3.34765625" y="5.6494140625">{}<y:LabelModel>
                                          <y:SmartNodeLabelModel distance="4.0"/>
                                        </y:LabelModel>
                                        <y:ModelParameter>
                                          <y:SmartNodeLabelModelParameter labelRatioX="0.0" labelRatioY="0.0"
                                          nodeRatioX="0.0" nodeRatioY="0.0" offsetX="0.0" offsetY="0.0" upX="0.0"
                                          upY="-1.0"/>
                                        </y:ModelParameter>
                                      </y:NodeLabel>
                                      <y:Shape type="rectangle"/>
                                    </y:ShapeNode>
                                </data> 
                                </node>\n""".format(func.name, 20, len(display_func)*8, colour, display_func)
            for callee in func.functions_i_call:
                if callee not in commonly_called:
                    if not(suppress_common_nodes and callee.is_common):
                            output_str += """<edge source="{}" target="{}"> <data key="d9">
                                        <y:PolyLineEdge>
                                        <y:LineStyle color="#000000" type="line" width="1.0"/>
                                        <y:Arrows source="none" target="standard"/>
                                        <y:BendStyle smoothed="false"/>
                                        </y:PolyLineEdge>
                                        </data>
                                     </edge>\n""".format(func.name, callee.name)

        output_str += '</graph>\n<data key="d0">    <y:Resources/>  </data>\n</graphml>'
        return output_str

    @staticmethod
    def to_graphml(program):
        output_str = """<?xml version="1.0" encoding="UTF-8"?>
<graphml xmlns="http://graphml.graphdrawing.org/xmlns" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" \
xsi:schemaLocation="http://graphml.graphdrawing.org/xmlns http://graphml.graphdrawing.org/xmlns/1.0/graphml.xsd">
"""
        output_str += '<graph id="{}" edgedefault="directed">\n'.format(program.program_name)

        for func in program.get_functions():
            output_str += '<node id="{}"> <data key="name">{}</data> </node>\n'.format(func.name, ProgramConverter.get_display_name(func))
            for callee in func.functions_i_call:
                output_str += '<edge source="{}" target="{}"> <data key="calls">1</data> </edge>\n'.format(func.name, callee.name)

        output_str += '</graph>\n</graphml>'
        return output_str