# -----------------------------------------
# Sextant
# Copyright 2014, Ensoft Ltd.
# Author: James Harkin, Patrick Stevens
# -----------------------------------------
#API for performing queries on the database

#!/usr/bin/python3
import argparse
import requests, urllib  # for different kinds of exception
import logging

from . import db_api
from .export import ProgramConverter


def query(remote_neo4j, input_query, display_neo4j='', program_name=None,
          argument_1=None, argument_2=None, suppress_common=False):
    """
    Run a query against the database at remote_neo4j.

    input_query may be in ('functions-calling', 'functions-called-by',
    'all-call-paths', 'whole-program', 'shortest-call-path', 'functions',
    'programs').

    If we wish to display a different URL to the user than remote_neo4j,
    display_neo4j should be used for that different URL.

    :param remote_neo4j: location to sent HTTP requests to
    :param input_query: string as detailed above
    :param display_neo4j: Neo4j URL to display in messages to the user
    :param program_name: string name of program
    :param argument_1: string argument 1 to the query
    :param argument_2: string argument 2 to the query
    :param suppress_common: whether or not to suppress links to common functions

    """

    if display_neo4j:
        display_url = display_neo4j
    else:
        display_url = remote_neo4j

    try:
        db = db_api.SextantConnection(remote_neo4j)
    except requests.exceptions.ConnectionError as err:
        logging.error("Could not connect to Neo4J server {}. Are you sure it is running?".format(display_url))
        logging.error(str(err))
        return 2
    #Not supported in python 2
    #except (urllib.exceptions.MaxRetryError):
     #   logging.error("Connection was refused to {}. Are you sure the server is running?".format(remote_neo4j))
      #  return 2
    except Exception as err:
        logging.exception(str(err))
        return 2

    prog = None
    names_list = None
    #If a function calls itself we don't want to output these calls in the
    # shortest path query if remove_self_calls is True don't output a nodes
    #calls to itself
    remove_self_calls = False

    if input_query == 'functions-calling':
        if argument_1 is None:
            print('Supply one function name to functions-calling.')
            return 1
        prog = db.get_all_functions_calling(program_name, argument_1)
    elif input_query == 'functions-called-by':
        if argument_1 is None:
            print('Supply one function name to functions-called-by.')
            return 1
        prog = db.get_all_functions_called(program_name, argument_1)
    elif input_query == 'all-call-paths':
        if argument_1 is None and argument_2 is None:
            print('Supply two function names to calls-between.')
            return 1
        prog = db.get_call_paths(program_name, argument_1, argument_2)
    elif input_query == 'whole-program':
        prog = db.get_whole_program(program_name)
    elif input_query == 'shortest-call-path':
        if argument_1 is None and argument_2 is None:
            print('Supply two function names to shortest-path.')
            return 1
        prog = db.get_shortest_path_between_functions(program_name, argument_1, argument_2)
    elif input_query == 'functions':
        if program_name is not None:
            func_names = db.get_function_names(program_name)
            if func_names:
                names_list = list(func_names)
            else:
                print('No functions were found in program %s on server %s.' % (program_name, display_url))
        else:
            list_of_programs = db.get_program_names()
            if not list_of_programs:
                print('Server %s database empty.' % (display_url))
                return 0

            func_list = [db.get_function_names(prog_name)
                         for prog_name in list_of_programs]

            if not func_list:
                print('Server %s contains no functions.' % (display_url))
            else:
                names_list = func_list
    elif input_query == 'programs':
        list_found = list(db.get_program_names())
        if not list_found:
            print('No programs were found on server {}.'.format(display_url))
        else:
            names_list = list_found
    else:
        print('Query unrecognised.')
        return 2

    if prog:
        print(ProgramConverter.to_yed_graphml(prog, suppress_common))
    elif names_list is not None:
        print(names_list)
    else:
        print('Nothing was returned from the query.')


def audit(remote_neo4j):
    db = db_api.SextantConnection(remote_neo4j)

    return db.programs_with_metadata()
