# -----------------------------------------
# Sextant
# Copyright 2014, Ensoft Ltd.
# Author: Patrick Stevens, using work from Patrick Stevens and James Harkin
# -----------------------------------------
# Given a program file to upload, or a program name to delete from the server, does the right thing.

__all__ = ("upload_program", "delete_program")

from .db_api import SextantConnection, Validator
from .objdump_parser import get_parsed_objects

import logging


def upload_program(file_path, db_url, display_url='',
                   alternative_name=None,
                   not_object_file=False):
    """
    Uploads a program to the remote database.

    Raises requests.exceptions.ConnectionError if the server didn't exist.
    Raises IOError if file_path doesn't correspond to a file.
    Raises ValueError if the desired alternative_name (or the default, if no
    alternative_name was specified) already exists in the database.
    :param file_path: the path to the local file we wish to upload
    :param db_url: the URL of the database (eg. http://localhost:7474)
    :param display_url: alternative URL to display instead of db_url
    :param alternative_name: a name to give the program to override the default
    :param object_file: bool(the file is an objdump text output file, rather than a compiled binary)

    """

    if not display_url:
        display_url = db_url

    connection = SextantConnection(db_url)

    program_names = connection.get_program_names()
    if alternative_name is None:
        if Validator.sanitise(file_path) in program_names:
            raise ValueError("There is already a program under this name; "
                             "please delete the previous one with the same name"
                             " and retry, or rename the input file.")
    else:
        if Validator.sanitise(alternative_name) in program_names:
            raise ValueError("There is already a program under this name; "
                             "please delete the previous one with the same name"
                             " and retry, or rename the input file.")

    parsed_objects = get_parsed_objects(filepath=file_path,
                                        sections_to_view=['.text'],
                                        not_object_file=not_object_file,
                                        ignore_function_pointers=False)

    logging.info('Objdump has parsed!')

    if alternative_name is None:
        program_representation = connection.new_program(
                                     Validator.sanitise(file_path))
    else:
        program_representation = connection.new_program(
                                     Validator.sanitise(alternative_name))

    for obj in parsed_objects:
        for called in obj.what_do_i_call:
            if not program_representation.add_function_call(obj.name, called[-1]): # called is a tuple (address, name)
                logging.error('Validation error: {} calling {}'.format(obj.name, called[-1]))

    logging.info('Sending {} named objects to server {}...'.format(len(parsed_objects), display_url))
    program_representation.commit()
    logging.info('Sending complete! Exiting.')


def delete_program(program_name, db_url):
    """
    Deletes a program with the specified name from the database.
    :param program_name: the name of the program to delete
    :param db_url: the URL of the database (eg. http://localhost:7474)
    :return: bool(success)
    """
    connection = SextantConnection(db_url)
    connection.delete_program(program_name)
    print('Deleted {} successfully.'.format(program_name))
