# -----------------------------------------
# Sextant
# Copyright 2014, Ensoft Ltd.
# Author: Patrick Stevens, James Harkin
# -----------------------------------------
#Testing module

import unittest

from db_api import Function
from db_api import FunctionQueryResult
from db_api import SextantConnection
from db_api import Validator


class TestFunctionQueryResults(unittest.TestCase):
    def setUp(self):
        # we need to set up the remote database by using the neo4j_input_api
        self.remote_url = 'http://ensoft-sandbox:7474'

        self.setter_connection = SextantConnection(self.remote_url)
        self.program_1_name = 'testprogram'
        self.upload_program = self.setter_connection.new_program(self.program_1_name)
        self.upload_program.add_function('func1')
        self.upload_program.add_function('func2')
        self.upload_program.add_function('func3')
        self.upload_program.add_function('func4')
        self.upload_program.add_function('func5')
        self.upload_program.add_function('func6')
        self.upload_program.add_function('func7')
        self.upload_program.add_function_call('func1', 'func2')
        self.upload_program.add_function_call('func1', 'func4')
        self.upload_program.add_function_call('func2', 'func1')
        self.upload_program.add_function_call('func2', 'func4')
        self.upload_program.add_function_call('func3', 'func5')
        self.upload_program.add_function_call('func4', 'func4')
        self.upload_program.add_function_call('func4', 'func5')
        self.upload_program.add_function_call('func5', 'func1')
        self.upload_program.add_function_call('func5', 'func2')
        self.upload_program.add_function_call('func5', 'func3')
        self.upload_program.add_function_call('func6', 'func7')

        self.upload_program.commit()

        self.one_node_program_name = 'testprogram1'
        self.upload_one_node_program = self.setter_connection.new_program(self.one_node_program_name)
        self.upload_one_node_program.add_function('lonefunc')

        self.upload_one_node_program.commit()
        
        self.empty_program_name = 'testprogramblank'
        self.upload_empty_program = self.setter_connection.new_program(self.empty_program_name)

        self.upload_empty_program.commit()

        self.getter_connection = SextantConnection(self.remote_url)

    def tearDown(self):
        self.setter_connection.delete_program(self.upload_program.program_name)
        self.setter_connection.delete_program(self.upload_one_node_program.program_name)
        self.setter_connection.delete_program(self.upload_empty_program.program_name)
        del(self.setter_connection)

    def test_17_get_call_paths(self):
        reference1 = FunctionQueryResult(parent_db=None, program_name=self.program_1_name)
        reference1.functions = [Function(self.program_1_name, 'func1'), Function(self.program_1_name, 'func2'),
                                Function(self.program_1_name, 'func3'),
                                Function(self.program_1_name, 'func4'), Function(self.program_1_name, 'func5')]
        reference1.functions[0].functions_i_call = reference1.functions[1:4:2]  # func1 calls func2, func4
        reference1.functions[1].functions_i_call = reference1.functions[0:4:3]  # func2 calls func1, func4
        reference1.functions[2].functions_i_call = [reference1.functions[4]]  # func3 calls func5
        reference1.functions[3].functions_i_call = reference1.functions[3:5]  # func4 calls func4, func5
        reference1.functions[4].functions_i_call = reference1.functions[0:3]  # func5 calls func1, func2, func3
        self.assertEquals(reference1, self.getter_connection.get_call_paths(self.program_1_name, 'func1', 'func2'))
        self.assertIsNone(self.getter_connection.get_call_paths('not a prog', 'func1', 'func2')) # shouldn't validation
        self.assertIsNone(self.getter_connection.get_call_paths('notaprogram', 'func1', 'func2'))
        self.assertIsNone(self.getter_connection.get_call_paths(self.program_1_name, 'notafunc', 'func2'))
        self.assertIsNone(self.getter_connection.get_call_paths(self.program_1_name, 'func1', 'notafunc'))

    def test_02_get_whole_program(self):
        reference = FunctionQueryResult(parent_db=None, program_name=self.program_1_name)
        reference.functions = [Function(self.program_1_name, 'func1'), Function(self.program_1_name, 'func2'),
                               Function(self.program_1_name, 'func3'),
                               Function(self.program_1_name, 'func4'), Function(self.program_1_name, 'func5'),
                               Function(self.program_1_name, 'func6'), Function(self.program_1_name, 'func7')]
        reference.functions[0].functions_i_call = reference.functions[1:4:2]  # func1 calls func2, func4
        reference.functions[1].functions_i_call = reference.functions[0:4:3]  # func2 calls func1, func4
        reference.functions[2].functions_i_call = [reference.functions[4]]  # func3 calls func5
        reference.functions[3].functions_i_call = reference.functions[3:5]  # func4 calls func4, func5
        reference.functions[4].functions_i_call = reference.functions[0:3]  # func5 calls func1, func2, func3
        reference.functions[5].functions_i_call = [reference.functions[6]]  # func6 calls func7


        self.assertEqual(reference, self.getter_connection.get_whole_program(self.program_1_name))
        self.assertIsNone(self.getter_connection.get_whole_program('nottherightprogramname'))
        
    def test_03_get_whole_one_node_program(self):
        reference = FunctionQueryResult(parent_db=None, program_name=self.one_node_program_name)
        reference.functions = [Function(self.one_node_program_name, 'lonefunc')]
        
        self.assertEqual(reference, self.getter_connection.get_whole_program(self.one_node_program_name))

    def test_04_get_whole_empty_program(self):
        reference = FunctionQueryResult(parent_db=None, program_name=self.empty_program_name)
        reference.functions = []
        
        self.assertEqual(reference, self.getter_connection.get_whole_program(self.empty_program_name))

    def test_05_get_function_names(self):
        reference = {'func1', 'func2', 'func3', 'func4', 'func5', 'func6', 'func7'}
        self.assertEqual(reference, self.getter_connection.get_function_names(self.program_1_name))

    def test_06_get_function_names_one_node_program(self):
        reference = {'lonefunc'}
        self.assertEqual(reference, self.getter_connection.get_function_names(self.one_node_program_name))

    def test_07_get_function_names_empty_program(self):
        reference = set()
        self.assertEqual(reference, self.getter_connection.get_function_names(self.empty_program_name))

    def test_09_validation_is_used(self):
        self.assertFalse(self.getter_connection.get_function_names('not alphanumeric'))
        self.assertFalse(self.getter_connection.get_whole_program('not alphanumeric'))
        self.assertFalse(self.getter_connection.check_program_exists('not alphanumeric'))
        self.assertFalse(self.getter_connection.check_function_exists('not alphanumeric', 'alpha'))
        self.assertFalse(self.getter_connection.check_function_exists('alpha', 'not alpha'))
        self.assertFalse(self.getter_connection.get_all_functions_called('alphaprogram', 'not alpha function'))
        self.assertFalse(self.getter_connection.get_all_functions_called('not alpha program', 'alphafunction'))
        self.assertFalse(self.getter_connection.get_all_functions_calling('not alpha program', 'alphafunction'))
        self.assertFalse(self.getter_connection.get_all_functions_calling('alphaprogram', 'not alpha function'))
        self.assertFalse(self.getter_connection.get_call_paths('not alpha program','alphafunc1', 'alphafunc2'))
        self.assertFalse(self.getter_connection.get_call_paths('alphaprogram','not alpha func 1', 'alphafunc2'))
        self.assertFalse(self.getter_connection.get_call_paths('alphaprogram','alphafunc1', 'not alpha func 2'))

    def test_08_get_program_names(self):
        reference = {self.program_1_name, self.one_node_program_name, self.empty_program_name}
        self.assertEqual(reference, self.getter_connection.get_program_names())


    def test_11_get_all_functions_called(self):
        reference1 = FunctionQueryResult(parent_db=None, program_name=self.program_1_name)  # this will be the 1,2,3,4,5 component
        reference2 = FunctionQueryResult(parent_db=None, program_name=self.program_1_name)  # this will be the 6,7 component
        reference3 = FunctionQueryResult(parent_db=None, program_name=self.program_1_name)  # this will be the 7 component
        reference1.functions = [Function(self.program_1_name, 'func1'), Function(self.program_1_name, 'func2'),
                                Function(self.program_1_name, 'func3'),
                                Function(self.program_1_name, 'func4'), Function(self.program_1_name, 'func5')]
        reference2.functions = [Function(self.program_1_name, 'func6'), Function(self.program_1_name, 'func7')]
        reference3.functions = []

        reference1.functions[0].functions_i_call = reference1.functions[1:4:2]  # func1 calls func2, func4
        reference1.functions[1].functions_i_call = reference1.functions[0:4:3]  # func2 calls func1, func4
        reference1.functions[2].functions_i_call = [reference1.functions[4]]  # func3 calls func5
        reference1.functions[3].functions_i_call = reference1.functions[3:5]  # func4 calls func4, func5
        reference1.functions[4].functions_i_call = reference1.functions[0:3]  # func5 calls func1, func2, func3

        reference2.functions[0].functions_i_call = [reference2.functions[1]]

        self.assertEquals(reference1, self.getter_connection.get_all_functions_called(self.program_1_name, 'func1'))
        self.assertEquals(reference1, self.getter_connection.get_all_functions_called(self.program_1_name, 'func2'))
        self.assertEquals(reference1, self.getter_connection.get_all_functions_called(self.program_1_name, 'func3'))
        self.assertEquals(reference1, self.getter_connection.get_all_functions_called(self.program_1_name, 'func4'))
        self.assertEquals(reference1, self.getter_connection.get_all_functions_called(self.program_1_name, 'func5'))

        self.assertEquals(reference2, self.getter_connection.get_all_functions_called(self.program_1_name, 'func6'))

        self.assertEquals(reference3, self.getter_connection.get_all_functions_called(self.program_1_name, 'func7'))

        self.assertIsNone(self.getter_connection.get_all_functions_called(self.program_1_name, 'nottherightfunction'))
        self.assertIsNone(self.getter_connection.get_all_functions_called('nottherightprogram', 'func2'))

    def test_12_get_all_functions_called_1(self):
        reference1 = FunctionQueryResult(parent_db=None, program_name=self.one_node_program_name)
        reference1.functions = []

        d=self.getter_connection.get_all_functions_called(self.one_node_program_name, 'lonefunc')
        self.assertEquals(reference1, self.getter_connection.get_all_functions_called(self.one_node_program_name,
                                                                                      'lonefunc'))
        self.assertIsNone(self.getter_connection.get_all_functions_called(self.one_node_program_name,
                                                                          'not the right function'))
        self.assertIsNone(self.getter_connection.get_all_functions_called('not the right program', 'lonefunc'))

    def test_13_get_all_functions_called_blank(self):
        reference1 = FunctionQueryResult(parent_db=None, program_name=self.empty_program_name)
        reference1.functions = []

        self.assertIsNone(self.getter_connection.get_all_functions_called(self.empty_program_name,
                                                                          'not the right function'))

    def test_14_get_all_functions_calling(self):
        reference1 = FunctionQueryResult(parent_db=None, program_name=self.program_1_name)  # this will be the 1,2,3,4,5 component
        reference2 = FunctionQueryResult(parent_db=None, program_name=self.program_1_name)  # this will be the 6,7 component
        reference3 = FunctionQueryResult(parent_db=None, program_name=self.program_1_name)  # this will be the 7 component
        reference1.functions = [Function(self.program_1_name, 'func1'), Function(self.program_1_name, 'func2'),
                                Function(self.program_1_name, 'func3'),
                                Function(self.program_1_name, 'func4'), Function(self.program_1_name, 'func5')]

        reference1.functions[0].functions_i_call = reference1.functions[1:4:2]  # func1 calls func2, func4
        reference1.functions[1].functions_i_call = reference1.functions[0:4:3]  # func2 calls func1, func4
        reference1.functions[2].functions_i_call = [reference1.functions[4]]  # func3 calls func5
        reference1.functions[3].functions_i_call = reference1.functions[3:5]  # func4 calls func4, func5
        reference1.functions[4].functions_i_call = reference1.functions[0:3]  # func5 calls func1, func2, func3

        reference2.functions = [Function(self.program_1_name, 'func6'), Function(self.program_1_name, 'func7')]

        reference2.functions[0].functions_i_call = [reference2.functions[1]]

        reference3.functions = [Function(self.program_1_name, 'func6')]

        reference3.functions = []

        self.assertEquals(reference1, self.getter_connection.get_all_functions_calling(self.program_1_name, 'func1'))
        self.assertEquals(reference1, self.getter_connection.get_all_functions_calling(self.program_1_name, 'func2'))
        self.assertEquals(reference1, self.getter_connection.get_all_functions_calling(self.program_1_name, 'func3'))
        self.assertEquals(reference1, self.getter_connection.get_all_functions_calling(self.program_1_name, 'func4'))
        self.assertEquals(reference1, self.getter_connection.get_all_functions_calling(self.program_1_name, 'func5'))

        self.assertEquals(reference2, self.getter_connection.get_all_functions_calling(self.program_1_name,'func7'))

        self.assertEquals(reference3, self.getter_connection.get_all_functions_calling(self.program_1_name, 'func6'))

        self.assertIsNone(self.getter_connection.get_all_functions_calling(self.program_1_name, 'nottherightfunction'))
        self.assertIsNone(self.getter_connection.get_all_functions_calling('nottherightprogram', 'func2'))

    def test_15_get_all_functions_calling_one_node_prog(self):
        reference1 = FunctionQueryResult(parent_db=None, program_name=self.one_node_program_name)
        reference1.functions = []
        self.assertEquals(reference1, self.getter_connection.get_all_functions_calling(self.one_node_program_name,
                                                                                       'lonefunc'))
        self.assertIsNone(self.getter_connection.get_all_functions_calling(self.one_node_program_name,
                                                                           'not the right function'))
        self.assertIsNone(self.getter_connection.get_all_functions_calling('not the right program', 'lonefunc'))

    def test_16_get_all_functions_calling_blank_prog(self):
        reference1 = FunctionQueryResult(parent_db=None, program_name=self.empty_program_name)
        reference1.functions=[]

        self.assertIsNone(self.getter_connection.get_all_functions_called(self.empty_program_name,
                                                                          'not the right function'))



    def test_18_get_call_paths_between_two_functions_one_node_prog(self):
        reference = FunctionQueryResult(parent_db=None, program_name=self.one_node_program_name)
        reference.functions = [] # that is, reference is the empty program with name self.one_node_program_name

        self.assertEquals(self.getter_connection.get_call_paths(self.one_node_program_name, 'lonefunc', 'lonefunc'),
                          reference)
        self.assertIsNone(self.getter_connection.get_call_paths(self.one_node_program_name, 'lonefunc', 'notafunc'))
        self.assertIsNone(self.getter_connection.get_call_paths(self.one_node_program_name, 'notafunc', 'notafunc'))

    def test_10_validator(self):
        self.assertFalse(Validator.validate(''))
        self.assertTrue(Validator.validate('thisworks'))
        self.assertTrue(Validator.validate('th1sw0rks'))
        self.assertTrue(Validator.validate('12345'))
        self.assertFalse(Validator.validate('this does not work'))
        self.assertTrue(Validator.validate('this_does_work'))
        self.assertFalse(Validator.validate("'")) # string consisting of a single quote mark

if __name__ == '__main__':
    unittest.main()