import datetime
import postgresql
import re

from events import add_filter, add_action

split_re = re.compile(r'\s+|,|!|;|-|:')
word_valid_char = re.compile(r'^[A-Za-z]+$')

ignored_words = set([
    "a", "able", "about", "above", "according", "across", "actual", "actually", "adj",
"afterwards", "against", "almost", "alone", "along", "already", "also", "although", "always",
"am", "among", "amongst", "an", "and", "anyhow", "anyone", "anything", "appear",
"are", "aren't", "around", "as", "at", "b", "be", "became", "because",
"become", "becomes", "becoming", "been", "beforehand", "began", "behind", "being", "below",
"beside", "besides", "better", "between", "beyond", "billion", "both", "but", "by",
"called", "can", "can't", "cannot", "cant", "co", "co.", "could", "couldn't",
"crap", "currently", "d", "did", "didn't", "difference", "directly", "do", "does",
"doesn't", "don't", "done", "dont", "down", "during", "e", "e.g.", "each",
"eg", "eight", "eighty", "either", "else", "elsewhere", "ended", "ending", "enough",
"even", "ever", "every", "everyone", "everything", "everywhere", "except", "f", "few",
"fifty", "finding", "five", "following", "for", "former", "formerly", "forty", "found",
"four", "from", "further", "g", "getting", "go", "going", "gone", "h",
"had", "has", "hasn't", "have", "haven't", "he", "he'd", "he'll", "he's",
"hence", "her", "here", "here's", "hereafter", "hereby", "herein", "hereupon", "hers",
"herself", "him", "himself", "his", "how", "however", "hundred", "i", "i'd",
"i'll", "i'm", "i've", "i.e.", "if", "im", "in", "inc", "inc.",
"including", "indeed", "instead", "into", "is", "isn't", "it", "it's", "its",
"itself", "j", "just", "know", "knowing", "known", "kruft", "kudos", "l",
"later", "latter", "latterly", "let", "let's", "like", "likely", "look", "ltd",
"m", "made", "makes", "making", "many", "may", "maybe", "meantime", "meanwhile",
"meeting", "might", "million", "miss", "missed", "moreover", "mostly", "mr", "mrs",
"much", "must", "my", "myself", "n", "namely", "need", "needs", "neither",
"nevertheless", "nine", "ninety", "no", "nobody", "none", "nonetheless", "noone", "nor",
"not", "note", "nothing", "now", "nowhere", "o", "of", "often", "once",
"one", "one's", "only", "onto", "or", "other", "others", "otherwise", "our",
"ours", "ourselves", "overall", "p", "part", "per", "perhaps", "picked", "picking",
"place", "please", "pm", "possible", "provide", "q", "questions", "rather", "really",
"reason", "recently", "rid", "rotten", "s", "saw", "see", "seeing", "seem",
"seemed", "seeming", "seems", "seen", "seven", "seventy", "several", "she", "she'd",
"she'll", "she's", "should", "shouldn't", "showed", "showing", "shown", "since", "six",
"sixty", "so", "some", "somehow", "someone", "something", "sometime", "sometimes", "somewhere",
"still", "stopping", "such", "sure", "t", "take", "taken", "taking", "ten",
"than", "thanks", "that", "that'll", "that's", "that've", "the", "their", "them",
"themselves", "then", "thence", "there", "there'd", "there'll", "there're", "there's", "there've",
"thereafter", "thereby", "therefore", "therein", "thereupon", "these", "they", "they'd", "they'll",
"they're", "they've", "things", "think", "thinking", "thirty", "this", "those", "though",
"thought", "thousand", "three", "through", "throughout", "thru", "thus", "to", "together",
"too", "took", "toward", "towards", "trillion", "try", "twenty", "two", "u",
"under", "unless", "unlike", "unlikely", "until", "up", "upon", "us", "v",
"v.", "vast", "versus", "very", "via", "vs", "vs.", "w", "want",
"was", "wasn't", "way", "we", "we'd", "we'll", "we're", "we've", "were",
"weren't", "what", "what'll", "what's", "what've", "whatever", "whence", "whenever", "where's",
"whereafter", "whereas", "whereby", "wherein", "whereupon", "wherever", "whether", "while", "whither",
"who'd", "who'll", "who's", "whoever", "whole", "whom", "whomever", "whose", "will",
"wish", "with", "within", "without", "won't", "would", "wouldn't", "y", "yes",
"yet", "you", "you'd", "you'll", "you're", "you've", "youll", "your", "youre",
"yours", "yourself", "yourselves"
])

def get_connection(kernel):
    conf = kernel.configs['/']

    username = conf.get('database', 'username')
    password = conf.get('database', 'password')
    database = conf.get('database', 'database')

    db = postgresql.open('pq://%s:%s@localhost/%s' % (username, password, database))
    return db


def index_command(kernel, *args):
    """
    Create a search index for pages in the site.
    """
    print('Indexing site...')

    db = get_connection(kernel)

    select_search_page = db.prepare('select page_id, last_modified from search_pages where url = $1')
    insert_search_page = db.prepare('insert into search_pages (url, last_modified) values ($1, $2) RETURNING page_id')
    update_search_page = db.prepare('update search_pages set last_modified = $1 where page_id = $2')

    select_term_id = db.prepare('select term_id from search_terms where term = $1')
    add_term = db.prepare('insert into search_terms (term) values ($1) returning term_id')

    add_search = db.prepare('insert into search_term_pages (term_id, page_id) values ($1, $2)')

    for page in kernel.pages.values():
        rows = select_search_page(page.full_output_path)

        with db.xact():
            if len(rows) == 0:
                print('  - added new resource %s' % page.url)
                rows = insert_search_page(page.full_output_path, page.last_modified)
                page_id = rows[0][0]
            elif page.last_modified > rows[0][1]:
                print('  - updating changed %s' % page.url)
                page_id = rows[0][0]
                update_search_page(page.last_modified, page_id)

            for word in get_words(page):
                rows = select_term_id(word)
                if len(rows) == 0:
                    rows = add_term(word)
                term_id = rows[0][0]

                add_search(term_id, page_id)

    print('Complete')


def search_command(kernel, *args):
    """
    Test search for a specific keyword or set of keywords
    """
    db = get_connection(kernel)


    '''select * from search_pages p where exists (select 1 from search_terms t, search_term_pages tp where t.term = 'decrypt' and tp.term_id = t.term_id and tp.page_id = p.page_id) and exists (select 1 from search_terms t, search_term_pages tp where t.term = 'started' and tp.term_id = t.term_id and tp.page_id = p.page_id)'''

    exists_sql = []
    for x in range(1, len(args[0]) + 1):
        exists_sql.append('exists (select 1 from search_terms t, search_term_pages tp where t.term = $%s and tp.term_id = t.term_id and tp.page_id = p.page_id)' % x)

    sql = '''select p.url
                 from search_pages p
                 where ''' + ' and '.join(exists_sql)

    #print(sql)
    search = db.prepare(sql)
    rows = search(*args[0])
    for row in rows:
        print(row[0])


def get_words(page):
    words = set()

    for word in split_re.split(page.content):
        word = word.lower()
        if len(word) > 2 and word not in ignored_words and word_valid_char.match(word):
            words.add(word)

    return words

def process_commands(commands):
    commands['index'] = index_command
    commands['search'] = search_command
    return commands


add_filter('commands', process_commands)