
import storytext.guishared, util, logging, os, time, sys
from storytext.definitions import UseCaseScriptError
from storytext import applicationEvent, applicationEventDelay, applicationEventRemove
from difflib import SequenceMatcher

from java.lang import Boolean, IllegalStateException, IndexOutOfBoundsException, RuntimeException, NullPointerException, Exception
from java.text import ParseException
from java.util import ArrayList
from threading import Lock

from org.eclipse.jface.bindings.keys import KeyStroke
from org.eclipse.swt import SWT
from org.eclipse.swt.browser import Browser, ProgressListener
from org.eclipse.swt.custom import  CCombo, CTabFolder, CTabFolder2Adapter
from org.eclipse.swt.graphics import Point
from org.eclipse.swt.widgets import Button, Combo, Control, DateTime, Event, ExpandBar, Label, Link, List, Listener, Menu, MenuItem, \
    Shell, Spinner, Table, TableColumn, TabFolder, Text, ToolItem, Tree

from org.hamcrest.core import IsAnything

from org.eclipse.swtbot.swt.finder import SWTBot
from org.eclipse.swtbot.swt.finder.exceptions import WidgetNotFoundException
from org.eclipse.swtbot.swt.finder.finders import UIThreadRunnable, ContextMenuFinder
from org.eclipse.swtbot.swt.finder.keyboard import Keystrokes, KeyboardFactory
from org.eclipse.swtbot.swt.finder.results import Result, VoidResult
from org.eclipse.swtbot.swt.finder.utils import SWTBotPreferences
from org.eclipse.swtbot.swt.finder.widgets import AbstractSWTBot, AbstractSWTBotControl, SWTBotButton, SWTBotBrowser, SWTBotCCombo, SWTBotCheckBox, \
    SWTBotCombo, SWTBotCTabItem, SWTBotDateTime, SWTBotExpandBar, SWTBotExpandItem, SWTBotLabel, SWTBotLink, SWTBotList, SWTBotMenu, \
    SWTBotRadio, SWTBotShell, SWTBotSpinner, SWTBotTabItem, SWTBotTable, SWTBotTableColumn, SWTBotText, SWTBotToolbarDropDownButton, \
    SWTBotToolbarPushButton,  SWTBotToolbarRadioButton,SWTBotToolbarSeparatorButton, SWTBotToolbarToggleButton, SWTBotTree, \
    SWTBotTreeItem

applicationEventType = 1234 # anything really, just don't conflict with the real SWT events

class PythonResult(Result):
    def __init__(self, method, args):
        self.method = method
        self.args = args
    
    def run(self):
        return self.method(*self.args)

def runOnUIThread(method, *args):
    try:
        return UIThreadRunnable.syncExec(PythonResult(method, args))
    except IllegalStateException:
        raise UseCaseScriptError, "The GUI has already exited"
    except NullPointerException, e:
        # Temporary code to try to find intermittent Windows error
        print "Caught intermittent Windows NullPointerException!"
        e.printStackTrace()
        raise

class WidgetAdapter(storytext.guishared.WidgetAdapter):
    popupMenuContexts = {}
    contextFinders = []
    def getChildWidgets(self):
        return [] # don't use this...
        
    def getWidgetTitle(self):
        return ""
        
    def getLabel(self):
        if isinstance(self.widget, (SWTBotText, SWTBotCombo, SWTBotSpinner, FakeSWTBotCCombo)) or \
               not hasattr(self.widget.widget, "getText"):
            return self.getFromUIThread(util.getTextLabel, self.widget.widget)
        try:
            return util.getItemText(self.widget.getText())
        except:
            return ""

    def getDialogTitle(self):
        def _getDialogTitle():
            if isinstance(self.widget.widget, Control):
                shell = self.widget.widget.getShell()
                if shell.getParent():
                    id = shell.getData("org.eclipse.swtbot.widget.key")
                    return id or shell.getText()
            return ""
        return self.getFromUIThread(_getDialogTitle)

    def getType(self):
        # SWT name, not the SWTBot name
        return self.widget.widget.__class__.__name__
        
    def isAutoGenerated(self, name):
        return len(name) == 0

    def getTooltip(self):
        try:
            return self.widget.getToolTipText()
        except:
            return ""

    def getName(self):
        return self.widget.getId() or ""
    
    def getActionId(self):
        data = runOnUIThread(self.widget.widget.getData)
        return data.getAction().getId() if hasattr(data, "getAction") else ""
    
    def findPossibleUIMapIdentifiers(self):
        ids = storytext.guishared.WidgetAdapter.findPossibleUIMapIdentifiers(self)
        actionId = self.getActionId()
        if actionId:
            pos = ids.index("Type=" + self.getType())
            ids.insert(pos, "Action=" + actionId)
        return ids
    
    def getNameForAppEvent(self):
        return self.getName() or self.getType().lower()

    def getFromUIThread(self, method, *args):
        try:
            return runOnUIThread(method, *args)
        except:
            return ""
    
    def getContextName(self):
        parent = runOnUIThread(self.widget.widget.getParent)
        for contextFinder in self.contextFinders:
            name = contextFinder(self.widget.widget, parent)
            if name is not None:
                return name
        return self.getContextNameFromAncestor(parent)
    
    def getMenuContextNameFromAncestor(self, parent):
        def getParentText():
            item = parent.getParentItem()
            return util.getItemText(item.getText()) if item else self.getPopupMenuContext(parent)
        return runOnUIThread(getParentText)
    
    def getMenuContextFromWidget(self, widget):
        if widget:
            adapter = WidgetMonitor.makeAdapter(widget)
            if adapter:
                return adapter.findPossibleUIMapIdentifiers()[0].replace("Label=", "").replace("=", ":")
            
        return "Popup Menu"
    
    def getPopupMenuContext(self, menu):
        if menu in self.popupMenuContexts:
            return self.popupMenuContexts.get(menu)
        parentShell = menu.getParent()
        widget = self.findWidgetWithMenu(parentShell, menu)
        context = self.getMenuContextFromWidget(widget)
        self.popupMenuContexts[menu] = context
        return context
    
    def findWidgetWithMenu(self, widget, menu):
        if widget.getMenu() == menu:
            return widget
        
        if hasattr(widget, "getChildren"):
            for child in widget.getChildren():
                fromChild = self.findWidgetWithMenu(child, menu)
                if fromChild:
                    return fromChild
    
    def getContextNameFromAncestor(self, parent):
        if isinstance(parent, Menu):
            return self.getMenuContextNameFromAncestor(parent)
        else:
            return util.getContextNameForWidget(parent)
        
    def checkMenuVisible(self):
        menu = self.widget.widget.getParent()
        return menu.getVisible()
        
    def isPreferred(self):
        return self.isInstanceOf(SWTBotMenu) and runOnUIThread(self.checkMenuVisible)
        

storytext.guishared.WidgetAdapter.adapterClass = WidgetAdapter    
        
class BasicRecordListener(Listener):
    def __init__(self, method, event):
        self.method = method
        self.event = event
        
    def handleEvent(self, e): 
        storytext.guishared.catchAll(self.method, e, self.event)

        
class SignalEvent(storytext.guishared.GuiEvent):
    def __init__(self, name, widget, argumentParseData, *args):
        self.generationModifiers = argumentParseData.split(",") if argumentParseData else []
        storytext.guishared.GuiEvent.__init__(self, name, widget, *args)
        
    def connectRecord(self, method):
        eventType = self.getRecordEventType()
        try:
            runOnUIThread(self.addListeners, eventType, BasicRecordListener(method, self))
        except: # Get 'widget is disposed' sometimes, don't know why...
            pass
        
    @classmethod
    def getRecordEventType(cls):
        return getattr(SWT, cls.getAssociatedSignal(None))
        
    def addListeners(self, *args):
        # Three indirections: WidgetAdapter -> SWTBotMenu -> MenuItem
        return self.widget.widget.widget.addListener(*args)
    
    def addFilter(self, *args):
        self.widget.widget.widget.getDisplay().addFilter(*args)
    
    def removeFilter(self, *args):
        self.widget.widget.widget.getDisplay().removeFilter(*args)

    def generate(self, *args):
        try:
            self._generate(*args)
        except (IllegalStateException, IndexOutOfBoundsException), _:
            pass # get these for actions that close the UI. But only after the action is done :)

    def shouldRecord(self, event, *args):
        return DisplayFilter.instance.getEventFromUser(event, self.isTriggeringEvent)
    
    def isTriggeringEvent(self, *args):
        return False

    def delayLevel(self, event, *args):
        # If there are events for other shells, implies we should delay as we're in a dialog
        return DisplayFilter.instance.otherEventCount(event, self.isTriggeringEvent)

    def widgetDisposed(self):
        return runOnUIThread(self.widget.widget.widget.isDisposed)

    def widgetVisible(self):
        return self.widget.isVisible()
        
    def widgetSensitive(self):
        return self.widget.isEnabled()
        
    def describeWidget(self):
        return "of type " + self.widget.getType()
    
    @classmethod
    def getSignalsToFilter(cls):
        return [ cls.getRecordEventType() ]


class StateChangeEvent(SignalEvent):
    def outputForScript(self, *args):
        return ' '.join([self.name, self.getStateText(*args) ])

    def isStateChange(self, *args):
        return True


class SelectEvent(SignalEvent):    
    def _generate(self, *args):
        self.widget.click()
        if self.allowsIdenticalCopies(): # is a menu
            try:
                runOnUIThread(self.hideMenu)
            except UseCaseScriptError:
                pass # Means we're finishing, don't care if we can't hide the menu then...
                
    def hideMenu(self):
        menuItem = self.widget.widget.widget
        if not menuItem.isDisposed():
            menu = menuItem.getParent()
            menu.setVisible(False)

    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Selection"
    
    def allowsIdenticalCopies(self):
        return self.widget.isInstanceOf(SWTBotMenu)


class LinkSelectEvent(SelectEvent):
    def _generate(self, *args):
        # There is self.widget.click(), but it is very low level, seizes the mouse pointer,
        # and fails utterly under KDE. See https://bugs.eclipse.org/bugs/show_bug.cgi?id=337548
        text = self.widget.getText()
        startPos = text.find(">") + 1
        endPos = text.rfind("<")
        hyperlinkText = text[startPos:endPos]
        self.widget.click(hyperlinkText)
            
        
class RadioSelectEvent(SelectEvent):
    def shouldRecord(self, event, *args):
        return SignalEvent.shouldRecord(self, event, *args) and event.widget.getSelection()

    def getSelectedButton(self):
        method = SWTBotRadio.getDeclaredMethod("otherSelectedButton", [])
        method.setAccessible(True)
        return method.invoke(self.widget.widget, [])
        
    def _generate(self, *args):
        if self.widget.isInstanceOf(SWTBotRadio):
            # Workaround for bug in SWTBot which doesn't handle Eclipse radio buttons properly
            # See https://bugs.eclipse.org/bugs/show_bug.cgi?id=344484 for details
            selectedButton = self.getSelectedButton()
            runOnUIThread(selectedButton.widget.setSelection, False)
        SelectEvent._generate(self)
        
# just so we can distinguish later on...
class SWTBotRadioMenu(SWTBotMenu):
    def click(self):
        # This case also isn't handled correctly by SWTBot!
        # See https://bugs.eclipse.org/bugs/show_bug.cgi?id=397649    
        selectedMenuItem = runOnUIThread(self.getSelectedMenuItem)
        if selectedMenuItem:
            SWTBotMenu(selectedMenuItem).click() # Should have same effect, i.e. disable it
        SWTBotMenu.click(self)

    def getSelectedMenuItem(self):
        menu = self.widget.getParent()
        for item in menu.getItems():
            if item.getStyle() & SWT.RADIO and item.getSelection():
                return item
            
class DropDownButtonEvent(SelectEvent):
    def shouldRecord(self, event, *args):
        return SignalEvent.shouldRecord(self, event, *args) and event.detail & SWT.ARROW == 0
    
class DropDownGenerateFilter(Listener):
    def __init__(self, argumentString):
        self.done = False
        self.argumentString = argumentString
                
    def handleEvent(self, e):
        if isinstance(e.widget, Menu) and not self.done:
            if e.widget.getItemCount():
                self.describeMenu(e.widget)
            for item in e.widget.getItems():
                if util.getItemText(item.getText()) == self.argumentString:
                    SWTBotMenu(item).click()
                    self.done = True
                    
    def describeMenu(self, menu):
        from describer import Describer
        desc = Describer()
        desc.logger.info("\nChoosing from Drop Down Menu:")
        desc.logger.info(desc.getMenuDescription(menu))
        
    
class DropDownSelectionEvent(SelectEvent):
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "DropDownSelection"
    
    @classmethod
    def getRecordEventType(cls):
        return SWT.Selection
    
    def parseArguments(self, argumentString):
        return argumentString
    
    def _generate(self, argumentString):
        genFilter = DropDownGenerateFilter(argumentString)
        runOnUIThread(self.addFilter, SWT.Show, genFilter)
        try:
            item = self.widget.menuItem(argumentString)
            if not genFilter.done:
                item.click()
            # We caused the menu to be visible
            # Need to hide it again to avoid unwanted effects
            runOnUIThread(self.hideRootMenu, item)
        except WidgetNotFoundException:
            # thrown if there are no items
            if not genFilter.done:
                print "Got menu with no items"
        runOnUIThread(self.removeFilter, SWT.Show, genFilter)
            
    def connectRecord(self, method):
        class RecordFilter(Listener):
            def handleEvent(listenerSelf, e): #@NoSelf
                if isinstance(e.widget, MenuItem):
                    storytext.guishared.catchAll(method, e, self)
                    for listener in e.widget.getListeners(SWT.Selection):
                        if isinstance(listener, BasicRecordListener):
                            e.widget.removeListener(SWT.Selection, listener)
                            break
                    self.removeFilter(SWT.Selection, listenerSelf)
        def arrowClicked(event, *args):
            # The MenuItem object is very transient, must use Filter to find it
            if event.detail & SWT.ARROW != 0:
                self.addFilter(SWT.Selection, RecordFilter())
                
        SelectEvent.connectRecord(self, arrowClicked)

    def shouldRecord(self, *args):
        return True # Never going to get these programmatically, don't bother...
    
    def outputForScript(self, event, *args):
        return self.name + " " + util.getItemText(event.widget.getText())
                
    def hideRootMenu(self, item):
        menu = util.getRootMenu(item.widget)
        menu.setVisible(False)
    
class TabEvent(SelectEvent):
    def findTabWithText(self, text):
        for item in self.widget.widget.widget.getItems():
            if item.getText() == text:
                return item
        
    def parseArguments(self, text):
        # Seems we can only get tab item text in the UI thread (?)
        item = runOnUIThread(self.findTabWithText, text)
        if item:
            return item
        else:
            raise UseCaseScriptError, "Could not find tab labelled '" + text + "' in TabFolder."
    
    def getItemText(self, item):
        return item.getText()
    
    def outputForScript(self, event, *args):
        # Text may have changed since the application listeners have been applied
        return ' '.join([self.name, self.getItemText(event.item)])

    
class TabSelectEvent(TabEvent):
    swtbotItemClass = SWTBotTabItem
    def _generate(self, tab):
        self.swtbotItemClass(tab).activate()
            

class CTabSelectEvent(TabSelectEvent):
    swtbotItemClass = SWTBotCTabItem
    def isStateChange(self):
        return True

    def implies(self, *args):
        # State change because it can be implied by CTabCloseEvents
        # But don't amalgamate them together, allow several tabs to be selected in sequence
        return False


class CTabCloseEvent(TabEvent):
    def connectRecord(self, method):
        class RecordListener(CTabFolder2Adapter):
            def close(listenerSelf, e): #@NoSelf
                storytext.guishared.catchAll(method, e, self)

        runOnUIThread(self.widget.widget.widget.addCTabFolder2Listener, RecordListener())
        
    @classmethod
    def getRecordEventType(cls):
        return getattr(SWT, cls.getAssociatedSignal(None))

    def _generate(self, tab):
        # SWTBotCTabItem(tab).close() unfortunately seems to create additional activate and selection events
        self.simulate(tab)

    def shouldRecord(self, *args):
        return DisplayFilter.instance.hasEventOfType([ SWT.MouseUp ], self.widget.widget.widget)

    def isTriggeringEvent(self, e):
        return e.type == SWT.MouseUp

    @classmethod
    def getSignalsToFilter(cls):
        return [ SWT.MouseUp, SWT.Dispose ]

    @classmethod
    def getAssociatedSignal(cls, widget):
        return "CloseTab"
    
    def simulate(self,tab):
        X, Y = self.getCloseBoxXY(tab)
        displayLoc = runOnUIThread(self.widget.widget.widget.toDisplay, X, Y)
        display = runOnUIThread(tab.getDisplay)
        EventPoster(display).moveClickAndReturn(displayLoc.x, displayLoc.y)

    def getCloseBoxXY(self, tab):
        rect = util.getPrivateField(tab, "closeRect")
        X = rect.x + rect.width/2
        Y = rect.y + rect.height/2
        return X, Y
        
class ShellCloseEvent(SignalEvent):    
    def _generate(self, *args):
        # SWTBotShell.close appears to close things twice, just use the ordinary one for now...
        class CloseRunnable(VoidResult):
            def run(resultSelf): #@NoSelf
                self.widget.widget.widget.close()
                
        UIThreadRunnable.asyncExec(CloseRunnable())
        
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Close"
    

class ResizeEvent(StateChangeEvent):
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Resize"
    
    def parseArguments(self, argumentString):
        words = argumentString.split()
        width = int(words[1])
        height = int(words[-1])
        return width, height

    def _generate(self, arguments):
        width, height = arguments
        runOnUIThread(self.widget.widget.widget.setSize, width, height)

    def dimensionText(self, dimension):
        return str((dimension / 10) * 10)
    
    def getSize(self):
        size = self.widget.widget.widget.getSize()
        return size.x, size.y 
        
    def getStateText(self, *args):
        width, height = self.getSize()
        return "width " + self.dimensionText(width) + " and height " + self.dimensionText(height)

class FreeTextEvent(SignalEvent):
    def connectRecord(self, method):
        pass # Intended for events in filechoosers etc, which we cannot record anyway
    
    @classmethod
    def getSignalsToFilter(cls):
        return []
    
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "TypeText"
    
    def shouldRecord(self, *args):
        return False
    
    def parseArguments(self, text):
        return text
    
    def generate(self, argumentString):
        keyboard = util.callPrivateMethod(self.widget.widget, "keyboard")
        try:
            keyStroke = KeyStroke.getInstance(argumentString)
            key = keyStroke.getNaturalKey()
            keyboard.pressShortcut(keyStroke.getModifierKeys(), key, chr(key))
        except Exception:
            keyboard.typeText(argumentString + "\n", SWTBotPreferences.TYPE_INTERVAL)


class SpinnerSelectEvent(StateChangeEvent):
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Selection"
    
    def getStateText(self, *args):
        return self.widget.getText()
    
    def parseArguments(self, argumentString):
        if argumentString.isdigit():
            return int(argumentString)
        else:
            raise UseCaseScriptError, "Cannot set Spinners to non-numeric values!"
    
    def _generate(self, argument):
        self.widget.setSelection(argument)

class PhysicalEventListener(Listener):
    def handleEvent(listenerSelf, e): #@NoSelf
        TextEvent.physicalEventWidget = e.widget

    
class TextEvent(StateChangeEvent):
    physicalEventWidget = None
    physicalEventListener = None
    def __init__(self, *args):
        StateChangeEvent.__init__(self, *args)
        self.stateText = self.getStateText()
    
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Modify"
    
    def connectRecord(self, method):
        StateChangeEvent.connectRecord(self, method)
        if self.isTyped() and not TextEvent.physicalEventListener:
            TextEvent.physicalEventListener = PhysicalEventListener()
            runOnUIThread(self.addFilter, SWT.KeyDown, TextEvent.physicalEventListener)
            runOnUIThread(self.addFilter, SWT.MouseDown, TextEvent.physicalEventListener)
        
    def parseArguments(self, text):
        return text
        
    def selectAll(self):
        self.widget.selectAll()
        
    def isTriggeringEvent(self, e):
        # Editing sometimes generates two modify events, one "inside" the other
        # Without this we risk the inner one being rejected because the outer one hasn't run,
        # and the outer being rejected because the inner one has updated the text already
        # Don't include the Enter presses from TextActivateEvent below...
        return (e.type == SWT.Modify and e.widget is self.widget.widget.widget) or \
               (e.type == SWT.KeyDown and e.character != SWT.CR)

    def isTyped(self):
        return "typed" in self.generationModifiers

    def _generate(self, argumentString):
        self.widget.setFocus()
        if self.isTyped() and argumentString:
            self.selectAll()
            self.widget.typeText(argumentString)
        else:
            self.widget.setText(argumentString)

    def getStateText(self, *args):
        return self.widget.getText()
    
    def textChanged(self, newStateText):
        return self.stateText is None or self.stateText != newStateText
    
    def shouldRecord(self, event, *args):
        if not self.isEditable():
            return False

        newStateText = self.getStateText()
        if not self.textChanged(newStateText):
            return False
        
        self.stateText = newStateText
        if newStateText and self.isTyped() and self.physicalEventWidget is not self.widget.widget.widget:
            return False
        
        return not self.widget.widget.widget in CComboSelectEvent.internalWidgets and StateChangeEvent.shouldRecord(self, event, *args)
        
    def isEditable(self):
        return not (self.widget.widget.widget.getStyle() & SWT.READ_ONLY) 
    
    def implies(self, stateChangeOutput, stateChangeEvent, *args):
        if isinstance(stateChangeEvent, TextEvent) and \
            stateChangeEvent.widget.widget.widget == self.widget.widget.widget and self.isTyped() != stateChangeEvent.isTyped():
            return True
            
        if self.isTyped():
            currOutput = self.outputForScript(stateChangeEvent, *args)
            return StateChangeEvent.implies(self, stateChangeOutput, stateChangeEvent, *args) and \
               self.hasGainedOrLostCharacters(currOutput, stateChangeOutput)
        else:
            return StateChangeEvent.implies(self, stateChangeOutput, stateChangeEvent, *args)
            
    @classmethod
    def hasGainedOrLostCharacters(cls, text1, text2):
        matcher = SequenceMatcher(None, text1, text2)
        blocks = matcher.get_matching_blocks()
        totalMatchLength = sum((block[2] for block in blocks))
        return totalMatchLength == min(len(text1), len(text2)) and cls.mismatchingSectionCount(blocks) <= 1
    
    @staticmethod
    def mismatchingSectionCount(blocks):
        # We want to make sure text has only been inserted or deleted in a block, as a user would do
        startMismatch = blocks[0][0] > 0 or blocks[0][1] > 0
        middleMismatches = max(len(blocks) - 2, 0)
        endMismatch = blocks[-2][0] + blocks[-2][2] != blocks[-1][0] or \
                      blocks[-2][1] + blocks[-2][2] != blocks[-1][1]
        return int(startMismatch) + middleMismatches + int(endMismatch)
        
            
class TextActivateEvent(SignalEvent):
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Activate"
    
    @classmethod
    def getRecordEventType(cls):
        return SWT.Traverse
    
    @classmethod
    def getSignalsToFilter(cls):
        return [ cls.getRecordEventType(), SWT.DefaultSelection ]
    
    def isTraverseReturn(self, event):
        return event.type == SWT.Traverse and event.detail == SWT.TRAVERSE_RETURN

    def isTriggeringEvent(self, event):
        return event.widget in CComboChangeEvent.internalWidgets and self.isTraverseReturn(event)
    
    def shouldRecord(self, event, *args):
        return not self.widget.widget.widget in CComboChangeEvent.internalWidgets and self.isTraverseReturn(event) and \
            (not self.widget.isInstanceOf(FakeSWTBotCCombo) or \
             DisplayFilter.instance.hasEventOfType([ SWT.Traverse ], util.getPrivateField(self.widget.widget.widget, "text")))
    
    def _generate(self, argumentString):
        self.widget.setFocus()
        self.widget.typeText("\n")

class TextContentAssistEvent(SignalEvent):
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "ContentAssist"
    
    @classmethod
    def getRecordEventType(cls):
        return SWT.KeyDown
    
    def shouldRecord(self, event, *args):
        return event.type == SWT.KeyDown and event.character == " " and (event.stateMask & SWT.CTRL) != 0
    
    def _generate(self, argumentString):
        self.widget.pressShortcut([ Keystrokes.CTRL, Keystrokes.SPACE ])


class ComboSelectEvent(StateChangeEvent):
    def __init__(self, *args):
        StateChangeEvent.__init__(self, *args)
        self.stateText = None

    def getStateText(self, *args):
        widget = self.widget.widget.widget
        return DisplayFilter.instance.itemTextCache.get(widget, widget.getText())
    
    def parseArguments(self, argumentString):
        index = runOnUIThread(self.widget.widget.widget.indexOf, argumentString)
        if index == -1:
            raise UseCaseScriptError, "Could not find item labelled '" + argumentString + "' in Combo box."
        return index
    
    def _generate(self, index):
        self.widget.setSelection(index)
        
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Selection"

    def shouldRecord(self, event, *args):
        newStateText = self.getStateText()
        if self.stateText is not None and self.stateText == newStateText:
            return False
        self.stateText = newStateText
        return StateChangeEvent.shouldRecord(self, event, *args)
    
    def implies(self, stateChangeOutput, stateChangeEvent, *args):
        currOutput = self.outputForScript(*args)
        _, currSelection = currOutput.split(self.name, 1)
        _, oldSelection = stateChangeOutput.split(stateChangeEvent.name, 1)
        return (self.isImpliedEventType(stateChangeEvent) and currSelection == oldSelection) or StateChangeEvent.implies(self, stateChangeOutput, *args)

    def isImpliedEventType(self, event):
        return isinstance(event, ComboSelectEvent)
    
class CComboSelectEvent(ComboSelectEvent):
    internalWidgets = []
    def __init__(self, *args):
        ComboSelectEvent.__init__(self, *args)
        self.addWidgets()

    def addWidgets(self):
        list_ = util.getPrivateField(self.widget.widget.widget, "list")
        text_ = util.getPrivateField(self.widget.widget.widget, "text")
        self.internalWidgets.append(list_)
        self.internalWidgets.append(text_)

    def isTriggeringEvent(self, e):
        return e.widget in self.internalWidgets or StateChangeEvent.isTriggeringEvent(self, e)
    
    def isImpliedEventType(self, event):
        return isinstance(event, CComboChangeEvent)

class ComboTextEvent(TextEvent):
    def parseArguments(self, *args):
        if not runOnUIThread(self.isEditable):
            raise UseCaseScriptError, "Cannot edit text in this Combo Box as it is readonly"
        return TextEvent.parseArguments(self, *args)
    
    def selectAll(self):
        # Strangely, there is no selectAll method...
        selectionPoint = Point(0, len(self.widget.getText()))
        runOnUIThread(self.widget.widget.widget.setSelection, selectionPoint)
    

class CComboChangeEvent(CComboSelectEvent):
    def parseArguments(self, text):
        if runOnUIThread(self.widget.widget.widget.getStyle) & SWT.READ_ONLY:
            raise UseCaseScriptError, "Cannot edit text in this Combo Box as it is readonly"
        return text
    
    def _generate(self, argumentString):
        try:
            self.widget.setFocus()
            self.widget.setText(argumentString)
        except RuntimeException, e:
            raise UseCaseScriptError, e.getMessage()

    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Modify"
    
    def implies(self, stateChangeOutput, stateChangeEvent, *args):
        return StateChangeEvent.implies(self, stateChangeOutput, stateChangeEvent, *args)

    def getStateText(self, *args):
        return self.widget.getText()
    

class StoryTextSwtBotTable(SWTBotTable):    
    def select(self, indices):
        # When clicking in a cell, SWTBot likes to select the entire row first, generating mouse events and all
        # This can cause trouble, e.g. the cells can translate it into clicking the first column
        runOnUIThread(self.widget.deselectAll)
        for i in indices:
            runOnUIThread(self.widget.select, i)


class TableSelectEvent(StateChangeEvent):
    def __init__(self, *args):
        StateChangeEvent.__init__(self, *args)
        self.getIndexer()

    def getIndexer(self):
        return TableIndexer.getIndexer(self.widget.widget.widget)

    @classmethod
    def getAssociatedSignal(cls, widget):
        return "MouseDown"
    
    @classmethod
    def getAssociatedSignatures(cls, widget):
        return [ "CellSelection" ]
    
    def parseArguments(self, argumentString):
        return self.getIndexer().getViewCellIndices(argumentString)
            
    def _generate(self, cell):
        self.widget.click(*cell)
        
    def getStateText(self, event, *args):
        row, col = self.findCell(event)
        return self.getIndexer().getCellDescription(row, col)
    
    def clickCount(self):
        return 1
    
    def shouldRecord(self, event, *args):
        if event.count != self.clickCount():
            return False
        row, _ = self.findCell(event)
        if row is not None:
            return StateChangeEvent.shouldRecord(self, event, *args)
        else:
            DisplayFilter.instance.logger.debug("Rejected event, could not find matching row in table")
            return False
    
    def findCell(self, event):
        return self.getCell(event.x, event.y, event.widget)
    
    def getCell(self, xPos, yPos, table):
        pt = Point(xPos, yPos)
        firstRow = table.getTopIndex()
        columnCount = table.getColumnCount()
        for rowIndex in range(firstRow, firstRow + table.getItemCount()):
            item = table.getItem(rowIndex)
            if columnCount:
                for col in range(columnCount):
                    rect = item.getBounds(col)
                    if rect.contains(pt):
                        return rowIndex, col
            else:
                rect = item.getBounds()
                # No columns. For some reason, widths of cells aren't reliable in this case
                # Use the y information only and ignore event.x and the cell widths
                if yPos < rect.y + rect.height:
                    return rowIndex, None
        return None, None

    def implies(self, stateChangeOutput, stateChangeEvent, *args):
        currOutput = self.outputForScript(*args)
        return currOutput == stateChangeOutput
    
class TableDoubleClickEvent(TableSelectEvent):
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "MouseDoubleClick"
    
    @classmethod
    def getAssociatedSignatures(cls, widget):
        return [ "CellDoubleClick" ]
    
    def clickCount(self):
        return 2
    
    def _generate(self, cell):
        self.widget.doubleClick(*cell)
    
    def implies(self, stateChangeOutput, stateChangeEvent, *args):
        return isinstance(stateChangeEvent, TableSelectEvent) or SelectEvent.implies(self, stateChangeOutput, stateChangeEvent, *args)
    
    
class TableIndexer(storytext.guishared.TableIndexer):    
    def getRowCount(self):
        return runOnUIThread(self.widget.getItemCount)

    def getCellValue(self, row, col):
        return self.widget.getItem(row).getText(col)
    
    def getColumnText(self, col):
        return self.widget.getColumn(col).getText()
    
    def findColumnIndex(self, columnName):
        return runOnUIThread(storytext.guishared.TableIndexer.findColumnIndex, self, columnName)
    
    def updateTableInfo(self):
        runOnUIThread(storytext.guishared.TableIndexer.updateTableInfo, self)
        
    def findRowNames(self):
        return runOnUIThread(storytext.guishared.TableIndexer.findRowNames, self)
    
    def cacheCorrect(self):
        return self.getRowCount() == len(self.rowNames) and not all((r.startswith("<unnamed>") for r in self.rowNames))

    def checkNameCache(self):
        if not self.cacheCorrect():
            self.updateTableInfo()

    def rowNameCorrect(self, row):
        return self.primaryKeyColumn is None or self.rowNames[row] == runOnUIThread(self.getCellValueToUse, row, self.primaryKeyColumn)
        
    def getCellDescription(self, row, col, **kw):
        if not self.cacheCorrect() or not self.rowNameCorrect(row):
            self.updateTableInfo()
        return storytext.guishared.TableIndexer.getCellDescription(self, row, col, **kw)

    def getViewCellIndices(self, *args, **kw):
        self.checkNameCache()
        try:
            row, col = storytext.guishared.TableIndexer.getViewCellIndices(self, *args, **kw)
            if self.rowNameCorrect(row):
                return row, col
        except UseCaseScriptError: # If we failed to find it, update the info and try again
            pass
        
        self.updateTableInfo()
        return storytext.guishared.TableIndexer.getViewCellIndices(self, *args, **kw)
    
    
class TableColumnHeaderEvent(SignalEvent):
    def __init__(self, *args):
        SignalEvent.__init__(self, *args)
        self.columnsFound = set()
    
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Selection"
    
    @classmethod
    def getAssociatedSignatures(cls, widget):
        return [ "ColumnSelection" ]
    
    def addColumnListeners(self, *args):
        for column in self.widget.widget.widget.getColumns():
            if column not in self.columnsFound:
                self.columnsFound.add(column)
                column.addListener(*args)

    def addListeners(self, *args):
        self.addColumnListeners(*args)            
        class PaintListener(Listener):
            def handleEvent(lself, e): #@NoSelf
                storytext.guishared.catchAll(self.addColumnListeners, *args)
        self.widget.widget.widget.addListener(SWT.Paint, PaintListener())
        
    def outputForScript(self, event, *args):
        return " ".join([ self.name, event.widget.getText() ])
    
    def parseArguments(self, argumentString):
        try:
            return self.widget.header(argumentString)
        except WidgetNotFoundException:
            raise UseCaseScriptError, "Could not find column labelled '" + argumentString + "' in table."
        
    def _generate(self, column):
        column.click()

class TreeIndexer(storytext.guishared.TreeIndexer):
    def populate(self):
        self.allItems = {}
        self.allDescriptions = {}
        for item in runOnUIThread(self.widget.getItems):
            runOnUIThread(self.storeSubItems, item)

    def storeSubItems(self, item):
        self.storeItem(item)
        if hasattr(item, "getItems"):
            for subItem in item.getItems():
                self.storeSubItems(subItem)
    
    def getDescriptionToStore(self, item):
        return DisplayFilter.instance.itemTextCache.pop(item, item.getText())        

class TreeEvent(SignalEvent):
    indexerClass = TreeIndexer
    def parseArguments(self, argumentString):
        if argumentString:
            indexer = self.indexerClass.getIndexer(self.widget.widget.widget)
            item = indexer.getItem(argumentString)
            if not item or item.isDisposed():
                indexer.populate()
                item = indexer.getItem(argumentString)
            if item:
                return self.getItemClass()(item)
            else:
                raise UseCaseScriptError, "Could not find item labelled '" + argumentString + "' in " + self.getClassDesc() + "."
        else:
            return ""
    
    def getClassDesc(self):
        return self.widget.widget.widget.__class__.__name__.lower()

    def getTextToRecord(self, item):
        indexer = self.indexerClass.getIndexer(self.widget.widget.widget)
        desc = indexer.getItemDescription(item)
        if not desc:
            indexer.populate()
            desc = indexer.getItemDescription(item)
        return desc
        
    def outputForScript(self, event, *args):
        if event.item is not None:
            # Text may have changed since the application listeners have been applied
            text = self.getTextToRecord(event.item)
            return ' '.join([self.name, text])
        else:
            return self.name
        
    def getItemClass(self):
        if isinstance(self.widget.widget.widget, ExpandBar):
            return SWTBotExpandItem
        return SWTBotTreeItem


class ExpandEvent(TreeEvent):
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Expand"

    def parseArguments(self, argumentString):
        item = TreeEvent.parseArguments(self, argumentString)
        if hasattr(item, "rowCount") and item.rowCount() == 0:
            raise UseCaseScriptError, "Item labelled '" + argumentString + "' in " + self.getClassDesc() + " is not expandable."
        return item
    
    def _generate(self, item):
        item.expand()

class CollapseEvent(TreeEvent):
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Collapse"

    def _generate(self, item):
        item.collapse()

class TreeClickEvent(TreeEvent):
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Selection"
    
    @staticmethod
    def getPairings(argumentString):
        pairings = []
        arg1SoFar, arg2 = "", argumentString
        for _ in range(argumentString.count(",")):
            arg1, arg2 = arg2.split(",", 1)
            if arg1SoFar:
                arg1SoFar += ","
            arg1SoFar += arg1
            pairings.append((arg1SoFar, arg2))
        return pairings
            
    def parseArguments(self, argumentString):
        if argumentString:
            try:
                return [ TreeEvent.parseArguments(self, argumentString) ]
            except UseCaseScriptError:
                if "," in argumentString:
                    for arg1, arg2 in self.getPairings(argumentString):
                        try:
                            return self.parseArguments(arg1) + self.parseArguments(arg2)
                        except UseCaseScriptError:
                            pass
                raise                    
        else:
            return []

    def shouldRecord(self, event, *args):
        # Seem to get selection events even when nothing has been selected...
        # Record if there is no item, or if we've pressed control (deselecting) or
        # if whatever we selected is still in the selection.
        return TreeEvent.shouldRecord(self, event, *args) and \
            (event.item is None or event.stateMask == SWT.CTRL or event.item in event.widget.getSelection())

    def _generate(self, items):
        if items:
            self.selectItems(items)
        else:
            self.widget.unselect()
            
    def selectItems(self, items):
        # Swtbot select and click methods doesn't seem to generate all events that a mouse click does.
        # In particular the select methods generate events without the item field set which can cause great confusion.
        runOnUIThread(self.widget.widget.widget.setSelection, [ item.widget for item in items ])
        items[0].click()
        for item in items[1:]:
            self.postControlEvent(SWT.KeyDown)
            item.click()
            self.postControlEvent(SWT.KeyUp)
                        
    def postControlEvent(self, eventType):
        event = Event()
        event.type = eventType
        event.keyCode = SWT.CTRL
        event.character = '\0'
        runOnUIThread(self.widget.display.post, event)
        
    def isStateChange(self):
        return True
    
    def outputForScript(self, event, *args):
        items = event.widget.getSelection()
        if not items:
            return self.name

        args = map(self.getTextToRecord, event.widget.getSelection())
        return ' '.join([self.name, ",".join(args)])
        
    def implies(self, stateChangeOutput, stateChangeEvent, *args):
        currOutput = self.outputForScript(*args)
        return currOutput.startswith(stateChangeOutput)


class TreeDoubleClickEvent(TreeEvent):
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "DefaultSelection"

    def _generate(self, item):
        item.doubleClick()

    def implies(self, stateChangeLine, stateChangeEvent, swtEvent, *args):
        return isinstance(stateChangeEvent, TreeClickEvent) and \
               stateChangeLine == stateChangeEvent.name + " " + stateChangeEvent.getTextToRecord(swtEvent.item)

class ListClickEvent(StateChangeEvent):
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Selection"
    
    def implies(self, stateChangeOutput, stateChangeEvent, *args):
        currOutput = self.outputForScript(*args)
        return currOutput.startswith(stateChangeOutput)

    def getStateText(self, *args):
        return ",".join(self.widget.selection())

    def _generate(self, indices):
        if len(indices) == 0:
            self.widget.unselect()
        else:
            self.widget.select(indices)
            
    def parseArguments(self, argumentString):
        return self.getIndices(argumentString) if argumentString else []
            
    def shouldRecord(self, *args):
        return not self.widget.widget.widget in CComboSelectEvent.internalWidgets and StateChangeEvent.shouldRecord(self, *args)

    def getIndices(self, argumentString):
        indices = []
        for itemText in argumentString.split(","):
            index = self.widget.indexOf(itemText)
            if index >= 0:
                indices.append(index)
            else:
                raise UseCaseScriptError, "Could not find item labelled '" + itemText + "' in list."
        return indices

class DateTimeEvent(StateChangeEvent):
    def __init__(self, *args, **kw):
        StateChangeEvent.__init__(self, *args, **kw)
        self.dateFormat = self.getDateFormat()

    def getDateFormat(self):
        if runOnUIThread(self.widget.widget.widget.getStyle) & SWT.TIME:
            return util.getDateFormat(SWT.TIME)
        else:
            return util.getDateFormat(SWT.DATE)
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "Selection"
    
    def getStateText(self, *args):
        return self.dateFormat.format(self.widget.getDate())
    
    def parseArguments(self, argumentString):
        try:
            return self.dateFormat.parse(argumentString)
        except ParseException:
            raise UseCaseScriptError, "Could not parse date/time argument '" + argumentString + \
                  "', not of format '" + self.dateFormat.toPattern() + "'."

    def _generate(self, currDate):
        self.widget.setDate(currDate)
        
        
class IconClickEvent(SignalEvent):
    @classmethod
    def getAssociatedSignal(cls, widget):
        return "ClickIcon"
    
    def shouldRecord(self, event, *args):
        label = event.widget
        return label.getImage() is not None and not label.getText() and SignalEvent.shouldRecord(self, event, *args)

    @classmethod
    def getRecordEventType(cls):
        return SWT.MouseUp
    
    def _generate(self, argumentString):
        method = AbstractSWTBotControl.getDeclaredMethod("click", [ Boolean.TYPE ])
        method.setAccessible(True)
        return method.invoke(self.widget.widget, [ True ])


class EventFinishedListener(Listener):
    def __init__(self, event, method):
        self.event = event
        self.method = method
        
    def handleEvent(self, e2):
        if e2 is self.event:
            storytext.guishared.catchAll(self.method, self.event)
            if not self.event.widget.isDisposed():
                self.event.widget.removeListener(self.event.type, self)


class DisplayFilter:
    instance = None
    def otherEventCount(self, event, isTriggeringEvent):
        relevantEvents = [ e for e in self.eventsFromUser if e is not event and not isTriggeringEvent(e) ]
        return len(relevantEvents)
        
    def getEventFromUser(self, event, isTriggeringEvent):
        if event in self.eventsFromUser:
            return not self.hasPreviousEventOnShell(event, isTriggeringEvent)
        else:
            if len(self.eventsFromUser) == 0:
                self.logger.debug("Rejecting event, it has not yet been seen in the display filter")
            else:
                self.logger.debug("Received event " + event.toString())
                self.logger.debug("Rejecting event, not yet processed " + repr([ e.toString() for e in self.eventsFromUser ]))
            return False
        
    def hasEvents(self):
        return len(self.eventsFromUser) > 0

    def __init__(self, widgetEventTypes):
        self.widgetEventTypes = widgetEventTypes
        self.eventsFromUser = []
        self.delayedAppEvents = []
        self.itemTextCache = {}
        self.logger = logging.getLogger("storytext record")
        DisplayFilter.instance = self
        
    def getShell(self, widget):
        # Note : widget might be an Item rather than a widget!
        if widget is not None and not widget.isDisposed():
            if hasattr(widget, "getShell"):
                return widget.getShell()
            elif hasattr(widget, "getParent"):
                return self.getShell(widget.getParent())

    def hasPreviousEventOnShell(self, event, isTriggeringEvent):
        widget = event.widget
        currShell = self.getShell(widget)
        if not currShell:
            return False

        for e in self.eventsFromUser:
            if e is event:
                return False
            elif not isTriggeringEvent(e) and self.getShell(e.widget) is currShell:
                self.logger.debug("Previous event on shell found: " + repr(e))
                return True
        return False
        
    def hasEventOfType(self, eventTypes, widget):
        return any((event.type in eventTypes and event.widget is widget for event in self.eventsFromUser))
        
    def addFilters(self, display):
        class DisplayListener(Listener):
            def handleEvent(listenerSelf, e): #@NoSelf
                storytext.guishared.catchAll(self.handleFilterEvent, e)

        for eventType in self.getAllEventTypes():
            self.logger.debug("Adding filter for events of type " + str(eventType))
            runOnUIThread(display.addFilter, eventType, DisplayListener())
            
        self.addApplicationEventFilter(display)

    def handleEventFinished(self, e):        
        # Any application events that were delayed should be no longer, if they haven't been handled yet
        for appEvent in self.delayedAppEvents:
            applicationEventDelay(appEvent, fromLevel=len(self.eventsFromUser), increase=False)
        self.delayedAppEvents = []
        self.logger.debug("Filter removed for event " + e.toString())
        self.eventsFromUser.remove(e)

    def cacheItemText(self, e):
        if e.item and not e.item.isDisposed():
            self.itemTextCache[e.item] = e.item.getText()
        elif hasattr(e.widget, "getSelectionIndex") and hasattr(e.widget, "getText") and e.type == SWT.Selection:
            self.itemTextCache[e.widget] = e.widget.getText()
            
    def handleFilterEvent(self, e):
        if self.shouldCheckWidget(e.widget, e.type):
            self.logger.debug("Filter for event " + e.toString())
            self.eventsFromUser.append(e)
            runOnUIThread(e.widget.addListener, e.type, EventFinishedListener(e, self.handleEventFinished))
            # Safe guard against the application changing the text before we can record
            self.cacheItemText(e)
        else:
            self.logger.debug("Filter ignored event " + e.toString())

    def addApplicationEventFilter(self, display):
        class ApplicationEventListener(Listener):
            def handleEvent(listenerSelf, e): #@NoSelf
                if e.text:
                    storytext.guishared.catchAll(self.registerApplicationEvent, e.text, e.data or "system")
        runOnUIThread(display.addFilter, applicationEventType, ApplicationEventListener())
 
    @classmethod       
    def registerApplicationEvent(cls, name, category, **kw):
        delayLevel = len(cls.instance.eventsFromUser) if cls.instance else 0
        if delayLevel:
            cls.instance.delayedAppEvents.append(name)
        applicationEvent(name, category, delayLevel=delayLevel, **kw)

    def shouldCheckWidget(self, widget, eventType):
        if not util.isVisible(widget):
            return False
        for cls, types in self.widgetEventTypes:
            if isinstance(widget, cls) and eventType in types and not self.hasComplexAncestors(widget):
                return True
        return False

    def hasComplexAncestors(self, widget):
        return isinstance(widget.getParent(), DateTime)

    def getAllEventTypes(self):
        eventTypeSet = set()
        for _, eventTypes in self.widgetEventTypes:
            eventTypeSet.update(eventTypes)
        return eventTypeSet
    
    @classmethod
    def removeApplicationEvent(cls, *args, **kw):
        applicationEventRemove(*args, **kw)
    
# There is no SWTBot class for these things, so we make our own. We aren't actually going to use it anyway...    
class FakeSWTBotTabFolder(AbstractSWTBot):
    pass

class FakeSWTBotCTabFolder(AbstractSWTBot):
    pass

# There is no way to type text in SWTBotCCombo, so make our own.
class FakeSWTBotCCombo(SWTBotCCombo):
    def typeText(self, text):
        self.setFocus()
        self.keyboard().typeText(text, SWTBotPreferences.TYPE_INTERVAL)

class BrowserUpdateMonitor(ProgressListener):
    def __init__(self, widget):
        self.widget = widget
    
    def changed(self, e):
        pass
    
    def completed(self, e):
        storytext.guishared.catchAll(self.onCompleted, e)
        
    def onCompleted(self, e):
        DisplayFilter.registerApplicationEvent(self.widget.getNameForAppEvent() + " to finish loading", "browser")


class EventPoster:
    def __init__(self, display):
        self.display = display

    def moveMouseAndWait(self, x, y):
        runOnUIThread(storytext.guishared.catchAll, self.postMouseMove, x, y)
        runOnUIThread(storytext.guishared.catchAll, self.waitForCursor, x, y)
       
    def postMouseMove(self, x ,y):
        event = Event()
        event.type = SWT.MouseMove
        event.x = x
        event.y = y
        self.display.post(event)

    def clickMouse(self, button):        
        runOnUIThread(storytext.guishared.catchAll, self.postMouseDown, button)
        runOnUIThread(storytext.guishared.catchAll, self.postMouseUp, button)

    def postMouseDown(self, button=1):
        event = Event()
        event.type = SWT.MouseDown
        event.button = button
        self.display.post(event)

    def postMouseUp(self, button=1):
        event = Event()
        event.type = SWT.MouseUp
        event.button = button
        self.display.post(event)
        
    def checkAndPostKeyPressed(self, keyModifiers):
        if keyModifiers & SWT.CTRL != 0:
            runOnUIThread(storytext.guishared.catchAll, self.postKeyPressed, SWT.CTRL, '\0')
            
    def checkAndPostKeyReleased(self, keyModifiers):
        if keyModifiers & SWT.CTRL != 0:
            runOnUIThread(storytext.guishared.catchAll, self.postKeyReleased, SWT.CTRL, '\0')
            
    def postKeyPressed(self, code, character):
        event = Event()
        event.type = SWT.KeyDown
        event.keyCode = code
        event.character = character
        self.display.post(event)
        
    def postKeyReleased(self, code, character):
        event = Event()
        event.type = SWT.KeyUp
        event.keyCode = code
        event.character = character
        self.display.post(event)
   
    def waitForCursor(self, x, y):
        while self.display.getCursorLocation().x != x and self.display.getCursorLocation().y != y:
            time.sleep(0.1)

    def moveClickAndReturn(self, *args, **kw):
        self.performAndReturn(self.moveAndClick, *args, **kw)
        
    def moveAndClick(self, x, y, keyModifiers=0, count=1, button=1):
        self.moveMouseAndWait(x, y)
        self.checkAndPostKeyPressed(keyModifiers)
        for _ in range(count):
            self.clickMouse(button)
        self.checkAndPostKeyReleased(keyModifiers)
        
    def performAndReturn(self, method, *args, **kw):
        currPos = runOnUIThread(self.display.getCursorLocation)
        method(*args, **kw)
        time.sleep(0.1)
        self.moveMouseAndWait(currPos.x, currPos.y)

class WidgetMonitor:
    startupError = None
    swtbotMap = { Button        : (SWTBotButton,
                                    [ (SWT.RADIO, SWTBotRadio),
                                     (SWT.CHECK, SWTBotCheckBox) ]),
                  MenuItem      : (SWTBotMenu, 
                                    [ (SWT.RADIO, SWTBotRadioMenu) ]),
                  Shell         : (SWTBotShell, []),
                  ToolItem      : ( SWTBotToolbarPushButton,
                                    [ (SWT.DROP_DOWN, SWTBotToolbarDropDownButton),
                                      (SWT.RADIO    , SWTBotToolbarRadioButton),
                                      (SWT.SEPARATOR, SWTBotToolbarSeparatorButton),
                                      (SWT.TOGGLE   , SWTBotToolbarToggleButton) ]),
                  Spinner       : (SWTBotSpinner, []),
                  Text          : (SWTBotText, []),
                  Label         : (SWTBotLabel, []),
                  Link          : (SWTBotLink, []),
                  List          : (SWTBotList, []),
                  Combo         : (SWTBotCombo, []),
                  CCombo        : (FakeSWTBotCCombo, []),
                  Table         : (StoryTextSwtBotTable, []),
                  TableColumn   : (SWTBotTableColumn, []),
                  Tree          : (SWTBotTree, []),
                  ExpandBar     : (SWTBotExpandBar, []),
                  DateTime      : (SWTBotDateTime, []),
                  TabFolder     : (FakeSWTBotTabFolder, []),
                  CTabFolder    : (FakeSWTBotCTabFolder, []),
                  Browser       : (SWTBotBrowser, [])
                  }
    def __init__(self, uiMap):
        self.bot = self.createSwtBot()
        self.widgetsMonitored = set()
        self.allMenus = set()
        self.uiMap = uiMap
        # Do this here, when things will be loaded with the right classloader
        # Might affect which event types are used. Has to be set up like this so RCP works.
        self.uiMap.scriptEngine.importCustomEventTypesFromSimulator(eventTypes)
        self.displayFilter = self.getDisplayFilterClass()(self.getWidgetEventTypes())
        self.widgetMonitorLock = Lock()
        self.failureHandlers = set()

    # This method should not be overridden by subclasses
    def handleReplayFailure(self, errorText, events):
        for handler in self.failureHandlers:
            done = handler.handleReplayFailure(errorText, events)
            if done:
                return
        if "Could not find row identified by" in errorText:
            # Problems with row identification in Table Cells
            # A common cause is that someone has edited the row identifier and not committed the change
            # We press Enter and hope for the best...
            if any((runOnUIThread(self.hasTableWithEditor, e) for e in events)):
                keyboard = KeyboardFactory.getSWTKeyboard()
                keyboard.pressShortcut([ Keystrokes.CR ])
        elif "MenuItem has already been disposed" in errorText or "no widget found" in errorText:
            runOnUIThread(self.recheckPopupMenus)

    def hasTableWithEditor(self, event):
        widget = event.widget.widget.widget
        if not widget.isDisposed() and isinstance(widget, Table) and len(widget.getChildren()) > 0:
            return any((not child.isDisposed() and child.isVisible() for child in widget.getChildren()))
        else:
            return False

    def recheckPopupMenus(self):
        for menu in self.allMenus:
            self.uiMap.logger.debug("Rechecking popup menu " + str(id(menu)))
            if not menu.isDisposed():
                self.monitorNewWidgets(menu)

    def getDisplayFilterClass(self):
        return DisplayFilter

    def createSwtBot(self):
        return SWTBot()
        
    def getWidgetEventTypes(self):
        return self.getWidgetEventInfo(lambda eventClass: eventClass.getSignalsToFilter())

    @classmethod
    def getWidgetEventTypeNames(cls):
        return cls.getWidgetEventInfo(lambda eventClass: [ eventClass.getAssociatedSignal(None) ])

    @classmethod
    def getWidgetEventInfo(cls, method):
        allEventTypes = []
        eventTypeDict = dict(eventTypes)
        for widgetClass, (defaultSwtbotClass, styleSwtbotInfo) in cls.swtbotMap.items():
            currEventTypes = set()
            for swtBotClass in [ defaultSwtbotClass] + [ cls for _, cls in styleSwtbotInfo ]:
                for eventClass in eventTypeDict.get(swtBotClass, []):
                    currEventTypes.update(method(eventClass))
            if currEventTypes:
                allEventTypes.append((widgetClass, currEventTypes))
        return allEventTypes
    
    def setUp(self):
        self.forceShellActive()
        self.setUpDisplayFilter()
        allWidgets = self.findAllWidgets()
        if len(allWidgets) > 0:
            self.uiMap.logger.debug("Monitoring all widgets in active shell...")
            self.monitorAllWidgets(list(allWidgets))
            self.uiMap.logger.debug("Done Monitoring all widgets in active shell.")
            return True
        else:
            return False
        
    def forceShellActive(self):
        if os.pathsep == ":": # os.name == "java", so can't find out that way if we're on UNIX
            # Need to do this for running under Xvfb on UNIX
            # Seems to throw exceptions occasionally on Windows, so don't bother
            runOnUIThread(self.bot.getFinder().getShells()[0].forceActive)

    def getDisplay(self):
        return self.bot.getDisplay()

    def setUpDisplayFilter(self):
        display = self.getDisplay()
        self.addMonitorFilter(display)
        self.displayFilter.addFilters(display)

    def addMonitorFilter(self, display):
        class MonitorListener(Listener):
            def handleEvent(listenerSelf, e): #@NoSelf
                storytext.guishared.catchAll(self.widgetShown, e)

        monitorListener = MonitorListener()
        runOnUIThread(display.addFilter, SWT.Show, monitorListener)
        runOnUIThread(display.addFilter, SWT.Paint, monitorListener)
        runOnUIThread(display.addFilter, SWT.Selection, monitorListener)
        
    def widgetShown(self, e):
        if self.shouldMonitor(e.widget):
            if isinstance(e.widget, Menu):
                e.widget.addListener(e.type, EventFinishedListener(e, self.monitorWidgetsFromEvent))
            else:
                self.monitorNewWidgets(e.widget, e.type == SWT.Show)
            
    def shouldMonitor(self, widget):
        # Don't try to monitor widgets before the shells they appear in!
        return widget not in self.widgetsMonitored and \
            (isinstance(widget, Shell) or not isinstance(widget, Control) or widget.getShell() in self.widgetsMonitored)
            
    def monitorWidgetsFromEvent(self, e):
        if not e.widget.isDisposed():
            self.monitorNewWidgets(e.widget, e.type == SWT.Show)
            
    def monitorNewWidgets(self, parent, findInvisible=True):
        if findInvisible:
            self.bot.getFinder().setShouldFindInvisibleControls(True)

        widgets = self.findDescendants(parent)
        if findInvisible:
            self.bot.getFinder().setShouldFindInvisibleControls(False)

        self.uiMap.logger.debug("Showing/painting widget of type " +
                                parent.__class__.__name__ + " " + str(id(parent)) + ", monitoring found widgets")
        if isinstance(parent, Menu):
            self.allMenus.add(parent)
        self.monitorAllWidgets(widgets)
        self.uiMap.logger.debug("Done Monitoring all widgets after showing/painting " + 
                                parent.__class__.__name__ + " " + str(id(parent)) + ".")
        
    def findDescendants(self, widget):
        if isinstance(widget, Menu):
            return ArrayList(self.getMenuItems(widget))
        else:
            matcher = IsAnything()
            return self.bot.widgets(matcher, widget)

    def getMenuItems(self, menu):
        items = []
        for item in menu.getItems():
            submenu = item.getMenu()
            if submenu:
                items += self.getMenuItems(submenu)
            else:
                items.append(item)
        return items

    def monitorAllWidgets(self, widgets):
        # Called both on the entire initial widget set and whenever a widgets is shown -> different threads
        # Use lock to avoid racing
        popupMenus = self.getPopupMenus(widgets)
        self.allMenus.update(popupMenus)
        widgets += popupMenus
        self.widgetMonitorLock.acquire()
        try:
            newWidgets = [ w for w in widgets if w not in self.widgetsMonitored and not w.isDisposed() ]
            self.uiMap.logger.debug(str(len(widgets)) + " widgets found, " + str(len(newWidgets)) + " new.")  
            self.widgetsMonitored.update(newWidgets)
        finally:
            self.widgetMonitorLock.release()
        for widget in self.makeAdapters(newWidgets):
            self.uiMap.monitorWidget(widget)
            self.monitorAsynchronousUpdates(widget)

    def monitorAsynchronousUpdates(self, widget):
        # Browsers load their stuff in the background, must wait for them to finish
        if widget.isInstanceOf(SWTBotBrowser):
            monitor = self.getBrowserUpdateMonitorClass()(widget)
            runOnUIThread(widget.widget.widget.addProgressListener, monitor)
            
    def getBrowserUpdateMonitorClass(self):
        return BrowserUpdateMonitor
    
    def addToolbarEmbeddedWidgets(self, widgets, matcher):
        # Should not be necessary, but SWTBot doesn't find these, as of 2.0.5
        # See https://bugs.eclipse.org/bugs/show_bug.cgi?id=356883
        # We work around and find them ourselves
        extraControls = []
        for widget in widgets:
            if isinstance(widget, ToolItem):
                control = runOnUIThread(widget.getControl)
                if control is not None:
                    extraControls.append(control)
        for control in extraControls:
            widgets.addAll(self.bot.widgets(matcher, control))

    def findAllWidgets(self):
        matcher = IsAnything()
        shell = runOnUIThread(self.getActiveShell)
        try:
            widgets = self.bot.widgets(matcher, shell)
        except TypeError:
            # Use member variable to allow other code to set this if things don't seem correctly set up
            self.startupError = "ERROR: classloader clash when trying to use SWTBot.\n" + \
                                "Possible causes: has Mockito been exported with the product (suggest to make dependency optional if so)?"
        if self.startupError:
            sys.stderr.write(self.startupError + "\n")
            SWTBotShell(shell).close()
            return []
        
        self.addToolbarEmbeddedWidgets(widgets, matcher)
        menus = self.bot.getFinder().findMenus(matcher)
        widgets.addAll(menus)
        return widgets
        
    def getPopupMenus(self, widgets):
        menus = []
        for widget in widgets:
            if isinstance(widget, Control):
                menuFinder = ContextMenuFinder(widget)
                menus += filter(lambda m: m not in self.widgetsMonitored, menuFinder.findMenus(IsAnything()))
        return menus

    @classmethod
    def findSwtbotClass(cls, widget, widgetClass):
        defaultClass, styleClasses = cls.swtbotMap.get(widgetClass)
        for currStyle, styleClass in styleClasses:
            if runOnUIThread(widget.getStyle) & currStyle:
                return styleClass
        return defaultClass

    def makeAdapters(self, widgets):
        adapters = []
        for widget in widgets:
            adapter = self.makeAdapter(widget)
            if adapter:
                adapters.append(adapter)
        return adapters

    @classmethod
    def makeAdapter(cls, widget):
        for widgetClass in cls.swtbotMap.keys():
            if isinstance(widget, widgetClass):
                swtbotClass = cls.findSwtbotClass(widget, widgetClass)
                try:
                    return WidgetAdapter.adapt(swtbotClass(widget))
                except RuntimeException:
                    # Sometimes widgets are already disposed
                    pass
        
    def getActiveShell(self):
        finder = self.bot.getFinder()
        activeShell = finder.activeShell()
        if activeShell is not None:
            return activeShell
        shells = filter(lambda s: s.getText() and s.isVisible(), finder.getShells())
        if shells:
            return shells[-1]
        
    def removeMousePointerIfNeeded(self):
        # If the mouse pointer is inside the window, this might cause accidental mouseovers and indeterminism. Relocate it to 0,0 if so
        # Recheck this if we resize the window
        display = self.getDisplay()
        shell = runOnUIThread(self.getActiveShell)
        self._removeMousePointerIfNeeded(display, shell)
        class MousePointerListener(Listener):
            def handleEvent(lself, e): #@NoSelf
                self._removeMousePointerIfNeeded(display, shell)
        runOnUIThread(shell.addListener, SWT.Resize, MousePointerListener())
        
    def _removeMousePointerIfNeeded(self, display, shell):
        def pointerInWindow():
            return shell.getClientArea().contains(display.getCursorLocation())
            
        if runOnUIThread(pointerInWindow):
            self.uiMap.logger.debug("Removing mouse pointer from window, to avoid accidental mouseovers")
            poster = EventPoster(display)
            poster.moveMouseAndWait(0, 0)
            self.uiMap.logger.debug("Mouse pointer now at 0,0")


        
eventTypes =  [ (SWTBotButton                   , [ SelectEvent ]),
                (SWTBotRadioMenu                , [ RadioSelectEvent ]),
                (SWTBotMenu                     , [ SelectEvent ]),
                (SWTBotToolbarPushButton        , [ SelectEvent ]),
                (SWTBotToolbarDropDownButton    , [ DropDownButtonEvent, DropDownSelectionEvent ]),
                (SWTBotToolbarRadioButton       , [ RadioSelectEvent ]),
                (SWTBotLabel                    , [ IconClickEvent ]),
                (SWTBotLink                     , [ LinkSelectEvent ]),
                (SWTBotRadio                    , [ RadioSelectEvent ]),
                (SWTBotSpinner                  , [ SpinnerSelectEvent ]),
                (SWTBotText                     , [ TextEvent, TextActivateEvent, TextContentAssistEvent ]),
                (SWTBotShell                    , [ ShellCloseEvent, ResizeEvent, FreeTextEvent ]),
                (StoryTextSwtBotTable           , [ TableColumnHeaderEvent, TableSelectEvent, TableDoubleClickEvent ]),
                (SWTBotTableColumn              , [ TableColumnHeaderEvent ]),
                (SWTBotTree                     , [ ExpandEvent, CollapseEvent, TreeClickEvent, TreeDoubleClickEvent ]),
                (SWTBotExpandBar                , [ ExpandEvent, CollapseEvent ]),
                (SWTBotList                     , [ ListClickEvent ]),
                (SWTBotCombo                    , [ ComboSelectEvent,  ComboTextEvent, TextActivateEvent ]),
                (FakeSWTBotCCombo               , [ CComboSelectEvent, CComboChangeEvent, TextActivateEvent ]),
                (FakeSWTBotTabFolder            , [ TabSelectEvent ]),
                (FakeSWTBotCTabFolder           , [ CTabSelectEvent, CTabCloseEvent ]),
                (SWTBotDateTime                 , [ DateTimeEvent ]),
                (SWTBotCheckBox                 , [ SelectEvent ]) ]
