#!python

"""PIL adapter.

Implements Tempest image operations using the Python Imaging Library:
http://www.pythonware.com/products/pil/
"""

import Image
import ImageChops

def render(parent):
    # load source image (in order to get dimensions)
    input_file = Image.open(parent.get_input_file())
    
    # create object for destination image
    output_file = Image.new('RGBA', input_file.size, 'white');
    
    # do any necessary preprocessing & transformation of the coordinates
    coordinates = parent.get_coordinates()
    max_rep = 0
    normal = {}
    
    for pair in coordinates:
        # normalize repeated coordinate pairs
        pair_key = '%sx%s' % pair
        if normal.has_key(pair_key):
            normal[pair_key][2] += 1
        else:
            normal[pair_key] = [ pair[0], pair[1], 1 ]
        # get the max repitition count of any single coord set in the data
        if normal[pair_key][2] > max_rep:
            max_rep = normal[pair_key][2]
    
    coordinates = normal.values()
    normal = None
    
    # load plot image (presumably greyscale)
    plot_file = Image.open(parent.get_plot_file())
    
    # calculate coord correction based on plot image size
    plot_correct = [ (plot_file.size[0] / 2), (plot_file.size[1] / 2) ]
    
    # colorize opacity for how many times at most a point will be repeated
    plot_file = colorize( plot_file, float(99 / max_rep) )
    
    # define a mask image for multiply operations
    white_mask = Image.new('RGBA', output_file.size, 'white')
    
    # paste one plot for each coordinate pair
    for pair in coordinates:
        x = pair[0] - plot_correct[0]
        y = pair[1] - plot_correct[1]
        
        # for how many times coord pair was repeated
        for i in range(0, pair[2]):
            # add to white mask centered on given coords
            point_mask = white_mask.copy()
            point_mask.paste(plot_file, (x, y))
            # multiply into output image
            output_file = ImageChops.multiply(output_file, point_mask)
    
    # open given spectrum and resize to 256px height
    color_file = Image.open(parent.get_color_file())
    
    # resizing lookup table, to simplify colorization
    color_file = color_file.resize( (color_file.size[0], 256,) )
    
    # get pixel access object (significantly quicker than getpixel method)
    color_pix = color_file.load()
    
    # apply color lookup table
    # note: optimized solution here courtesy of Nadia by way of Stackoverflow
    pix_data = output_file.getdata()
    output_file.putdata([color_pix[0, red] for (red, green, blue, alpha) in pix_data])
    
    # overlay heatmap over source image
    if parent.get_overlay() == True:
        output_file.putalpha( (256 * parent.get_opacity()) / 100 )
        input_file.paste(output_file, None, output_file)
        output_file = input_file
    
    # write destination image
    output_file.save(parent.get_output_file())
    
    # return true if successful
    return True

def colorize(image_file, opacity):
    # Image.blend(image1, image2, alpha) => image
    # 0.0 = image1, 1.0 = image2
    
    # blend with plain white
    white_point = Image.new('RGB', image_file.size, 'white')
    return Image.blend(image_file, white_point, ((100 - opacity) / 100) )
