#!/usr/bin/env python
# encoding: utf-8

"""TextRPG - Simple TextRPG module built on top of the 1d6 RPG library.

Usage: 
    - ministory_helper.py 
    Test the ministory (a testcase for the textrpg module)
    - textrpg.py 
    Start the internal test story of the textrpg. 

Plans: 
    - Simple to use functions in easy to read scriptfiles in the style of the ministory file. 
    - char.compete(other, skill_name) -> See who wins and by how much. 
    - a basic implementation as minimal api reference for anyrpg plugins.
    - Add the basic scripting function "python_interpreter(startup_data)", which shows an interactive python interpreter with the startup data already entered and interpreted. 


Ideas: 
    - Lazy loading modules, to be able to print stuff at once without having to print before the imports.


Basic design principles for the scripting language: 
    
    - The action is character centered wherever possible and useful. 
       -> char.say(text) instead of dialog(char, text)
    
    - Anything which affects only one character or any interaction between only a few characters which is initiated by one of them gets called from the character via char.action(). 
       -> char.compete_skill(char2, skill_name) instead of competition_skill(char1, char2, skill_name)
    
    - Anything which affects the whole scene, or a whole group of not necessarily interacting characters gets called as basic function via action() or as class in its own right via class.action(). 
       -> save([char1, char2]) instead of char1.save() char2.save()
    
    - The seperate class way should only be chosen, if the class can feel like a character in its own right and needs seperate states which may or may not be persistent over subsequent runs. 
       -> For example AI.choose_the_way(players_answer) or music.action()
    
    - Data should be stored inside the chars wherever possible. If a script gets started with the same character again, the situation should resemble the previous one as much as possible, except where dictated otherwise by the story_helper. 
       -> char.save() instead of 'on quit' store_char_data(char) + 'on start' load_char_data(char)
    
    - Actions should be written as verb_noun or simply verb. 
       -> char.say() and char.compete_skill() instead of char.text() and char.skill_compete()
    
    - In the story function, an action is a parameter of the story. 
       -> story(switch_background_image="bg_image.png")

The code for the TextRPG can be found at U{http://freehg.org/ArneBab/textrpg}

"""

try: 
    from anyrpg import __copyright__, __url__, __author__
except: 
    from rpg_lib.anyrpg import __copyright__, __url__, __author__

__version__   = '0.4.8' 
# __date__      = '7th March 2007' 


#print "...Loading rpg library..."


try: # Get the modules by relative path
    # AnyRPG classes
    from anyrpg import Char
    from anyrpg import Story as any_story
    # AnyRPG function for localizing. 
    from anyrpg import _
except: # get the modules by absolute path
    # AnyRPG classes
    from rpg_lib.anyrpg import Char
    from rpg_lib.anyrpg import Story as any_story
    # AnyRPG function for localizing. 
    from rpg_lib.anyrpg import _


# Changing things. I want to tell stories in lines, broken at word borders, so I want story() to use the split diag. 

class Story(any_story): 
    def __init__(self, *args, **kwds): 
        super(Story, self).__init__(*args, **kwds)
    
    def story(self, data=None, autoscroll=False, *args, **kwds): 
        """Tell a part of the story.
        
        Overridden to show "..." instead of blank lines (override commented out). """
        if data is not None: 
            data = _(data)
            for i in data.split("\n"): 
		self.split_diag(i, localize=False, autoscroll=True, *args, **kwds)
        

# Define helper functions. 
story_helper = Story()

### Lines needed in EVERY simple rpg scripting module (changing their effect for all places where they get used can only be done in the Story class) ###

def ask(question, *args, **kwds): 
    """Ask a question. 
    @return: The answer of the user."""
    return story_helper.ask(question, *args, **kwds)

def diag(text, localize=True, autoscroll=False, *args, **kwds): 
    """Output some text.

    The method used by diag is also used by story() to output text."""
    return story_helper.diag(text, localize=localize, autoscroll=autoscroll, *args, **kwds)

def story(text=None, *args, **kwds): 
    """Output story text line by line. Calls the method behind diag() for each line of text.
    
    Translation can be done for the whole text. 
    """
    return story_helper.story(text, *args, **kwds)

def give_exp(char, amount, *args, **kwds): 
    """Give one char experience and advance him/her with the exp."""
    return story_helper.give_exp(char, amount, *args, **kwds)

def save(chars=[], *args, **kwds): 
    """Save each passed char."""
    return story_helper.save(chars=chars, *args, **kwds)

def battle(me, other, *args, **kwds): 
    """Let two chars fight against each other. The first char is the attacker."""
    return me.battle(other, *args, **kwds)


### Test story lines ###

def greet(char): 
    story("Old man: Welcome traveller. You've come to the right place to learn about your heritage.")
    story("Sadly you aren' well liked around here.")
    story("Heges for example didn't like your father too much.")
    story("""Oh well, that was an understatment. You should better prepare yourself to face him. I think he'd love to see your face in the mud. Here's my knife. I like fights to be fair.
""")
    story("You say you don't know who we think you are?")
    story("Well, at least tell me your name then, and I'll tell you a bit about your father, after you won.")
    char.name = raw_input("My Name: ")
    story("You've got a nice name, " + char.name + ". Good luck!")


if __name__ == "__main__": 
    print "...Creating main character..."
    char = Char()
    greet(char)
    print "...Creating enemy character..."
    choss = Char(source='tag:1w6.org,2008:Hegen')
    choss.name = "Hegen"
    won = battle(char, choss)
    if won: 
        story(char.name + " won the fight.")
    else: 
        story(choss.name + " won the fight.")
    give_exp(char, 3)
    choss.upgrade(3)
    if won: 
        story("Well done " + char.name + ". I knew my trust in you was well placed. Now about your father...")
    else: 
        story("I didn't expect you to lose to him. Well, fate is a harsh teacher. Better luck next time!")
    
