"""The thread queue is an easy to use package that allows you to queue up runnable methods to be executed. The queue has
a limited size and will block further calls to enqueue until previous threads have been terminated. This is particularly
useful if you have a lot of threads to run, but only want to run a few simultaneously. An example would be a loop that
should spawn threads::

>>> def some_method(some_argument):
>>>     print(some_argument)

>>> tq = ThreadQueue(5) # Only run 5 threads simultaneously
>>> for argument in range(1, 20):
>>>     tq.enqueue(some_method, argument) # This call will block if the queue is full, and will
>>>                                       # enqueue the thread as soon as space becomes available
>>> tq.join() # This call will block until all threads are done and the queue is empty

.. moduleauthor:: Rolf Jagerman <rjagerman@gmail.com>

"""

__author__ = 'Rolf Jagerman'

import threading
import logging


class QueuedThread(threading.Thread):
    """A thread wrapper class used by the thread queue."""

    def __init__(self, thread_queue, runnable, *args, **kwargs):
        """Creates a new queued thread that will execute given method using given arguments.

        :param ThreadQueue thread_queue: The thread queue associated with this thread.
        :param function runnable: The runnable method that will be executed in this thread.
        """
        threading.Thread.__init__(self)
        self.thread_queue = thread_queue
        self.runnable = runnable
        self.args = args
        self.kwargs = kwargs

    def run(self):
        """Runs the supplied method with the arguments and notifies the thread queue when it finishes."""
        self.runnable(*self.args, **self.kwargs)
        self.thread_queue.deque(self)


class ThreadQueue:
    """An easy to use queue of threads that allows methods to be queued to be executed."""

    def __init__(self, number_of_threads=4):
        """Creates a new thread queue with given number of maximum threads.

        :param int number_of_threads: The maximum number of threads to simultaneously run.
        """
        self.number_of_threads = number_of_threads
        self.threads = []
        self.queue_lock = threading.Lock()
        self.enqueue_lock = threading.Lock()
        self.deque_lock = threading.Lock()
        self.full = threading.Condition(threading.Lock())

    def enqueue(self, runnable, *args, **kwargs):
        """Adds a new runnable method as a thread to the thread queue and starts executing it. If the queue is currently
        full, this method will block until space becomes available.

        :param function runnable: The runnable method to add to the thread queue.

        You might use it as following:

        >>> tq = ThreadQueue()
        >>> tq.enqueue(some_method, 'hello', 'world')

        This will execute :func:`some_method` in a thread, when the queue is ready.
        """
        with self.enqueue_lock:
            if len(self.threads) >= self.number_of_threads:
                logging.debug('waiting for queue space')
                with self.full:
                    self.full.wait()

        with self.queue_lock:
            logging.debug('queueing thread')
            thread = QueuedThread(self, runnable, *args, **kwargs)
            self.threads.append(thread)
            thread.start()

    def deque(self, queued_thread):
        """Removes a queued thread from the thread queue. This should not be called manually, but is instead done by the
        QueuedThread wrapper class when a supplied runnable method finished its execution.

        :param QueuedThread queued_thread: The queued thread to remove from the queue.
        """
        with self.queue_lock, self.full:
            logging.debug('removing queued thread')
            self.threads.remove(queued_thread)
            self.full.notify_all()

    def join(self):
        """Blocks execution until all threads in the thread queue finished executing and the queue is empty."""
        with self.enqueue_lock, self.full:
            while len(self.threads) > 0:
                logging.debug('waiting until threads are finished')
                self.full.wait()

