import sys
import os
import time
import atexit
from signal import SIGTERM
import json
import parsers
import servants
import logging

logger = logging.getLogger("daemon")


class Daemon(object):
		"""
		A generic daemon class.

		Usage: subclass the Daemon class and override the run() method
		"""
		def __init__(self, pidfile, stdin='/dev/null', stdout='/dev/null', stderr='/dev/null'):
				self.stdin = stdin
				self.stdout = stdout
				self.stderr = stderr
				self.pidfile = pidfile

		def daemonize(self):
				"""
				do the UNIX double-fork magic, see Stevens' "Advanced
				Programming in the UNIX Environment" for details (ISBN 0201563177)
				http://www.erlenstar.demon.co.uk/unix/faq_2.html#SEC16
				"""
				try:
						pid = os.fork()
						if pid > 0:
								# exit first parent
								sys.exit(0)
				except OSError, e:
						sys.stderr.write("fork #1 failed: %d (%s)\n" % (e.errno, e.strerror))
						sys.exit(1)

				# decouple from parent environment
				os.chdir("/")
				os.setsid()
				os.umask(0)

				# do second fork
				try:
						pid = os.fork()
						if pid > 0:
								# exit from second parent
								sys.exit(0)
				except OSError, e:
						sys.stderr.write("fork #2 failed: %d (%s)\n" % (e.errno, e.strerror))
						sys.exit(1)

				# redirect standard file descriptors
				sys.stdout.flush()
				sys.stderr.flush()
				si = file(self.stdin, 'r')
				so = file(self.stdout, 'a+')
				se = file(self.stderr, 'a+', 0)
				os.dup2(si.fileno(), sys.stdin.fileno())
				os.dup2(so.fileno(), sys.stdout.fileno())
				os.dup2(se.fileno(), sys.stderr.fileno())

				# write pidfile
				atexit.register(self.delpid)
				pid = str(os.getpid())
				file(self.pidfile, 'w+').write("%s\n" % pid)

		def delpid(self):
				os.remove(self.pidfile)

		def start(self):
				"""
				Start the daemon
				"""
				# Check for a pidfile to see if the daemon already runs
				try:
						pf = file(self.pidfile, 'r')
						pid = int(pf.read().strip())
						pf.close()
				except IOError:
						pid = None

				if pid:
						message = "pidfile %s already exist. Daemon already running?\n"
						sys.stderr.write(message % self.pidfile)
						sys.exit(1)

				# Start the daemon
				self.daemonize()
				self.run()

		def stop(self):
				"""
				Stop the daemon
				"""
				# Get the pid from the pidfile
				try:
						pf = file(self.pidfile, 'r')
						pid = int(pf.read().strip())
						pf.close()
				except IOError:
						pid = None

				if not pid:
						message = "pidfile %s does not exist. Daemon not running?\n"
						sys.stderr.write(message % self.pidfile)
						return
						# not an error in a restart

				# Try killing the daemon process
				try:
						while 1:
								os.kill(pid, SIGTERM)
								time.sleep(0.1)
				except OSError, err:
						err = str(err)
						if err.find("No such process") > 0:
								if os.path.exists(self.pidfile):
										os.remove(self.pidfile)
						else:
								print str(err)
								sys.exit(1)

		def restart(self):
				"""
				Restart the daemon
				"""
				self.stop()
				self.start()


class TorrentDaemon(Daemon):

		def restart(self, config_filepath):
			self.stop()
			self.start(config_filepath)

		def start(self, config_filepath):
			self.config_filepath = os.path.abspath(config_filepath)
			logger.info("Config file: %s", self.config_filepath)
			super(TorrentDaemon, self).start()

		def run(self):
			repeat_time = 86400

			while(True):
				try:
					config_file = open(self.config_filepath)
					config_json = json.load(config_file)
					config_file.close()
					logger.info("JSON. %s", json.dumps(config_json, indent=4))

					processed_data = read_config(config_json)
					repeat_time = processed_data['repeat_time']
					search_terms = processed_data['search_terms']
					choosen_parser = processed_data.get('parser')
					choosen_servant = processed_data.get('servant')

					logger.info("Repeat time: %s", repeat_time)
					logger.info("Search terms: %s", search_terms)

					if isinstance(choosen_parser, parsers.Parser):
						logger.info("Valid parser: %s", choosen_parser)
					else:
						logger.error("No valid parser.")

					if isinstance(choosen_servant, servants.Servant):
						logger.info("Valid servant: %s", choosen_servant)
					else:
						logger.error("No valid servant.")

					if (isinstance(choosen_parser, parsers.Parser) and
						isinstance(choosen_servant, servants.Servant)):

						for search in search_terms:
							keywords = search.get('keywords')
							if keywords is not None:
								del search['keywords']
								logger.info("Keywords: %s", keywords)
								logger.info("Additional arguments: %s", search)

								torrents = choosen_parser.get_torrents(keywords, **search)
								choosen_servant.serve_torrents(torrents)

								for torrent in torrents:
									logger.info("Torrent: %s", torrent)

				except IOError:
					logger.critical("IO error with config file.")

				time.sleep(repeat_time)


def read_config(config_json):
	data = {}

	for key, value in config_json.iteritems():
		if key == "search_terms":
				data['search_terms'] = value
		elif key == "repeat_time":
			data['repeat_time'] = value
		elif key == "parser" and type(value) is dict:
			data['parser'] = parsers.get_parser(value)
		elif key == "servant" and type(value) is dict:
			data['servant'] = servants.get_servant(value)

	return data
