import abc
import sys
import os
import subprocess
import requests
import logging
from bs4 import BeautifulSoup

logger = logging.getLogger("servants")


def get_servant(args):
	servant_type = args.get("type")
	del args["type"]

	module = __name__

	if '.' in servant_type:
		tmp = servant_type.split(".")
		module = tmp[0]
		servant_type = tmp[1]
		__import__(module)

	className = servant_type + "Servant"
	return getattr(sys.modules[module], className)(**args)


class Servant(object):
	__metaclass__ = abc.ABCMeta

	def __init__(self, **kwargs):
		pass

	@abc.abstractmethod
	def serve_torrents(self, torrents):
		return


class WebUIServant(Servant):

	def __init__(self, **kwargs):
		self.ip = kwargs.get("ip")
		self.port = kwargs.get("port")
		self.username = kwargs.get("username")
		self.pw = kwargs.get("password")

		if (self.ip is None or self.port is None or
			self.username is None or self.pw is None):
			logger.error("Not all necessary arguments were passed for WebUI servant")
			return

		self.link = ("http://%s:%s/gui/") % (self.ip, self.port)
		request_token_link = ("%stoken.html") % (self.link)
		r = requests.get(request_token_link, auth=(self.username, self.pw))

		soup = BeautifulSoup(r.text)
		self.TOKEN = soup.find_all('div')[0].string
		self.GUID = r.cookies['GUID']

	def serve_torrents(self, torrents):
		payload = {'token': self.TOKEN, 'action': 'add-url'}
		cookies = dict(GUID=self.GUID)

		for torrent in torrents:
			payload['s'] = torrent
			r = requests.get(self.link,
				cookies=cookies,
				params=payload,
				auth=(self.username, self.pw))

			logger.info("Url: %s", r.url)
			logger.info("Status code: %s", r.status_code)


class LocalServant(Servant):

	def serve_torrents(self, torrents):
		for torrent in torrents:
			if torrent.index("magnet:") == 0:
				self.open_torrent_client(torrent)

	def open_torrent_client(self, magnet_link):
		if sys.platform.startswith('darwin'):
			subprocess.call(('open', magnet_link))
		elif os.name == 'nt':
			os.startfile(magnet_link)
		elif os.name == 'posix':
			subprocess.call(('xdg-open', magnet_link))
