# -*- coding: utf-8 -*-
'''
Created on Tue 16 Aug 2011

@author: leewei
'''
# ==============================================================================
# Copyright© 2011 LShift - Lee Wei <leewei@lshift.net>
#
# Please view LICENSE for additional licensing information.
# ==============================================================================

from __future__          import with_statement
from config              import *
from datetime            import timedelta
from genshi              import HTML
from genshi.builder           import tag
from genshi.filters.transform import Transformer, StreamBuffer
from genshi.util         import *
from trac.core           import *
from trac.mimeview.api   import Context
from trac.wiki.formatter import format_to_html
from trac.web            import ITemplateStreamFilter
from estimationtools.utils import *
from trac.util.datefmt   import from_utimestamp
from trac.ticket.model   import Milestone
from genshi.path         import Path

class BurndownChartEmbed(Component):
    """
        Base class to embed '[[BurndownChart(...)]]' macros into
        /roadmap, /milestone.
    """
    implements(ITemplateStreamFilter)

    env = log = config    = None # make pylint happy
    __container_div_class = "bdc_container"
    __estimation_field    = get_estimation_field() # 'estimatedtime'

    def __init__(self, *args, **kwargs):
        # set default config options
        init_config(self.env, self.__estimation_field)

        if not self.env.config.has_option('components', 'estimationtools.*') \
        or self.env.config.get('components', 'estimationtools.*') != 'enabled':
            # No estimation tools plugin found. Disable plugin & log error.
            self.log.error("TracLShiftBurndownChart(%s)::" \
                "EstimationTools is not installed/enabled. Component disabled."\
                % (self.__class__.__name__,))
            self.env.disable_component(self)
            raise TracError("BurndownChartEmbed::Invalid install/config found.")

        Component.__init__(self, *args, **kwargs)

    ############################################################################
    # ITemplateStreamFilter methods #
    #  - Transform the generated content by filtering the Genshi event stream
    #    generated by the template, prior to its serialization
    ############################################################################
    def filter_stream(self, req, method, filename, stream, data):
        """ Dispatches incoming request to relevant filter stream handler. """

        if   filename == 'roadmap.html':
            return self._do_roadmap_filter(req, method, filename, stream, data)
        elif filename == 'milestone_view.html':
            return self._do_view_filter   (req, method, filename, stream, data)
        else:
            return stream

    def _milestone_has_hours(self, milestone):
        """
            Queries DB and returns whether milestone contains
            non-closed tickets with total(estimatedtime) > 0h.
        """

        conn = self.env.get_read_db()
        cursor = conn.cursor()
        cursor.execute(
        """
            SELECT value from ticket_custom
            WHERE name=%s AND ticket IN
                (SELECT id FROM ticket t
                 WHERE t.status<>'closed'
                 AND milestone=%s)
        """, (self.__estimation_field, milestone))
        rows = cursor.fetchall()

        for row in rows:
            _, number, _ = re.split(r'^(\d+).*$', row[0])
            if int(number) > 0:
                return True

        return False

    def _start_end_dates(self, milestone):
        """
            Retrieves resolution:(start_date, end_date) to display milestone's
            burndown chart by querying for (non-closed only) tickets in DB
            belonging to this milestone.

            Returns None if either start_date or end_date is not found.
        """

        if isinstance(milestone, Milestone):
            milestone = milestone.name

        start_end_dates = None
        conn = self.env.get_read_db()
        cursor = conn.cursor()
        cursor.execute(
        """
            SELECT min(time), max(time) FROM ticket
            WHERE status<>'closed' AND milestone=%s
            ORDER BY time ASC
        """, (milestone,))
        row = cursor.fetchone()

        if not row or not row[0] or not row[1]:
            return None
        else:
            return [ from_utimestamp(date) for date in row ]

    def _create_burndown_chart(self, args):
        """
            Generates [[BurndownChart(...)]] macro from supplied arguments
            to be expanded, and wraps this within a parent container div.

            Note: 8w+1d restriction in start-end date range enforced, due to
            limitation in Google's Chart API.
        """

        milestone_name, start_time, end_time, req = flatten(args)
        title = milestone_name.title()

        if (end_time - start_time) > timedelta(weeks=8, days=1):
            raise RuntimeError('range(dates)>(8w,1d): too large for Chart API')

        start_time -= timedelta(days=1)
        end_time   += timedelta(days=1)
        startdate   = start_time.date().isoformat()
        enddate     = end_time.date().isoformat()

        params = {
            'milestone' : milestone_name,
            'title'     : title,
            'startdate' : startdate,
            'enddate'   : enddate,
            'width'     : 1000,
            'height'    : 300
        }
        macro = '[[BurndownChart(%s)]]' % ','.join([
            '%s=%s' % (key, str(value)) for key,value in params.iteritems()
        ])

        chart = HTML(
                tag.div(
                    format_to_html(self.env, Context.from_request(req), macro),
                    class_=self.__container_div_class)
                ) | Transformer(".//p").unwrap()

        return chart

    def _do_filter_init(self, ms_selector, ms_id, req):
        """ Abstraction of common code for _do_{view,roadmap}_filters. """

        # return - content XPath selector
        classes = \
            " or ".join([("@class='%s'" % c) for c in ["info", "description"]])
        content = "%(ms_selector)s/child::div[%(classes)s]" % locals()

        # return - Transformer for BurndownChart
        chart_transform = Transformer() # default value

        # sanity check for whether any tickets belonging to milestone exist
        dates = self._start_end_dates(ms_id)

        # sanity check for whether among these, total(open tickets) > 0h
        has_hours = self._milestone_has_hours(ms_id)

        if dates and has_hours:
            chart_selector = "%(ms_selector)s/table/tr[1]" % locals()
            chart_macro = self._create_burndown_chart([ms_id, dates, req])
            chart_transform = \
                Transformer(chart_selector).append(tag.td(chart_macro))

        return content, chart_transform

    def _do_view_filter(self, req, method, filename, stream, data):
        """
            Genshi transformer for filter stream to view individual milestones.
            Returns genshi.core.Stream object, which can be .render('html')
        """

        ms_selector  = "//div[@class='milestone']"
        ms_id        = req.args['id']
        content, chart_transform = self._do_filter_init(ms_selector, ms_id, req)
        streambuffer = StreamBuffer()
        transform = Transformer(content) \
            .cut(streambuffer, accumulate=True).buffer().end() \
            .select(content).remove().end() \
            .select("%(ms_selector)s/h1" % locals()) \
            .after(tag.table(tag.tr(tag.td(streambuffer)))).end()

        return stream | transform | chart_transform

    def _do_roadmap_filter(self, req, method, filename, stream, data):
        """
            Genshi transformer for filter stream to view roadmap.
            Returns genshi.core.Stream object, which can be stringified
            using .render('html')
        """

        # deal with pesky form#prefs box interfering with layout
        stream = stream | Transformer("//div[@class='milestones']") \
               .before(tag.br(clear="all"))
        html   = stream.render('html')

        copy   = HTML(html)
        # milestones to generate charts for
        target_milestones = map(lambda x: x.name, data.get('milestones', {}))
        # milestones present in /roadmap page
        nodes  = Path("//div[@class='milestone']//em/text()")
        current_milestones = [ name for (_, name, _) in nodes.select(copy)]
        current_milestones = zip(
            xrange(1, len(current_milestones)+1), current_milestones
        )
        milestones_to_update = \
            [ (no, name) for (no, name) in current_milestones
                if name in target_milestones ]

        stream = HTML(html)
        for (index, milestone_name) in milestones_to_update:
            # wrap existing content within table
            ms_selector = "//div[@class='milestone'][%(index)d]" % locals()
            content, chart_transform = \
                self._do_filter_init(ms_selector, milestone_name, req)
            streambuffer = StreamBuffer()
            transform = Transformer(content) \
                .cut(streambuffer, accumulate=True).buffer().end() \
                .select(content).remove().end() \
                .select(ms_selector) \
                .append(tag.table(tag.tr(tag.td(streambuffer)))).end()
            stream = stream | transform | chart_transform

        return stream
