# Part of UPE
import cmd
import readline  # lint:ok


class manifest:

    def __init__(self, manifest='manifest.yaml'):
        self.manifest = str(manifest)

    def read(self):
        import yaml
        manifest = self.manifest
        with open(manifest, 'r') as data:
            rawdata = yaml.safe_load(data)
        package = str(rawdata['package'])
        pkgname = str(rawdata['pkgname'])
        version = int(rawdata['version'])
        release = float(rawdata['release'])
        try:
            codename = str(rawdata['codename'])
        except KeyError:
            codename = ''
        pkgrel = int(rawdata['pkgrel'])
        try:
            deps = tuple(rawdata['deps'])
            if deps == ():
                raise TypeError
        except TypeError:
            deps = None
        try:
            buildd = tuple(rawdata['buildd'])
            if buildd == ():
                raise TypeError
        except TypeError:
            buildd = None
        try:
            optd = rawdata['optd']
        except TypeError:
            optd = None
        arch = tuple(rawdata['arch'])
        licensing = str(rawdata['license'])  # To avoid overwriting license
        tags = rawdata['tags']
        print('Package internal name: ' + package)
        print('Package official name: ' + pkgname)
        if codename:
            print('Package version: {}.{} -- {}'.format(version, release,
                codename))
        else:
            print('Package version: {}.{}'.format(version, release))
        print('Package release: {}'.format(pkgrel))
        if not deps:
            print('No dependencies')
        else:
            print('Depends:')
            for d in deps:
                print(d)
            print('----')
        if not buildd:
            print('No build dependencies.')
        else:
            print('Build dependencies:')
            for d in buildd:
                print(d)
            print('----')
        if not optd:
            print('No optional dependencies')
        else:
            print('Optional dependencies:')
            for d in optd:
                if optd[d]:
                    print('{}: {}'.format(d, optd[d]))
                else:
                    print(d)
            print('----')
        if arch == ('all',):
            print('Compatible with all architectures.')
        else:
            print('Compatible with the following architectures:')
            for a in arch:
                print(a)
        print('Licensed under the "{}" license'.format(licensing))
        # <Version tags>
        if tags['version']['alpha']:
            print('Alpha version')
        elif tags['version']['beta']:
            print('Beta version')
        elif tags['version']['unstable']:
            print('Unstable version')
        # <Version tags/>
        if tags['lts']:
            print('Version with Long-Term Support')
        if tags['binary']:
            print('Precompiled binary')
        if tags['comercial']:
            print('Comercial software')

    def convert(self, of='manifest.json'):
        import yaml
        import json
        manifest = self.manifest
        with open(manifest, 'r') as m:
            data = yaml.safe_load(m)
        with open(of, 'w') as f:
            json.dump(data, f)

    def revert(self, of='manifest.yaml'):
        import json
        import yaml
        manifest = self.manifest
        with open(manifest, 'r') as m:
            data = json.load(m)
        with open(of, 'w') as f:
            yaml.dump(data, f)


class CmdError(Exception):
    pass


class buildcmd(cmd.Cmd):
    intro = "You should'nt be here. This is for internal use ONLY."
    prompt = 'UPE Build Shell ==> '

    def __init__(self, builddir, srcdir):
        from os.path import join  # lint:ok
        self.builddir = builddir
        self.srcdir = srcdir

    def default(line):
        raise CmdError('Tried to execute an illegal command: ' + line)

    def do_init(self):
        '''Create the build directory'''
        from os import mkdir
        mkdir(self.builddir)
        print('Created {} as build directory'.format(self.builddir))

    def do_adddir(self, directory, perm=0o777):
        '''Create a directory'''
        from os import makedirs
        makedirs(join(self.builddir, directory), perm)  # lint:ok
        print('Creted {} inside {}'.format(directory, self.builddir))

    def do_install(self, rsrc, rdst, perm=0o755, owner='root'):
        '''Copy the src file to dst'''
        from shutil import copyfile
        src = join(self.srcdir, rsrc)  # lint:ok
        dst = join(self.builddir, rdst)  # lint:ok
        copyfile(src, dst)
        pass


def make(manifest='manifest.json', builds='build.ubs', builddir='build',
    srcdir='src'):
    import json
    from os import chdir
    from os.path import join
    from platform import system, machine
    from shutil import make_archive
    with open(manifest, 'r') as man:
        data = json.load(man)
    data['build'] = {'os': system().lower(), 'arch': machine().lower()}
    package = data['package']
    validchars = tuple('abcdefghijklmnopqrstuvwxyz' + '0123456789' + '-_.')
    pkgcode = list(package.lower())
    n = -1
    for l in pkgcode:
        n = n + 1
        if not l in validchars:
            pkgcode[n] = '_'
    package = ''.join(pkgcode)
    with open(builds, 'r') as bs:
        l = None
        while l != '':
            l = bs.readline
            buildcmd(builddir, srcdir).onecmd(l)
    with open(join(builddir, 'manifest.json'), 'w') as destman:
        json.dump(data, destman)
    chdir(builddir)
    make_archive(join('..', '{}.pkg'.format(package)), 'bztar')
    chdir('..')