# Part of UPE


def _getrpath(path):
    import os
    ret = os.path.normpath(os.path.join(os.path.abspath(os.sep),
        os.path.relpath(path)))
    return ret


class DependencyError(Exception):
    pass


class NotCompatible(Exception):
    pass


def linstall(*args):
    from upe.config import logging as logconfig
    import logging
    import warnings
    lvl = getattr(logging, logconfig.loglevel)
    logging.basicConfig(filename=logconfig.defstyle['file'], level=lvl,
        format=logconfig.defstyle['format'])
    logging.warning('Deprecated: linstall()')
    print('WARNING: Used `linstall()`, which is deprecated.')
    install(args)


def install(pkgfile):
    '''Install a package from a local file'''
    from upe.config import logging as logconfig
    import logging
    lvl = getattr(logging, logconfig.loglevel)
    logging.basicConfig(filename=logconfig.defstyle['file'], level=lvl,
        format=logconfig.defstyle['format'])
    logging.info('Installing %s', pkgfile)
    try:
        import json
        import shutil
        from tempfile import mkdtemp
        import os
        import filecmp
        from platform import system, machine
        from upe import config
        from upe import query
    except ImportError as exc:
        logging.exception(exc)
        raise
    else:
        logging.debug('Imported needed modules')
    tmpdir = mkdtemp()
    logging.info('Created temporal directory')
    shutil.unpack_archive(pkgfile, tmpdir, format='bztar')
    logging.info('Unpacked archive')
    currentdir = os.getcwd()
    os.chdir(tmpdir)
    with open('manifest.json', 'r') as manifest:
        data = json.load(manifest)
    logging.info('Read manifest')
    if data['build']['os'] != system().lower() or data['build']['arch'] !=\
         machine().lower():
        try:
            raise NotCompatible
        except NotCompatible as exc:
            logging.exception(exc)
            raise
        else:
            logging.debug('Checked that this computer is compatible')
    package = data['package']
    logging.debug('Got package name: %s', package)
    deptest = query.pkgin(data['deps'])
    if not deptest[0]:
        try:
            raise DependencyError('Missing dependencies: {}'.format(deptest[1]))
        except DependencyError as exc:
            logging.exception(exc)
            raise
        else:
            logging.info('Checked dependencies')
    files = []
    for obj in os.walk(package):
        for directory in obj[1]:
            rpath = _getrpath(directory)
            if os.path.isdir(rpath) is False:
                os.mkdir(rpath, mode=oct(os.stat(directory).st_mode[-3:]))
                os.chown(rpath, os.stat(directory).st_uid,
                    os.stat(directory).st_gid)
            files.append(rpath)
            logging.debug('Created directory: %s', rpath)
        for file in obj[2]:
            rpath = _getrpath(file)
            if os.path.isfile(rpath) is True:
                if filecmp.cmp(file, rpath) is False:
                    copy = True
                else:
                    copy = False
            else:
                copy = True
            if copy is True:
                shutil.copy2(file, rpath)
                os.chown(rpath, os.stat(file).st_uid, os.stat(file).st_gid)
            files.append(rpath)
            logging.debug('Copied file: %s', rpath)
    newdata = {package: {}}
    for key in data:
        if key == 'package':
            continue
        newdata[package][key] = data[key]
    newdata[package]['files'] = files
    pkgdb = config.packages.database
    with open(pkgdb, 'r') as db:
        olddata = json.load(db)
    olddata.update(newdata)
    with open(pkgdb, 'w') as db:
        json.dump(olddata, db)
    logging.debug('Added manifest to the database')
    os.chdir(currentdir)
    logging.info('Successfully installed %s', pkgfile)


def remove(package, ignore=[]):
    '''Uninstall an installed package. It will check if it's needed by
    another package as dependency (except for those in ignore).'''
    from upe.config import logging as logconfig
    import logging
    lvl = getattr(logging, logconfig.loglevel)
    logging.basicConfig(filename=logconfig.defstyle['file'], level=lvl,
        format=logconfig.defstyle['format'])
    logging.info('Installing %s', pkgfile)
    try:
        import json
        import os
        from upe import config
        from upe import query
    except ImportError as exc:
        logging.exception(exc)
        raise
    else:
        logging.debug('Imported needed modules')
    depchk = query.checkneeded(package, ignore)
    if depchk[0] is True:
        try:
            raise DependencyError('Needed by ' + depchk[1])
        except DependencyError as exc:
            logging.exception(exc)
            raise
        else:
            logging.info('Checked dependencies')
    pkgdb = config.packages.database
    with open(pkgdb, 'r') as db:
        data = json.load(db)
    logging.debug('Read package database')
    for file in data[package]['files']:
        if os.path.isdir(file):
            try:
                os.rmdir(file)
            except OSError:
                logging.debug("Can't remove %s : non-empty directory", file)
            else:
                logging.debug('Removed directory: %s', file)
        elif os.path.isfile(file):
            os.remove(file)
            logging.debug('Removed file: %s', file)
    del(data[package])
    logging.info("Deleted %s 's entry from the database", package)
    with open(pkgdb, 'w') as db:
        json.dump(data, db)
