# -*- coding: utf-8 -*-
# Copyright (C) 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Test `DirectoryCache`."""


from __future__ import absolute_import

import time
import hashlib
import unittest
import random
import subprocess
import tempfile
import os
import shutil
import urllib2


__all__ = ("RandomFilesTestCase",)


class RandomFilesTestCase(unittest.TestCase):

    """Tests for directory cache."""

    def setUp(self):
        """Prepare."""
        from urlreader import URLReader
        from urlreader.caches import DirectoryCache
        # Start an HTTP server.
        port = random.randrange(8000, 8100)
        self.host = "http://localhost:%d/" % port
        self.htdocs = tempfile.mkdtemp()
        self.cache_dir = tempfile.mkdtemp()
        self.httpd = subprocess.Popen(
            ("python2", "-m", "SimpleHTTPServer", str(port)),
            cwd=self.htdocs,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE)
        # Wait until the server is ready.
        while True:
            try:
                urllib2.urlopen(self.host)
            except (urllib2.URLError, ValueError):
                time.sleep(0.1)
            else:
                break
        self.urlreader = URLReader(caches=(DirectoryCache(self.cache_dir),))

    def tearDown(self):
        """Stop server and remove temporary files."""
        self.httpd.terminate()
        self.httpd.wait()
        shutil.rmtree(self.htdocs)
        shutil.rmtree(self.cache_dir)

    def test_big(self):
        """Test that a big file can be cached."""
        subprocess.Popen(("dd", "if=/dev/urandom", "of=big",
                          "bs=1M", "count=10"),
                         cwd=self.htdocs,
                         stderr=subprocess.PIPE).communicate()
        # Collisions probably won't be a problem here.
        digest = hashlib.md5()
        for fragment in self.urlreader(self.host + "big"):
            digest.update(fragment)
        md5 = digest.digest()
        os.unlink(os.path.join(self.htdocs, "big"))
        digest = hashlib.md5()
        for fragment in self.urlreader(self.host + "big"):
            digest.update(fragment)
        md5_2 = digest.digest()
        self.assertEquals(md5, md5_2)
