# -*- coding: utf-8 -*-
# Copyright (C) 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
This module contains the base class for document caches.
"""


from __future__ import absolute_import

from urlreader.document import Document
from urlreader.exceptions import NonWritableCacheError


__all__ = ("Cache",)


class Cache(object):

    """A cache for document objects.

    It can be used like a dictionary, allowing adding, getting and
    deleting `urlreader.document.Document` or
    `urlreader.exceptions.ReaderError` objects indexed by their URLs.

    Some caches cannot update their content and raise
    `urlreader.exceptions.NonWritableCacheError` when trying to
    replace their elements.  Other caches will update their content
    only when assigning documents to their URLs.

    Some implementations might support also iterating over URLs of the
    cached documents.

    Subclasses should override the `__getitem__`, `__setitem__`,
    `__delitem__` and `__iter__` methods if needed.  By default the
    cache is empty, readonly and raises `NotImplementedError` when
    trying to iterate its content.

    To save document's content, some caches might store a different
    object instead of the one added to it, so after changing an entry
    it should be obtained from this cache before being used again.
    """

    def __getitem__(self, key):
        """Get a document from the cache."""
        assert self
        assert isinstance(key, basestring)
        raise KeyError("empty cache")

    def __setitem__(self, key, value):
        """Change a document stored in the cache."""
        assert self
        assert isinstance(key, basestring)
        assert isinstance(value, Document)
        raise NonWritableCacheError()

    def __delitem__(self, key):
        """Remove a document from the cache.

        Might do nothing if the cache doesn't support manual removal
        of documents.
        """
        assert self
        assert isinstance(key, basestring)

    def __contains__(self, key):
        """True if ``self[key]`` doesn't raise `KeyError`."""
        try:
            self[key]
        except KeyError:
            return False
        else:
            return True

    def __iter__(self):
        """Iterate URLs of documents in the cache.

        Not supported by all implementation.
        """
        assert self
        raise NotImplementedError("Cache.__iter__ not overridden")
