# -*- coding: utf-8 -*-
# Copyright (C) 2010, 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
This module contains a decorator to cache read-only property values.

Unlike other URLReader code, this uses its own cache (a separate
attribute of the object having this property) and keeps it until the
object is collected.

It is not used in core URLReader modules, but it could be useful in
implementations of caches or backends, and in other programs.
"""


__all__ = ("cachedproperty",)


def cachedproperty(getter):
    """Decorator for properties with cached values.

    A property is made using the decorated function as its getter.
    The docstring of this function is used also for the property.

    The property can be set to a value, using it instead of the
    *getter* return value as the cache.
    """
    name = getter.__name__
    attrib = "_cache__" + name

    def cached_getter(self):
        """Wrapper for property's getter."""
        try:
            return getattr(self, attrib)
        except AttributeError:
            value = getter(self)
            setattr(self, attrib, value)
            return value
    setattr(cached_getter, "__doc__", getter.__doc__)

    def cached_setter(self, value):
        """Property setter storing the value in cache."""
        setattr(self, attrib, value)

    return property(cached_getter, cached_setter, doc=getter.__doc__)
