# -*- coding: utf-8 -*-
# Copyright (C) 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
This module contains the base class for backends.
"""


from __future__ import absolute_import

from urlreader.exceptions import UnknownURLError


__all__ = ("Backend",)


class Backend(object):

    """An object obtaining documents identified by URLs.

    A single instance of this object should be used to fetch multiple
    objects, since some implementations might use shared connections
    for multiple documents.
    """

    @property
    def user_agent(self):
        """The HTTP ``User-Agent`` header used by the backend.

        Should be possible to set a different one.  If HTTP is not
        used, then it should be ignored.
        """
        assert self

    def fetch(self, url):
        """Return an `urlreader.document.Document` with this *url*.

        Must be overridden by subclasses.
        """
        assert self
        assert url
        raise UnknownURLError("Backend.fetch not overridden")

    def refetch(self, document):
        """Fetch the *document* again if changed.

        Returns another instance of `urlreader.document.Document` for
        the same URL, or `None` if the document is known to be
        unchanged.

        The document is refetched also if it cannot be determined
        otherwise to be not changed, or it doesn't have needed data to
        be checked (e.g. if it was obtained by a different backend).
        The default implementation always treats the document as
        changed.
        """
        return self.fetch(document.request_url)
