# -*- coding: utf-8 -*-
# Copyright (C) 2010, 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Test `urlreader.reader` and related modules.
"""


from __future__ import absolute_import

from shutil import rmtree
from tempfile import mkdtemp
import unittest

from urlreader.document import Document
from urlreader.caches import DictionaryCache, DirectoryCache


__all__ = ("DictCacheTestCase", "DirectoryCacheTestCase")


class OneUseDocument(Document):

    """`Document` subclass returning document's content only once."""

    @property
    def content(self):
        """Return the URL on first call, ``None`` on later calls."""
        value = self._url
        self._url = None
        try:
            return value.encode("ascii")
        except AttributeError:
            return value

    def __iter__(self):
        """Iterate the content."""
        yield self.content


class DictCacheTestCase(unittest.TestCase):

    """Test `dict_cache`."""

    def test_cached_document(self):
        """Test that a document can be cached"""
        cache = DictionaryCache()
        for url in ("http://example.org/", "http://other.example.com"):
            self.assert_(url not in cache)
            cache[url] = OneUseDocument(url)
            self.assert_(url in cache)
            self.assertEquals(cache[url].content, url.encode("ascii"))
            self.assert_(cache[url].content is None)
        for url in ("http://example.org/", "http://other.example.com"):
            self.assert_(cache[url].content is None)

    def test_single_element_cache(self):
        """Test that cache can limit its size"""
        cache = DictionaryCache(max_size=1)
        cache["http://example.org/"] = OneUseDocument("http://example.org/")
        self.assertEquals(cache["http://example.org/"].content,
                          "http://example.org/".encode("ascii"))
        self.assert_(cache["http://example.org/"].content is None)
        cache["http://example.com/"] = OneUseDocument("http://example.com/")
        self.assertEquals(cache["http://example.com/"].content,
                          "http://example.com/".encode("ascii"))
        self.assert_(cache["http://example.com/"].content is None)
        self.assert_("http://example.org/" not in cache)

    def test_invalid_max_size(self):
        """Test that cache size must be an integer"""
        DictionaryCache(max_size=None)
        DictionaryCache(max_size=2)
        for size in "large", "42", 3.1415926:
            self.assertRaises(TypeError, DictionaryCache, max_size=size)
        for size in -1, 0:
            self.assertRaises(ValueError, DictionaryCache, max_size=size)


class DirectoryCacheTestCase(unittest.TestCase):

    """Test `DirectoryCache`."""

    def setUp(self):
        """Prepare."""
        self.directory = mkdtemp()

    def tearDown(self):
        """Remove temporary files."""
        rmtree(self.directory)

    def test_cached_document(self):
        """Test that a document can be cached"""
        cache = DirectoryCache(self.directory)
        for url in ("http://example.org/", "http://other.example.com"):
            self.assert_(url not in cache)
            cache[url] = OneUseDocument(url)
            self.assert_(url in cache)
            self.assertEquals(cache[url].content, url.encode("ascii"))
            self.assertEquals(cache[url].content, url.encode("ascii"))
        for url in ("http://example.org/", "http://other.example.com"):
            self.assertEquals(cache[url].content, url.encode("ascii"))
