# -*- coding: utf-8 -*-
# Copyright (C) 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
This module contains custom exceptions used by URLReader.

Some code uses the standard `TypeError`, `ValueError` or
`NotImplementedError`, they are risen in case of bugs in URLReader or
programs using them.
"""


__all__ = ("ReaderError", "UnknownURLError", "HTTPError", "NotFoundError",
           "NonWritableCacheError")


class ReaderError(ValueError):

    """Exception raised if any document-fetching operation fails.

    The argument might an string describing why the exception
    occurred, or an exception raised in some inner code.

    All other exceptions defined in this module are subclasses of
    `ReaderError`.
    """


class UnknownURLError(ReaderError):

    """Exception raised if a backend cannot fetch specified URL."""


class HTTPError(ReaderError):

    """Exception raised in case of unexpected HTTP status codes.

    For 404 the `NotFoundError` exception is raised instead.
    """

    def __init__(self, code, *args):
        """Make an `HTTPError` of status code *code*."""
        super(HTTPError, self).__init__(*args)
        #: The HTTP status code, passed to the constructor.
        self.code = code


class NotFoundError(ReaderError):

    """Exception raised when tried to fetch a non-existent document.

    A separate operation on the same URL should raise this exception
    again (unless the document changes into an existing one), so it is
    cached like documents.

    Not all operations will raise it.
    """


class NonWritableCacheError(ReaderError):

    """Exception raised when assigning documents to read-only caches."""
