#!/usr/local/bin/python

Version = '4.16'      # The integral part is the document structure version
DateVer = 'August 2006'

# wb.py  CGI based billboard management
#
# This CGI script is an utility to manage a web based
# billboard where documents can be posted and edited by authorized users. 
# Documents can have an expiration date and can thus be removed (by running 
# this same program) when expired.
#
# Note: if the program is run with name: wt.py, debug mode is enabled by default

#####################################################################
# Start main customization section. The following variables MUST be 
# modified according to your configuration 

root_http   = '/BillBoards'            # This is relative to server's root
root_dir    = '/opt/HTDOCS/BillBoards' # Actual filepath to billboard dir
domain      = 'arcetri.astro.it'           # e-mail domain
smtphost    = 'hercules.arcetri.astro.it'  # smtp host

# The following variables are suitable defaults, but you may want to
# modify them for some special need

debug       = 0              # Set to 1 for debug output
def_ttl     = 1800           # Session expiring time (half an hour)
dolog       = 1              # Set to 0 for no logging
deflanguage = 'italiano.voc' # The file must be located in 'root_dir'
log_size    = 500000         # Max size of log files
log_number  = 4              # Number of preserved copies of log files 
                             # when rotating
auth_mode   = 'unix'         # Method for user authentication. Use: 'unix'
                             # if user identity is checked in Unix password file
                             # use: 'local' if user's identity is cehcked in
                             # local password file

#####################################################################


accepted_types = ('.pdf','.html','.htm','.txt','.jpg','.gif',  # Accepted attachment filetypes
                  '.png','.ps'                               )

import cgi
import sys,os
import fcntl
import re
import types
import string
import cPickle as pickle
import time,random

ident = 'wb.py - L.Fini (lfini@arcetri.astro.it). Version ' + Version + ', ' + DateVer
logger=None
cgimode=0

curlanguage=''

rem_addr='LOCAL'
user='None'
cur_bboard=''
 
pwname='wb.users'


# The following variables can be used:
#
#   BBOARD: billboard name
#     TITLE: Document title
# INDEX_URL: billboard index URL
#   DOC_URL: document URL

mailnew=(
'************************************************************************',
'A new document has been posted in billboard: %(BBOARD)s',
'',
'%(TITLE)s',
'',
'See: %(HREF)s',
'************************************************************************',
)

mailupd=(
'************************************************************************',
'The following document has been updated in billboard: %(BBOARD)s',
'',
'%(TITLE)s',
'',
'See: %(HREF)s',
'************************************************************************',
)

InitData = { 'allowed_users' : ['root'], 
             'index_header'  : ['<h3>Documents in billboard: <i>%(BBOARD)s</i></h3>',
                                '%(SEARCH)s<p><blockquote><ul>'],
             'index_item'    : ['<p><li> %(DATE)s &nbsp;&nbsp;&nbsp;%(EXPIR)s',
                                '<br><a href=%(HREF)s>%(TITLE)s</a>'],
             'index_item_exp': ['<p><li> %(DATE)s &nbsp;&nbsp;&nbsp;%(EXPIR)s <font color=red> Expired!</font>',
                                '<br><a href=%(HREF)s>%(TITLE)s</a>'],
             'index_footer'  : ['</ul></blockquote>'],
             'index_src'     : '%(N_ITEMS)s found',
             'doc_header'    : ['%(DATE)s %(EXPIR)s',
                                '<h3>%(TITLE)s</h3> %(AUTHOR)s<p>'],
             'doc_footer'    : [],
             'sort_order'    : 'date ascending',
             'author_fmt'    : 'By: %s',
             'noauthor_fmt'  : '',
             'mail_to'       : [],
             'show_future'   : True,         # Show documents with date in the future
             'search_fmt'    : 'Search: %s', # Search field format (see wb.ini)
             'expir_fmt'     : '(Expiration: %s)',
             'noexpir_fmt'   : '(Expiration: none)',
             'mail_new'      : mailnew,       # A long string: see above
             'mail_update'   : mailupd,       # A long string: see above
             'date_mode'     : 0,            # (0: numeric, 1: months as words)
             'language'      : deflanguage,
             'expir_required': False,
             'author_req'    : True,
             'delete_expired': False,        # If false expired files are not deleted but moved
                                             # to the "deleted" directory
}

DELETED  = 'deleted'   # Deleted files directory
COMMASPACE = ', '
TEMPFILE = 'wbdoc.tmp'
LOCKFILE = 'wb.lock'
NBSP15 = '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'
NBSP10 = '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp'

# Fake action for direct run 
fake_action    = 'http://www.arcetri.inaf.it/cgi-bin/PostHandler/~lfini/wb/wb.py'

vocabulary = { 'ERROR': 'Error', 
               'REPORT':'Report error to the sysmgr',
             }

############################################################################################
def plog(st):
  if logger: logger.info('User:%s From:%s%s - %s' % (user,rem_addr,cur_bboard,st))
  if not cgimode: print 'PLOG - %s' % st
  if debug: print '<br>PLOG - %s' % st

############################################################################################
httpmatch=re.compile('http://[^`\'\"><)(\s]+',re.I)
def makelink(mtch):
  return '<a href="'+mtch.group(0)+'">'+mtch.group(0)+'</a>'

def urlcvt(text):
  t=httpmatch.sub(makelink, text)
  return t

############################################################################################
class Session:
  def __init__(self,arc_dir,userid,ttl=def_ttl):
    self.lockf=os.path.join(arc_dir,LOCKFILE)
    self.ftemp=os.path.join(arc_dir,TEMPFILE)
    self.arc_dir=arc_dir
    self.ttl=ttl
    self.suspended=0
    self.token=''
    self.userid=userid

  def sesskey(self):
    return hex(random.randint(0, 0x7FFFFFFF))[2:]

  def __getlock__(self,token=''):
    try:
      self.fobj = open(self.lockf, 'w')
      fcntl.lockf(self.fobj,fcntl.LOCK_EX)           # Create a lock. The lock is released when
                                                     # the program terminates
    except:
      plog('error [%s] locking file: %s'%(sys.exc_type,self.lockf))
      err_exit('[%s] %s: %s'% (sys.exc_type,vocabulary['LOCKERR'],self.lockf))

    try:
      self.tempdoc=docload(self.ftemp)
    except:                          # No temporary file. Access granted
      self.tempdoc=None
      return 1
                                     # Temp file found
    if token:                                          # not first step
      if token==self.tempdoc.st_token:                 # Temp file is mine
        curtime=time.time()
        if self.tempdoc.session_expir>curtime:     # Check expiration
          plog('opened session. Token: %s'%token)
          return 1                          # Access granted
        else:                               # session time expired
          plog('session time expired (doctime: %d, time:%d)'%(int(self.tempdoc.session_expir),int(curtime)))
          self.endsession()
          err_exit(vocabulary['TMEXPIR'],type=0)

      else:                         # Temp file is not mine: this is a system error
          plog('system error: temporary file conflict')
          err_exit('Conflitto file temporaneo')

    else:                           # This is the first step of a session
      if self.tempdoc.session_expir>time.time():     # Check expiration
        scad=time.ctime(self.tempdoc.session_expir)
        if self.tempdoc.st_userid==self.userid:
          self.suspended=1
          msg='%(USER)s ' % vocabulary+self.tempdoc.st_userid+ \
              ' %(SUSPEND)s: '% vocabulary + os.path.basename(self.arc_dir) + \
              ' (%(EXPIRS)s: '% vocabulary + scad + ')'
          notes.add_note(msg,0)
          notes.add_note(vocabulary['RMSUSP'],0)
          plog('user: %s has a suspended session (expiration: %s)' % (self.userid,scad))
        else:
          plog('bboard locked by user: %s (expiration: %s)' % (self.tempdoc.st_userid,scad))
          msg='%(BBLOCKD)s: ' % vocabulary + self.tempdoc.st_userid+ \
              ' (%(EXPIRS)s: ' % vocabulary + scad + ')'
          notes.add_note(msg,1)
        return 0                    # 
      else:                         # Previous session is expired
        plog('previous session file expired')
        return 1

  def newsession(self):
    if self.__getlock__():
      self.token=self.sesskey()
      plog('created new session. Token: %s'%self.token)
      return self.token
    else:
      return ''
    
  def opensession(self,token):
    if self.__getlock__(token):
      self.tempdoc.st_token=token
      self.token=token
    else:
      self.token=''
    return self.token

  def __write__(self,fname,doc):
    try:
      fobj = open(fname, 'w')
      pickle.dump(doc,fobj,2)
      fobj.close()
      plog('document written to file: %s' % fname)
    except:
      plog('error [%] writing temp file: %s'%(sys.exc_type,fname))
      err_exit('['+sys.exc_type+'] %(WRERR)s '+fname)

  def docread(self,st_sid,token):
    fname=os.path.join(self.arc_dir,st_sid)
    try:
      document=docload(fname)
    except:
      plog('error [%s] reading file %s',(sys.exc_type,fname))
      err_exit('['+sys.exc_type+'] %(RDERR)s '+fname)
      document=None
    document.st_token=token
    return document

  def docwrite(self,doc):
    fname=os.path.join(self.arc_dir,doc.st_sid)
    self.__write__(fname,doc)

  def tempwrite(self,doc):
    doc.session_expir  = int(time.time()+self.ttl)
    self.__write__(self.ftemp,doc)

  def endsession(self):
    plog('end session: %s'%self.token)
    safeunlink(self.ftemp,sigerr=0)
  

############################################################################################
def writesess(arc_dir,document,ttl=def_ttl):   # writes a session file onto bboard dir.
                                               # ttl= time to live (in seconds) is the remainig
                                               # time for the session. It must be set to 0 when
                                               # the session is closed.
  if ttl>0:
    document.session_expir  = int(time.time()+ttl)
  else:
    document.session_expir  = 0
  fname=os.path.join(arc_dir,document.st_sid)
  fobj = open(fname, 'w')
  plog('writing session file: %s' % fname)
  pickle.dump(document,fobj,2)
  fobj.close()


#############################################################################################
def safemove(path,todir):       # moves files to given directory
  dir,fname=os.path.split(path)
  topath=os.path.join(todir,fname)
  try:
    os.rename(path,topath)
    plog('file %s moved to: %s'%(path,topath))
  except:
    plog('[%s] cannot move file %s to %s'%(sys.exc_type,path,topath))

#############################################################################################
def safeunlink(path,sigerr=1):      # removes a file trapping errors in order to prevent program crash
  try:
    os.unlink(path)
    plog('removed file: %s'%path)
  except:
    if sigerr: plog('[%s] cannot remove file: %s'%(sys.exc_type,path))
    
#############################################################################################
def removeall(arc_dir,document):         # removes a document and related attachments

  fname=os.path.join(arc_dir,document.st_sid)
  if InitData.get('delete_expired') and not debug:
    document.rem_attach(arc_dir)
    safeunlink(fname)
  else:
    todir = os.path.join(arc_dir,DELETED)
    if not os.path.isdir(todir): 
      plog('making directory for expired files: %s'%todir)
      os.mkdir(todir)
    document.mov_attach(arc_dir,todir)
    safemove(fname,todir)


def text_item(name,size=0,value=''):
  text = '<input type=text name="%s"' % name
  if value: text += ' value="%s"' % value
  if size>0: text += ' size=%d' % size
  text += '>'
  return text

def file_item(name,size=0,value=''):
  text = '<input type=file name="%s"' % name
  if value: text += ' value="%s"' % value
  if size>0: text += ' size=%d' % size
  text += '>'
  return text

def textarea_item(name,rows=20, cols=80, value=''):
  text = '<textarea name="%s" rows=%d cols=%d>\n' % (name,rows,cols)
  if value: text += value
  text += '\n</textarea>'
  return text

def hidden_item(name,value):
  text='<input type=hidden name="%s" value="%s">'%(name,value)
  return text

def expir_str(spec):
  if spec:
    ret = InitData['expir_fmt'] % spec
  else:
    ret = InitData['noexpir_fmt']
  return ret


class Notes:
  def __init__(self):
    self.notes=[]

  def add_note(self,msg,level):
    self.notes.append((level,msg))

  def clean(self):
    self.notes=[]

  def show(self):
    if self.notes:
      print '<hr><b>%(NOTES)s:</b><br>' % vocabulary
      for xt in self.notes:
        if xt[0]: 
          print '<br><font color="red">',xt[1],'</font>'
        else:
          print '<br>',xt[1]
      print '<hr>'

class Attachment:
  def __init__(self,title,orig_name,local_name,id):
    self.title=title
    self.orig_name=orig_name
    self.local_name=local_name
    self.id=id

class Document:
  def __init__(self,sesskey):
    self.attachments=[]
    self.max_id=0
    self.html=0
    self.href=''
    self.date=Date()
    self.expir=Date(0)
    self.expir_required=InitData['expir_required']
    self.version=int(float(Version))

    self.st_sid  = 'DOC-'+hex(int(time.time()*10))[2:]

    self.st_token = sesskey
    self.st_title=''
    self.st_author=''
    self.st_text=''
    self.st_html=''
    self.st_userid=''
                          # Auxiliary variables
    self.new=1            # Set to 0 when the document is permanently stored
    self.session_expir=0  # Set to time of current session expiration

  def v_update(self):     # Updates Document structure to current version
    ver=getattr(self,'version',1)
    self.new=getattr(self,'new',0)
    self.expir_required=getattr(self,'expir_required',0)
    self.session_expir=getattr(self,'session_expir',0)
    self.st_author=getattr(self,'st_author','')
    self.href=''

    if ver < 4:
      if self.expir.time>0:
        self.expir=Date(self.expir.time)
      else:
        self.expir=Date(0)
      self.date=Date(self.date.time)

    self.version=int(float(Version))



  def linkcvt(self):   # Converts links into HTML code
    self.st_text=urlcvt(self.st_text)

  def setuser(self,userid):
    self.st_userid=userid

  def freeze(self):
    self.new=0
    self.session_expir=0  # Set to time of current session expiration
    self.st_token=''
    self.st_userid=''
    if (not self.st_text) and len(self.attachments)==1:
      self.href=self.attachments[0].local_name
    else:
      self.href=""

  def attach_file(self,arc_dir,orig_name,title,fobj):
    (rt,ext)=os.path.splitext(orig_name)
    if ext.lower() not in accepted_types: 
      note=vocabulary['ATCHILL'] + ': ' + orig_name
      notes.add_note(note,1)
      note=vocabulary['ALLWDATCH'], COMMASPACE.join(accepted_types)
      notes.add_note(note,1)
      plog(note)
      return 
    self.max_id += 1
    atch_pref = 'ATC-'+self.st_sid[4:]
    local_name= '%s-%3.3d%s' % (atch_pref,self.max_id,ext.lower())
    atch_file= os.path.join(arc_dir,local_name)

    try:
      fout=open(atch_file,'w')
    except:
      note=vocabulary['ATCHERR'] + ': ' + orig_name
      notes.add_note(note,1)
      plog('error [%s] writing attachment file: %s (orig.name: %s)'%(sys.exc_type,atch_file,orig_name))
      return
      
    while 1:            # read in attachment file
      line = fobj.read(1024)
      if not line: break
      fout.write(line)
    fout.close()
    self.attachments.append(Attachment(title,orig_name,local_name,self.max_id))
    note=vocabulary['ATCHMNT'] + ' %d: "%s" [%s]' % (len(self.attachments),title,orig_name)
    notes.add_note(note,0)
    plog(note)

  def rem_attach(self,arc_dir,at_id=None):
    cn=0
    for at in self.attachments:
      if at_id is None or at.id == at_id:
        atch_file= os.path.join(arc_dir,at.local_name)
        safeunlink(atch_file)
        del self.attachments[cn]
      cn += 1
    return cn+1

  def mov_attach(self,arc_dir,rem_dir):
    cn=0
    for at in self.attachments:
      atch_file= os.path.join(arc_dir,at.local_name)
      safemove(atch_file,rem_dir)
      del self.attachments[cn]
      cn += 1
    return cn+1

############################################################################################
  def preview_out(self,bboard,http_path):
    items={'BBOARD': bboard, 'DATE': self.date, 'TITLE': self.st_title }
    if self.st_author:       # Include author specification only if an author is specified
      items['AUTHOR'] = InitData['author_fmt'] % self.st_author
    else:
      items['AUTHOR'] = InitData['noauthor_fmt']

    items['EXPIR']=expir_str(self.expir)
    items['HREF'] = '%s?%s&%s' % (action,bboard,self.st_sid)

    print '<!--    Begin document preview   -->' 
    print '\n'.join(InitData['doc_header']) % items
    if not self.st_html: 
      p1='<pre>'; p2='</pre>'
    else:
      p1=''; p2=''

    print  p1+self.st_text+p2
                                    # Process attachments
    if self.attachments:
      print '    <!--  Begin attachment list -->'
      print  '<hr><ul>'
      for k in self.attachments:
        print  '<li><a href=%s/%s>%s</a>' % (http_path,k.local_name,k.title)
      print  '</ul>'
      print '    <!--  End attachment list   -->'
    print '\n'.join(InitData['doc_footer']) % items
    print '<!--    End document preview     -->'

###############################################################################################
  def check(self):
    ret=1
                                        # Verify expiration
    if self.expir_required and not self.expir.defined():
      notes.add_note(vocabulary['NOEXPIR'],1)
      ret=0

                                        # Verify title
    self.st_title=self.st_title.strip()
    if self.st_title == '':
      notes.add_note(vocabulary['NOTITLE'],1)
      ret=0
					# Verify body or singlew attachment
    self.st_text=self.st_text.strip()
    if not self.st_text:                # If no body the document must
                                        # consist of a single attachment
      if len(self.attachments) != 1:
        notes.add_note(vocabulary['NOBODY'])
        ret=0
    
    return ret
    
###############################################################################################
#  Form making methods. 
###############################################################################################
  def form_head(self,bboard):                                   # common header for many forms
    action_s=action.replace('http:', 'https:',1)
    print '<form method="post" enctype="multipart/form-data" action="%s">' % action_s
    print hidden_item('st_bboard',bboard)
    print hidden_item('st_sid',self.st_sid)
    print hidden_item('st_userid',self.st_userid)
    print hidden_item('st_token',self.st_token)

###############################################################################################
  def preview_form(self,bboard,http_path):
    ok=self.check()
    notes.show()
    self.preview_out(bboard,http_path)
    if self.new:
      chkd='checked'
    else:
      chkd=''
    print '<hr>'
    self.form_head(bboard)
    print '<table cellpadding=10 border=1>'
    print '<tr><td><input type=submit name="cmd_resume" value="'+vocabulary['MODIFU']+'" style="background:#66cc66"></td>'
    if ok:
      disable=''
    else:
      disable='DISABLED'
    print '<td><input type=submit name="cmd_publish" value="'+vocabulary['PUBLISH']+'" style="background:#aaaaff"',disable,'>'
    if InitData['mail_to']:
      print '&nbsp;&nbsp; <input type=checkbox name=st_notify %s>%s' % (chkd,vocabulary['DONOTIFY']), '</td>'
    print '<td><input type=submit name="cmd_canc_edit" value="'+vocabulary['CANCEL']+'" style="background:red"></td>'
    print '</tr></table></form>'
    

###############################################################################################
  def confirm_form(self,bboard,http_path):   # Form to ask confirmation after delete request
    notes.show()
    op_header('%s <i>%s</i>:%s'%(vocabulary['RMREQST'],bboard,self.st_sid))
    print '<hr>'
    self.preview_out(bboard,http_path)
    print '<hr>'
    self.form_head(bboard)
    print '<center>'
    print '<input type=submit name="cmd_doremove" value="'+vocabulary['RMCNFRM']+'" style="background:red">'
    print NBSP10
    print '<input type=submit name="cmd_canc_rem" value="'+vocabulary['CANCEL']+'" style="background:#66cc66">'
    print '</center>'
    print '</form>'

###############################################################################################
  def edit_form(self,bboard):     # Form to allow editing of a document
    notes.show()
    self.form_head(bboard)
    print '<table border=2 cellpadding=5>'
    try:               # workaround for older (1.x) document files
      isnew=self.new 
    except: 
      self.new=0
    if self.new:
      print '<tr><th bgcolor="#aaaaff">',vocabulary['ADDDOC']+': <i>', bboard,'</i></th></tr>'
    else:
      print '<tr><th bgcolor="#aaaaff">',vocabulary['MODDOC']+' <i>%s</i>:%s</th></tr>'% (bboard,self.st_sid)
    print '<tr><td><b>%(TITLE)s: </b>' % vocabulary
    print text_item('st_title', size=50, value=self.st_title), '&nbsp;&nbsp;&nbsp;',
    print '<b>%(AUTHOR)s: </b>' % vocabulary,
    print text_item('st_author', size=20, value=self.st_author)
    print '</td></tr>'
    print '<tr><td><b>%(DATE)s: &nbsp;</b>'%vocabulary, text_item('st_date',size=10,value=self.date)
    print NBSP10
    if self.expir.defined():
      expir=self.expir.__str__()
    else:
      expir=''
    print '<b>%(EXPIRTN)s:&nbsp;</b>'%vocabulary, text_item('st_expir',size=10,value=expir)
    print NBSP10
    print '<font size=-1>%(HTMLFMT)s</font><input type=checkbox name="st_html"'%vocabulary,
    if self.st_html:
      print 'checked>'
    else:
      print '>'
    print '</td></tr>'
    print '<tr><td><b>%(TEXT)s:</b>'%vocabulary,
    print NBSP15, NBSP15
    print NBSP15, NBSP15
    print NBSP15, NBSP15
    print NBSP15
    print '<input type=submit name="cmd_convert" value="%s"><br>' % vocabulary['MAKELINKS']
    print textarea_item('st_text',value=self.st_text),'</td></tr>'
    if self.attachments:
      nal=0
      print '<tr><td bgcolor="#aaaaaa">%(ATCHMNTS)s:<br>'%vocabulary
      for at in self.attachments:
        nal += 1
        print '%2d. <input type=checkbox name="st_att_%3.3d" checked> %s [<i>%s</i>]<br>' % (nal,at.id,at.title,at.orig_name)
      print '</td></tr>'
    print '<tr><td bgcolor="#dddddd">'
    print '    <table cellpadding=5><tr><td align=center>%s</td>'%vocabulary['ATCHTIT'],
    print '<td align=center>%(SPECATCH)s</td><td></td></tr>'%vocabulary
    next_atch=len(self.attachments)+1
    print '    <tr><td>',text_item('st_attachtitle',size=20,value=vocabulary['ATCHMNT']+' %d'%next_atch),'</td>'
    print '    <td>',file_item('st_filetoattach', size=40),'</td>'
    print '    <td><input type=submit name="cmd_attach" value="'+vocabulary['ATTACH']+'" style="background:#6666cc"></td></tr>'
    print '    </table>'
    print '</td></tr>'
    print '<tr><td align="center">'
    print '<input type=submit name="cmd_preview" value="'+vocabulary['PREVIEW']+'" style="background:#66cc66">'
    print NBSP10
    print '<input type=submit name="cmd_canc_edit" value="    '+vocabulary['CANCEL']+'    " style="background:red">'
    print '</td></tr>'
    print '</table>'
    print '</form>'
################################################### END CLASS: Document

#############################################################################
# Functions to generate some forms
#############################################################################
def auth_head(bboard):          # outputs authorization form header
  action_s=action.replace('http:', 'https:',1)
  print '<form method="post" action="%s">' % action_s
  print hidden_item('st_bboard',bboard)
  print hidden_item('st_checkauth','1')
  print '<blockquote><b>%(USERNAME)s: </b><input type=text name="st_userid">'%vocabulary,
  print NBSP10
  print '<b>%(PASSWD)s: </b><input type=password name="st_passwd"></blockquote>'%vocabulary

######################################################################################
def auth_form0(bboard):             # Authorization form for new document definition
  op_header ('%s: %s' % (vocabulary['NEWDOC1'],bboard))
  auth_head(bboard)
  print hidden_item('cmd_newdoc','1')
  print '<center><input type=submit value="%(SUBMIT)s"></center>'%vocabulary
  print '</form>'
  footer(back=1)

##########################################################################################
def auth_form1(bboard,root_dir):    # Authorization form for document editing or removing
                                     # It also shows a list of editable documents
  op_header ('%s: %s' % (vocabulary['MODIFBB'],bboard))
  auth_head(bboard)
  doc_dir=os.path.join(root_dir,bboard)
  docl=document_list(doc_dir,InitData['sort_order'])
  print '<blockquote>'
  for n in docl:
    print '<input type=radio name="st_sid" value="%s">' % n['st_sid'],
    print '%s - <b>%s</b><br>' % (n['date'],n['st_title'])
  print '</blockquote><center>'
  print '<input type=submit name="cmd_edit" value="%(MODDOC1)s" '%vocabulary,
  print 'style="background:#66cc66">'
  print NBSP10
  print '<input type=submit name="cmd_remove" value="%(RMDOC)s" '%vocabulary,
  print 'style="background:red"> </center></form>'
  footer(back=1)

######################################################################################
def auth_form2(bboard):             # Authorization form for maintenance operations
  op_header ('%s: %s' % (vocabulary['MAINTN1'],bboard))
  auth_head(bboard)
  print hidden_item('cmd_manut','1')
  print '<center><input type=submit value="%(SUBMIT)s"></center>'%vocabulary
  print '</form>'
  footer(back=1)

###################################################################################
def htmlsafe(v):
  s=repr(v)
  return cgi.escape(s)

###################################################################################
def printinit(init):       # Prinit initialization data (debug support)
  if cgimode:
    fmt1= "<pre>"
    fmt2= '<b>%s</b>:'
    safef = htmlsafe
    fmt3 = "</pre>"
  else:
    fmt1= ""
    fmt2= '%s:'
    safef = lambda x: x
    fmt3 = ""
  print fmt1
  ks=init.keys()
  ks.sort()
  for k in ks:
    print fmt2%k,safef(init[k])
  print fmt3

#############################################################################
def err_exit(msg,type=1):          # Output an error message
  if cgimode:
    print '<h3>%(ERROR)s:' % vocabulary
    print '<font color="red"> %s</font></h3>' % msg
    if type: print '<h3>%(REPORT)s</h3>' % vocabulary
    footer()
  else:
    print 'Error: %s' % msg,
    if type: 
      print '[system error]'
    else:
      print
  sys.exit()

#############################################################################################
def set_language(rdir,lang):    # NB: messages in this routine cannot be made in other 
                                #     languages, because vocabulary is not yet available
  global vocabulary,curlanguage

  if lang != curlanguage:
    newvoc={}
    lfile=os.path.join(rdir,lang)

    try:
      execfile(lfile,{},newvoc)
      vocabulary=newvoc
      curlanguage=lang
    except:
      msg = '[%s] reading language file: %s'%(sys.exc_type,lfile)
      err_exit(msg)

################################################################################
def readinitfiles(rdir,bboard,dolog=False):   # Read initialization file (wb.ini)
                                              # merges definitions in InitData and
                                              # in vocabulary
  global vocabulary

  initfile = os.path.join(rdir,bboard,'wb.ini')
  if not os.path.exists(initfile): 
    plog('Warning. No init file: %s' % initfile)
    return
    
  it={}
  if dolog: plog('reading initialization from file: %s' % initfile)
  try:
    execfile(initfile,{},it)
  except:                    # no init file, use defaults
    plog('error [%s] reading initialization file: %s'%(sys.exc_type,initfile))
    err_exit('Error [%s] reading initialization file: %s'%(sys.exc_type,initfile))

  InitData.update(it)
  set_language(rdir,InitData['language'])   # Redefine vocabulary (if needed)

def docload(fname):  # Uses pickle.load() to load a document, checking
                     # proper document format
  fd=open(fname)
  doc=pickle.load(fd)
  fd.close()
  doc.v_update()
  return doc

################################################################################
################################################################################
################################################################################
# Class: Date  - manipulate dates (dd/mm/yy)
class Date:
  sep=re.compile('[^0-9]+')

  def __init__(self,datespec=None,mode=0):
    self.mode=mode
    self.date=[0,0,0,0,0,0]
    self.set(datespec)

  def set(self,datespec):             # set date. Both numerical (a time.time() value)
                                      # and string (dd/mm/yy) specification are allowed.
                                      # if datespec is None, current date is set
    if type(datespec) is str:
      self.__setstr(datespec)
    elif type(datespec) is int:
      self.__setint(datespec)
    else:
      self.__setint(int(time.time()))

  def defined(self):                # Returns True if the date is defined
    return self.date[0]>0

  def expired(self,tm=None):            # Returns: true if the date is prior than specified
                                        # time. If no time is given, uses current time
    if self.date[0]==0: return False    # If date is undefined, return 1
    if not tm:
      tm=time.time()

    ymd = time.localtime(tm)[0:3]
    if ymd[0]<self.date[0]: return  False
    if ymd[0]>self.date[0]: return  True
    if ymd[1]<self.date[1]: return  False
    if ymd[1]>self.date[1]: return  True
    if ymd[2]<self.date[2]: return  False
    if ymd[2]>self.date[2]: return  True
    return False
 
  def cmp(self,comp):          # Retuns -1, 0, 1 if Date is prior,same,later
                               # than argument
    if self.date[0]<comp.date[0]: return -1
    if self.date[0]>comp.date[0]: return  1
    if self.date[1]<comp.date[1]: return -1
    if self.date[1]>comp.date[1]: return  1
    if self.date[2]<comp.date[2]: return -1
    if self.date[2]>comp.date[2]: return  1
    return 0

  def setmode(self,mode):             # Change date display mode 0: numerical month
                                      #                          1: word month
    self.mode=mode

  def __setstr(self,str):
      dd=map(int,Date.sep.split(str)[0:3])
      dd.reverse()
      dd.extend([12,0,0])
      self.date=dd             # [year,month,day,hour,minute,sec]

  def __setint(self,val):
    if val==0: return
    self.date=time.localtime(val)[0:6]

  def __repr__(self): 
   if self.date[0]==0: return ''
   if self.mode==0:
     return '%2.2d/%2.2d/%4.4d' % (self.date[2],self.date[1],self.date[0])
   else:
     return '%2.2d %s %4.4d' % (self.date[2],
                                vocabulary['MONTHS'][self.date[1]-1],
                                self.date[0])

  def __str__(self): return self.__repr__()


#############################################################################################
def update_document(arc_dir,document,form,convert=0):   # Updates document field from form input
  plog('updating document %s'%document.st_sid)
  if 'st_text' in form:
    document.st_text = form.getvalue('st_text').rstrip()
    if convert: document.linkcvt()
  else:
    document.st_text = ''
  if 'st_title' in form:
    document.st_title = form.getvalue('st_title').strip()
  else:
    document.st_title = ''
  if 'st_author' in form:
    document.st_author = form.getvalue('st_author').strip()
  else:
    document.st_author = ''
  sdate=form.getvalue('st_date')
  document.date=Date(sdate)
  sdate=form.getvalue('st_expir')
  if sdate: document.expir=Date(sdate)
  document.st_html=form.getvalue('st_html')
  document.st_userid=form.getvalue('st_userid')
                                            # Manage attach remove
  for at in document.attachments:
    chk='st_att_%3.3d'%at.id
    if not chk in form: 
      natc=document.rem_attach(arc_dir,at.id)
      notes.add_note(vocabulary['ATCHRMVD'] + ' %d: "%s" [%s]'%(natc,at.title,at.orig_name),0)

                                            # Manage attach request
  if 'st_filetoattach' in form:
    atch= form['st_filetoattach']
    atch_title=form.getfirst('st_attachtitle')
    atch_name=atch.filename
    if atch_name:
      atch_read=atch.file
      document.attach_file(arc_dir,atch_name,atch_title,atch_read)
    else:
      if 'cmd_attach' in form:
        notes.add_note(vocabulary['ATCHUNSP'],1)

#############################################################################################
def check_passwd_local(userid,password): # Check userid authorizations in local password file
  import sha
                                         # first authenticate user
  ok=false
  pwfile =  os.path.join(root_dir,pwname)
  try:
    execfile(pwfile,{},users)
  except:                    # No password file: it will be created
    plog('error [%s] reading password file: %s'% (sys.exc_type,pwfile))
  else:
    a=sha.new(password)
    a.update(password)
    ok = (users.getkey(userid) == a.hexdigest())

  return ok


#############################################################################################
def check_passwd_unix(userid,password):  # Check userid authorizations in system password file
  import pwd,crypt
                                         # first authenticate user
  try:
    cryptedpw = pwd.getpwnam(userid)[1]
  except:
    plog('error [%s] in pwd.getpwnam(%s)'% (sys.exc_type,userid))
    ok=0
  else:
    ok = (crypt.crypt(password, cryptedpw) == cryptedpw)

  return ok

#############################################################################################
def check_user(userid,password,mode):    # Check userid authorizations with specified mode
                                         # first authenticate user
  if password is None: password=''

  if mode=='unix':
    ok=check_passwd_unix(userid,password)
  else:
    ok=check_passwd_local(userid,password)
    
  if not ok:
    plog('denied access to user %s (wrong password)'% userid)
    print '<h3>%(INVPW)s:' %vocabulary,'<i>%s</i></h3>' % userid
    footer(back=1)
    sys.exit()
  else:

    users = InitData['allowed_users']       # check if user is allowed
    if users:
      ok = userid in users
    else:
      ok = 1

    if not ok:
      plog('denied access to user %s (not in user list)'% userid)
      print '<h2>%(USER)s <i>' %vocabulary, userid, '</i> %(NOTAUTH)s'%vocabulary,bboard,'</h2>'
  return ok


#############################################################################################
def environ():        # prints environment status (for debug support)
  env=os.environ
  print '<hr>'
  print '<h3>sys.argv:',sys.argv,'</h3>'
  print '<h3>basename:',os.path.basename(sys.argv[0]),'</h3>'
  print "<h3>Environment variables</h3>"
  print '<pre>'
  for k in env.keys():
    print "<b>%s</b>: %s" % (k,env[k])
  print '</pre>'

  print '<h3>Customization parameters:</h3>'
  print '<pre>'
  print '<b>root_http:</b>', root_http
  print '<b>root_dir:</b>', root_dir
  print '<b>domain:</b>', domain
  print '<b>smtphost:</b>', smtphost
  print '<b>debug:</b>', debug
  print '<b>def_ttl:</b>', def_ttl
  print '<b>dolog:</b>', dolog
  print '<b>deflanguage:</b>', deflanguage
  print '<b>log_size:</b>', log_size
  print '<b>log_number:</b>', log_number
  print '</pre>'

def show_input(form,args):      # Printout arguments and input from form
  if form:
    print "<h3>POST input arguments</h3>"
    print "<pre>"
    for k in form.keys():
      val=form[k]
      if val.filename:
        print '<b>%s [filename: %s]</b>' % (k,val.filename)
      else:
        val=form.getlist(k)
        print '<b>%s</b>: %s' % (k,cgi.escape(",".join(val)))
  print "</pre>"
  print '<h3>Relevant input</h3>'
  print "<pre>"
  print '<b>action:</b>',action
  print '<b>URL args:</b>', args
  print '</pre>'

#############################################################################################
def clean_bboard(arc_dir,userid):     # Clean given bboard: 
                                      #   - removes expired documents,
                                      #   - removes temporary files
                                      #   - removes dangling attachments
  if cgimode:
    print '&nbsp;&nbsp;&nbsp;&nbsp;%s: %s<br>' % (vocabulary['CLEANG'],arc_dir)
  else:
    print '   %s: %s' % (vocabulary['CLEANG'],arc_dir)
  s=Session(arc_dir,userid)
  if not s.newsession():              # Lock the bboard while doing cleaning 
    if not s.suspended:
      return 0            

  notes.clean()                       # Remove previous messages
  notes.add_note(vocabulary['MAINTN2'],0)

  if s.suspended:
    notes.add_note('&nbsp;&nbsp;&nbsp;&nbsp; %s' % vocabulary['RMSUSP1'],0)
    safeunlink(s.ftemp)

  plog('removing expired documents from bboard: %s'%arc_dir)
  count=0
  files=os.listdir(arc_dir)
  for f in files:
    if f[0:4] == 'DOC-':
      fname= os.path.join(arc_dir,f)
      doc=docload(fname)
      if doc.expir.expired():
        count += 1
        removeall(arc_dir,doc)
  notes.add_note('&nbsp;&nbsp;&nbsp;&nbsp;  %s: %d' % (vocabulary['EXPDDOC'],count),0)

  plog('removing dangling attachments in dir: %s'%arc_dir)
  count=0
  for f in files:
    if f[0:4] == 'ATC-':
      docn=os.path.splitext('DOC-'+ f[4:])[0][0:14]  # docn is the file name of document 
                                                     # containig the attachemnt,
                                                     #  eg: DOC-29D581BBEL
      if docn not in files: 
        count +=1
        safeunlink(os.path.join(arc_dir,f))

  notes.add_note('&nbsp;&nbsp;&nbsp;&nbsp; %s: %d' % (vocabulary['DANGLATCH'],count),0)

  s.endsession()              # unlock the bboard
  return 1

#############################################################################################
def show_languages(rdir):
  import copy

  if cgimode:
    print '<h3>%s: <i>%s</i></h3>' % (vocabulary.get('DEFLANG'),deflanguage[:-4])
  else:
    print '%s: %s\n' % (vocabulary.get('DEFLANG'),deflanguage[:-4])
  saveverb=vocabulary.get('AVLANGS')

  files=os.listdir(rdir)
  files=filter(lambda x: x[-4:]=='.voc',files)
  allvoc={}
  for f in files:
    set_language(rdir,f)
    allvoc[f[:-4]]=copy.copy(vocabulary)

  vocs=allvoc.keys()

  if cgimode:
    print '<h3>%s: '%saveverb, COMMASPACE.join(vocs),'</h3>'
  else:
    print '%s: '%saveverb, COMMASPACE.join(vocs)

  allkeys = {}
  for k in vocs: 
    allkeys.update({}.fromkeys(allvoc[k].keys(),1))
  allkeys = allkeys.keys()
  allkeys.sort()
  if cgimode:
    keyfmt= '<b>%s:</b><br>'
    noword= '<font color="red">????</font>'
    wordfmt = '&nbsp;&nbsp;&nbsp;&nbsp; %s: <i>%s</i><br>'
  else:
    keyfmt= ' - %s:'
    noword= '????'
    wordfmt = '       %s: %s'
  for j in allkeys:
    print keyfmt % j
    for k in vocs:
      try:
        term=allvoc[k][j]
      except:
        term = noword
      print wordfmt % (k,term)

fmt_cgi = [ '<h3> %s: %s (%s: %s)</h3>',
            '<ul>',
            '<p><li> %s:' ,
            '<li> %s (%s: %s) - %s' ,
            '</ul>',
            '</ul><hr><h4>%s:&nbsp;&nbsp;',
            '</h4><hr><h4> %s</h4>'        ]
fmt_run = [ '%s: %s (%s: %s)',
            '',
            ' - %s:' ,
            '    o  %s (%s: %s) - %s' ,
            '',
            '',
            '=====================================================================\n%s' ]

#############################################################################################
def show_info(arc_dir):        # Shows some information related to given bboard
  arc_name=os.path.basename(arc_dir)

  if cgimode:
    fmt=fmt_cgi
    op_header('%s: <i>%s</i>' % (vocabulary['BBINFO'],arc_name))
  else:
    fmt=fmt_run
    print '%s: %s' % (vocabulary['BBINFO'],arc_name)

  s=Session(arc_dir,'')
  if not s.__getlock__():
    print fmt[0] % (vocabulary['BBLOCKD'],s.tempdoc.st_userid,
                  vocabulary['EXPIRS'],time.ctime(s.tempdoc.session_expir))

  dl=document_list(arc_dir)
  
  print fmt[1] 
  print fmt[2] % vocabulary['LANGF'], InitData['language']
  print fmt[2] % vocabulary['NDOCS'], len(dl)
  print fmt[2] % vocabulary['ALLWDUSR'], COMMASPACE.join(InitData['allowed_users'])
  print fmt[2] % vocabulary['NOTIFTO'], COMMASPACE.join(InitData['mail_to'])
  print fmt[2] % vocabulary['ALLWDATCH'], COMMASPACE.join(accepted_types)
  
  exp=filter(lambda x: x['expir'].expired(),dl)
  print fmt[2] % vocabulary['EXPDDOC'],
  if exp:
    print fmt[1]
    for d in exp:
      print fmt[3] % (d['date'],vocabulary['EXPIRD'],d['expir'],d['st_title'])
    print fmt[4]
  else:
    print '0'
  if cgimode:
    print fmt[5] % vocabulary['OPERS'], listfmt(action,arc_name,'&nbsp;-&nbsp&nbsp;'),
  else:
    print fmt[5]
  print fmt[6] %vocabulary['INITDATA']
  printinit(InitData)
  if cgimode: footer(back=1)

#############################################################################################
def show_document(bboard,document,root_dir,http_path):   # Shows the given document
  fpath=os.path.join(root_dir,bboard,document)
  doc=docload(fpath)
  doc.preview_out(bboard,http_path)
  print '</body>\n</html>'

#############################################################################################
def search_form(act,bb, sfmt):
  if not sfmt: 
    return ''
  else:
    return '<form method="post" action="%s">\n' % act + \
           '<input type=hidden name=st_bboard value="%s">\n' %bb + \
           sfmt % '<input type=text name="cmd_search">' + '</form>'

#############################################################################################
def show_index(arc_dir,what=''):        # Shows an index of documents in given bboard
                                        # a document list may be provided by caller
  arc_name=os.path.basename(arc_dir)

  docl=document_list(arc_dir,InitData['sort_order'],what)
  items={'BBOARD': arc_name, 
         'N_ITEMS': str(len(docl)),
         'SEARCH': search_form(action,arc_name,InitData['search_fmt']),
         'STRING': what }
  print  '\n'.join(InitData['index_header']) % items
  if what: print InitData['index_src'] % items
  print '<!-- Begin index list  -->'
  for doc in docl:
    if doc['st_author']:       # Include author specification only if an author is specified
      items['AUTHOR'] = InitData['author_fmt'] % doc['st_author']
    else:
      items['AUTHOR'] = InitData['noauthor_fmt']
    if not doc['href']:
      items['HREF']='"%s?%s&%s"' % (action,arc_name,doc['st_sid'])
    else:
      items['HREF']='"%s/%s/%s"' % (root_http,arc_name,doc['href'])
    items['DATE']=doc['date']
    items['TITLE']=doc['st_title']
    items['EXPIR'] = expir_str(doc['expir'])
    if doc['expir'].expired():
      index_fmt = '\n'.join(InitData['index_item_exp'])
    else:
      index_fmt =  '\n'.join(InitData['index_item'])
    if index_fmt: print  index_fmt % items

  print '<!-- End index list    -->'
  print '\n'.join(InitData['index_footer']) % items
  print '</body>\n</html>'

#############################################################################################
def selectsort(order):   # Selects a sort method for a list of documents
  if order.find('expir')>=0: 
    field='expir'
  else:
    field='date'

  if order.find('desc')>=0:
    return lambda a,b: b[field].cmp(a[field])
  else:
    return lambda a,b: a[field].cmp(b[field])

#############################################################################################
def document_list(arc_dir,order='',ss=None):   # Generates a list of document in given bboard
  docs=filter(lambda x: x[:4]=='DOC-', os.listdir(arc_dir))
  docl=[]
  if ss: mtc=re.compile(ss,re.I)
  for n in docs:
    fpath=os.path.join(arc_dir,n)
    doc=docload(fpath)
    if ss:
      if not mtc.search(doc.st_text): continue
    docl.append({'date'        : doc.date,
                 'expir'       : doc.expir,
                 'href'        : doc.href,
                 'st_title'    : doc.st_title,
                 'st_author'   : doc.st_author,
                 'st_sid'      : doc.st_sid})

  if order:
    sortfunc=selectsort(order)
    docl.sort( sortfunc )
  return docl

#############################################################################################
def set_log(arc_dir):       # Set up logging support 
                            # The plog routine is used to write log records
  import logging
  import logging.handlers
  global logger

  logger = logging.getLogger('wb')
  try:
    logfile=os.path.join(arc_dir,'wb.log')
    hdlr = logging.handlers.RotatingFileHandler(logfile,'a',log_size,log_number)
    formatter = logging.Formatter('%(asctime)s %(message)s')
    hdlr.setFormatter(formatter)
    logger.addHandler(hdlr) 
    logger.setLevel(logging.INFO)
  except:
    plog('Error [%s] opening log file: %s'%(logfile,sys.exc_type))
    err_exit('[%s]'%sys.exc_type+' %(LOGFERR)s: '%vocabulary + logfile)

#############################################################################################
def op_header(txt):       # Returns a properly formatted header for all operation pages
  print '<table width="100%" cellpadding=5><tr>' 
  print '<td bgcolor=#5555ff width=45><a href=%s/about.html>' % root_http
  print '<img src=%s/wb.png alt="%s"></a></td>' % (root_http,vocabulary['ALT1']) 
  print '<td bgcolor=#ccccff><h3>',NBSP10,'%s</h3></td>' % txt 
  print '<td bgcolor=#5555ff width=45><a href=%s/help.html>' % root_http
  print '<img src=%s/qm.png alt="%s"></a></td>' % (root_http,vocabulary['ALT2']) 
  print '</tr></table>'

#############################################################################################
def listfmt(action,arc,sep):                     # returns a properly formatted line for a 
                                                 # list of available billboards
  ret  = '<a href="%s?%s">%s</a>%s' % (action,arc,arc,sep)
  ret += '&nbsp;&nbsp;[<a href="%s?%s&edit">' % (action,arc) + vocabulary['MODIF']+'</a>]'
  ret += '&nbsp;&nbsp;[<a href="%s?%s&new">' % (action,arc) + vocabulary['NEWDOC']+'</a>]'
  ret += '&nbsp;&nbsp;[<a href="%s?%s&info">' % (action,arc) + vocabulary['INFO']+'</a>]'
  ret += '&nbsp;&nbsp;[<a href="%s?%s&clean">' % (action,arc) + vocabulary['MAINTN']+'</a>]'
  return ret

#############################################################################################
def dirlist(root_dir):                 # Returns a list of directories
  try:
    files=os.listdir(root_dir)
  except:
    plog('error: cannot read directory: %s'% root_dir)
    err_exit('%(READDIR)s: '%vocabulary + root_dir)

  list=[]
  for f in files: 
    if f[0] == '.': continue          # Ignore hidden directories
    fn=os.path.join(root_dir,f)
    if os.path.isdir(fn): list.append(f)

  list.sort()
  return list

#############################################################################################
def bboard_list(root_dir):             # Print list of billboards, properly formatted in HTML
  global vocabulary

  op_header(vocabulary['BBLIST'])
  
  list=dirlist(root_dir)
  if list:
    print '<blockquote><table cellpadding=5>'
    for f in list: 
      print '<tr><td>', listfmt(action,f,'</td><td>:</td><td>'), '</tr>'
    print '</table></blockquote>'

    print '<h4>&bull; <a href=%s?.log>%s</a><h4>' % (action,vocabulary['VIEWLOG'])

  else:
    print '<h4>%(NOBBLIST)s</h4>'%vocabulary
  footer()


#############################################################################################
def footer(back=0):           # Prints HTML document footer (for management functions)

  if back: 
    print '<h4><a href="%s">'% action
    print '%(BACKTOLIST)s</a></h4>'% vocabulary
  print '<hr><font size=1><center>Powered by: %s</center></font>' % ident
  print '</body>\n</html>'

#############################################################################################
def notify(bboard,doc,userid,new):      # Send mail to the list of addresses to be notified
  import smtplib
  from email.MIMEText import MIMEText

  items = {'TITLE'    : doc.st_title,
           'AUTHOR'   : doc.st_author,
           'BBOARD'   : bboard,
           'DATE'     : doc.date,
           'HREF'     : full_action+'?'+bboard+'&'+doc.st_sid }
  items['EXPIR']=expir_str(doc.expir)
  sender= userid + '@' + domain
  toaddrs=InitData['mail_to']

  if new:
    msg=MIMEText('\n'.join(InitData['mail_new']) % items)
    msg['Subject']='%(SADDED)s: ' % vocabulary + bboard
  else:
    msg=MIMEText('\n'.join(InitData['mail_update']) % items)
    msg['Subject']='%(SMODIF)s: ' %vocabulary + bboard

  msg['From']=sender
  msg['To']=COMMASPACE.join(toaddrs)
  msg.epilogue=''
  try:
    s=smtplib.SMTP(smtphost)
    s.sendmail(sender, toaddrs, msg.as_string())
    s.quit()
    plog('sent notification to %s'% msg['To'])
  except:
    plog('email notification error: [%s]'%sys.exc_type)
    plog('     sender: %s' % sender)
    plog('         to: %s' % msg['To'])
    plog('   smtphost: %s' % smtphost)
    err_exit('['+sys.exc_type+ ']%(NOTIFERR)s:' % vocabulary+ COMMASPACE.join(toaddrs))

#############################################################################################
def begin_get(rdir,bboard,cmd=''):                    # Called at beginning of a GET session
  welcome = '##### wb version %s -  begin CGI [GET] step' % Version

  dolog=False
  if cmd: 
    plog(welcome + ': ' + cmd)
    dolog=True
  readinitfiles(rdir,bboard,dolog)
  if(debug): 
    print "<h3>Initialization data:</h3>"
    printinit(InitData)

#############################################################################################
def view_logfile(root_dir,http_root):                # Show log file content
  log_match=re.compile('wb.log')
  dirlist=os.listdir(root_dir)
  files=filter(log_match.match,os.listdir(root_dir))
  files.sort()
  if files:
    print '<h3>', vocabulary['SELLOG'], '</h3><ul>'
    for k in files:
      print '<li>', '<a href=%s/%s>%s</a>' %(root_http,k,k)
    print '</ul>'
  else:
    print '<h3>', vocabulary['NOLOG'], '</h3><ul>'


#############################################################################################
def clean_all(root_dir):                   # Clean all billboards
  list=dirlist(root_dir)
  for l in list:
    arc_dir=os.path.join(root_dir,l)
    clean_bboard(arc_dir)

#############################################################################################
def run_cgi():           # This part runs when invoked by the Apache http server
  global action,full_action,cgimode,dolog,rem_addr,user,vocabulary,cur_bboard

  cgimode=1        

  print "Content-Type: text/html\n"     # HTML is following
  print '<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN">'
  print '<html>\n<head>'
  print '<meta name="generator" content="%s">' % ident
  print '<link rel="icon" href="%s/wb.ico" type="image/x-icon">' % root_http
  print '<link rel="shortcut icon" href="wb.ico" type="image/x-icon">'
  print '</head>\n<body>'

  if debug: 
    import cgitb; cgitb.enable()        # enable debug HTML output
    environ()                           # Printout environment

  rem_addr=os.environ['REMOTE_ADDR']    # Extract "action" from input data
  user='None'
  form=cgi.FieldStorage()
  request = os.environ['REQUEST_URI'].split('?')
  action = request[0]
  full_action = 'http://'+os.environ['SERVER_NAME']+action

  set_language(root_dir,deflanguage)    # set default vocabulary

  if os.environ['REQUEST_METHOD'] == 'GET':
                                             # Request method is GET: session begins here
                                    # Extract input arguments
    if len(request) > 1:
      args=request[1].split('&')
      args.extend(('',''))
      args.append('')
      bboard=args[0]
      cmd=args[1]
      doc=args[2]
    else: 
      bboard_list(root_dir)                  # If no arguments, show a list of available
                                             # bboard and operations
      sys.exit()                             # and return

    if debug: show_input(None,args)

    if bboard == '.voc':              # reserved billboard name: this is a request 
                                      # for available languages
      show_languages(root_dir)
      footer(back=1)
      sys.exit()

    if bboard == '.log':              # reserverd billboard name: this is a request
                                      # to see the log file
      view_logfile(root_dir,root_http)
      footer(back=1)
      sys.exit()

    cur_bboard= ' ['+bboard+']'
    arc_dir=os.path.join(root_dir,bboard)   # get bboard directory path
    if not os.path.exists(arc_dir): 
      plog('no billboard: %s'% arc_dir)
      err_exit('%(BBNOTEXS)s: '% vocabulary + arc_dir)

    http_path=root_http + '/' + bboard

    if dolog: set_log(root_dir)       # start logging

                                  # Check input command
    if cmd == 'new':              # New document request
      begin_get(root_dir,bboard,'NEW')
      auth_form0(bboard)
    elif cmd =='edit' :           # Modify document request
      begin_get(root_dir,bboard, 'EDIT')
      auth_form1(bboard,root_dir)
    elif cmd[0:4] == 'DOC-':      # Show document
      begin_get(root_dir,bboard)
      show_document(bboard,cmd,root_dir,http_path)
    elif cmd == 'info':           # Show bboard info
      begin_get(root_dir,bboard)
      show_info(arc_dir)
    elif cmd == 'clean':          # Perform bboard maintenance
      begin_get(root_dir,bboard, 'CLEAN')
      auth_form2(bboard)
    else:
      begin_get(root_dir,bboard)
      show_index(arc_dir)

    sys.exit()                    # terminate session

  ########################################################################################
                                  # We get here only for request method == POST 
  if debug: show_input(form,[])
  
  bboard=form.getfirst('st_bboard')
  if bboard:
    cur_bboard= ' ['+bboard+']'
  arc_dir=os.path.join(root_dir,bboard)
  http_path=root_http + '/' + bboard
  readinitfiles(root_dir,bboard) 

  if 'cmd_search' in form:        # This is a search request
    show_index(arc_dir,form.getfirst('cmd_search'))
    sys.exit()

  userid=form.getfirst('st_userid')
  user=userid
  if dolog: set_log(root_dir)
  plog('##### wb version %s -  begin CGI [POST] step' % Version)

  st_token=form.getfirst('st_token')      # check authorization token
  if st_token:                            # if st_token is defined authentication was
                                          # performed in a previous cycle of the same session
    plog('token from previous step: %s'%st_token)
  else:
    passwd=form.getfirst('st_passwd')     # authenticate user before going on
    if not check_user(userid,passwd,auth_mode):   
      footer(back=1)
      sys.exit()

  plog('granted access to user %s'% userid)
  arc_dir=os.path.join(root_dir,bboard)
  if(debug): 
    print "<h3>Initialization data:</h3>"
    printinit(InitData)
  st_sid=form.getvalue('st_sid')
  s=Session(arc_dir,userid)

#---------------------------------------
  if 'cmd_newdoc' in form:                            # New document request
    plog('new document request')
    st_token=s.newsession()
    if st_token:
      document=Document(st_token)
      document.setuser(userid)
      document.edit_form(bboard)
      s.tempwrite(document)                          # Store new (empty) document
      footer()
    else:
      notes.show()
      footer(back=1) 
#---------------------------------------
  elif 'cmd_edit' in form:                 # edit request
    plog('document edit request (doc: %s)'%st_sid)
    if not st_sid:
      plog('no document selected')
      err_exit(vocabulary['NOSEL'],0)
      
    st_token=s.newsession()
    if st_token:
      document=s.docread(st_sid,st_token)
      if document:
        document.setuser(userid)
        document.edit_form(bboard)           # show the edit form
        s.tempwrite(document)                 # store modified document (temporary)
        footer()
      else:
        footer(back=1)
    else:
      notes.show()
      footer(back=1)
#---------------------------------------
  elif 'cmd_canc_edit' in form:                # Edit operation cancelled
    plog("document update cancelled")
    if s.opensession(st_token):
      print '<h2>%(OPCANC)s</h2>' % vocabulary
      document=s.tempdoc
      document.rem_attach(arc_dir)   # Remove all attachments
      s.endsession()
    else:
      notes.show()
    footer(back=1)
#---------------------------------------
  elif 'cmd_resume' in form:                   # Resume edit request
    plog('resume edit request for document: %s'%st_sid)
    if s.opensession(st_token):
      document=s.tempdoc
      document.edit_form(bboard)               # show the edit form
      s.tempwrite(document)                    # store modified document (temporary)
      footer()
    else:
      notes.show()
      footer(back=1)
#---------------------------------------
  elif 'cmd_preview' in form:                    # preview request
    plog('preview request for document: %s'%st_sid)
    if s.opensession(st_token):
      document=s.tempdoc
      update_document(arc_dir,document,form)      # Merge form data into document
      document.preview_form(bboard,http_path)
      s.tempwrite(document)                     # store modified document (temporary)
      footer()
    else:
      notes.show()
      footer(back=1)
#---------------------------------------
  elif 'cmd_publish' in form:       # publish request
    plog('publish request for document: %s'%st_sid)
    if s.opensession(st_token):
      document=s.tempdoc
      new=document.new
      document.freeze()
      s.docwrite(document)                  # store modified document (final)
      txt = '%s: "%s" &nbsp; <font size=-1>(ID: <i>%s</i>)</font></h3>'% (vocabulary['DOCPUBL'],document.st_title, document.st_sid)
      op_header(txt)
      if InitData['mail_to'] and form.has_key('st_notify'): 
        notify(bboard,document,userid,new)
      s.endsession()
    else:
      notes.show()

    footer(back=1)
#---------------------------------------
  elif 'cmd_convert' in form:               # Link convert request
    plog('link convert request (doc: %s)'%st_sid)
    if s.opensession(st_token):
      document=s.tempdoc
      update_document(arc_dir,document,form,convert=1)      
      document.edit_form(bboard)           # merge form data into document
      s.tempwrite(document)                 # store modified document (temporary)
      footer()
    else:
      notes.show()
      footer(back=1)
#---------------------------------------
  elif 'cmd_attach' in form:               # Attach request (handled by update_document)
    plog('attach request (doc: %s)'%st_sid)
    if s.opensession(st_token):
      document=s.tempdoc
      update_document(arc_dir,document,form)      
      document.edit_form(bboard)           # merge form data into document
      s.tempwrite(document)                 # store modified document (temporary)
      footer()
    else:
      notes.show()
      footer(back=1)
#---------------------------------------
  elif 'cmd_remove' in form:
    plog('remove request for document: %s'%st_sid)
    if not st_sid:
      plog('no document selected')
      err_exit(vocabulary['NOSEL'],0)
      
    st_token=s.newsession()
    if st_token:
      document=s.docread(st_sid,st_token)
      if document:
        document.setuser(userid)
        document.confirm_form(bboard,http_path)
        s.tempwrite(document)                     # store modified document (temporary)
        footer()
      else:
        footer(back=1)
    else:
      notes.show()
      footer(back=1)

#---------------------------------------
  elif 'cmd_doremove' in form:
    plog('remove confirmation for document: %s'%st_sid)
    if s.opensession(st_token):
      document=s.tempdoc
      removeall(arc_dir,document)
      print '<h3>%(DOCRMVD)s: ' %vocabulary,
      print '<b>%s</b> ' % document.st_sid,
      print '%(ANDATCH)s</h3>' % vocabulary
      s.endsession()
    else:
      notes.show()
    footer(back=1)
#---------------------------------------
  elif 'cmd_canc_rem' in form:         # remove operation cancelled
    plog("remove request cancelled")
    if s.opensession(st_token):
      print "<h3>%(OPCANC)s</h3>"%vocabulary
      s.endsession()
    else:
      notes.show()
    footer(back=1)
#---------------------------------------
  elif 'cmd_manut' in form:         # Maintenance operation
    clean_bboard(arc_dir,userid)
    notes.show()
    footer(back=1)
#---------------------------------------
  else:                                  # No command in form input
    plog('no valid command in form')
    err_exit(vocabulary['NOCMD'])

#############################################################################################
def show_file(bboard,document):
  fname=os.path.join(bboard,document)
  a=docload(fname)

  for k in dir(a):
    attr=getattr(a,k)
    if not callable(attr):
      print "document.%s: "%k, attr

#############################################################################################
def add_user(userid,passwd):
  import sha

  pwfile =  os.path.join(root_dir,pwname)
  users={}
  if dolog: plog('reading local password file: %s' % pwfile)
  try:
    execfile(pwfile,{},users)
  except:                    # No password file: it will be created
    pass

  if users.has_key(userid):
    err_exit(vocabulary['DUPID'] + " " + userid,0)
  a=sha.new(passwd)
  a.update(passwd)
  users[userid]=a.hexdigest()

  try:
    fd=open(pwfile,'w')
  except:
    err_exit(vocabulary['PWFILERR'] + pwfile)

  keys = users.keys()
  keys.sort()
  print >>fd,'# password file generated on',time.asctime()
  for k in keys:
    print >>fd,'%s = "%s"' % (k,users[k])
  print >>fd,'# end'
    
  fd.close()
  print 'OK'
  
#############################################################################################
def run_standalone():

  global cgimode,action,rem_addr,user,root_dir

  cgimode=0        

  rem_addr='LOCAL'
  user=os.environ['USER']
  action = fake_action

  if len(sys.argv) < 2: helpexit()

### Stub to allow local testing. This allows to run wb with a "wrong" root_dir, provided
### it is run from the root directory.
  if not os.path.isdir(root_dir): root_dir = '.'
############################################################################################
  if sys.argv[1] == '-s':
    if len(sys.argv) < 4: helpexit()
    show_file(sys.argv[2],sys.argv[3])

  elif sys.argv[1] == '-v':
    if len(sys.argv) > 2: 
      bill_dir = sys.argv[2]
    else:
      bill_dir = '.'
    set_language(bill_dir,deflanguage)
    show_languages(bill_dir)

  elif sys.argv[1] == '-i':
    if len(sys.argv) < 3: helpexit()
    rdir,bboard = os.path.split(sys.argv[2])
    if rdir=='': rdir='.'
    readinitfiles(rdir,bboard) 
    show_info(sys.argv[2])

  elif sys.argv[1] == '-m':
#   set_language(root_dir,deflanguage)
#   if dolog: set_log(root_dir)
#   if len(sys.argv) >= 3: 
#     bboard = sys.argv[2]
#     clean_dir = os.path.join(root_dir,bboard)
#     clean_bboard(clean_dir)
#   else:
#     clean_all(root_dir)
    print 'Function -m has been removed'

  elif sys.argv[1] == '-u':
    set_language(root_dir,deflanguage)
    if len(sys.argv) < 4: helpexit()
    add_user(sys.argv[2],sys.argv[3])
  else:
    helpexit()

def helpexit():
  print '\n',ident,'\n'
  print 'Usage: wb.py -i directory     show billboard info'
  print '       wb.py -s bboard doc    show document file structure'
  print '       wb.py -u user passwd   add user/password to local password file'
  print '       wb.py -v directory     show available languages'
  sys.exit()

#############################################################################################
def main():
  global cgimode,notes,debug,language_file

  if os.path.basename(sys.argv[0]) == 'wt.py': debug=1

  notes = Notes()
  if 'SERVER_SOFTWARE' in os.environ:
    cgimode=1        
    run_cgi()
  else:
    cgimode=0        
    run_standalone()


if __name__ == '__main__': main()
