import os
import os.path
import time
import subprocess
from setuptools import setup, find_packages


def read(fname):
    with open(os.path.join(os.path.dirname(__file__), fname)) as fp:
        return fp.read()


def runcmd(cmd):
    p = subprocess.Popen(cmd, stdout=subprocess.PIPE,
                         stderr=subprocess.PIPE)
    out, err = p.communicate()
    return out, err


# Figure out the version.
# (this is loosely based on what Mercurial does)
version = None
try:
    if os.path.isdir(os.path.join(os.path.dirname(__file__), '.hg')):
        cmd = ['hg', 'log', '-r', '.', '--template', '{tags}\n']
        tags, err = runcmd(cmd)
        versions = [t for t in tags.split() if t[0].isdigit()]

        cmd = ['hg', 'id', '-i']
        hgid, err = runcmd(cmd)
        hgid = hgid.strip()

        if versions:
            # Use tag found at the current revision. Add dirty flag if any.
            version = versions[-1]
            if hgid.endswith('+'):
                version += '+'
        else:
            # Use latest tag.
            cmd = ['hg', 'parents', '--template', '{latesttag}+{latesttagdistance}-']
            version, err = runcmd(cmd)
            version += hgid
        if version.endswith('+'):
            version += time.strftime('%Y%m%d')
except OSError:
    # Mercurial isn't installed, or not in the PATH.
    version = None

if version:
    f = open("wikked/__version__.py", "w")
    f.write('# this file is autogenerated by setup.py\n')
    f.write('version = "%s"\n' % version)
    f.close()


try:
    from wikked import __version__
    version = __version__.version
except ImportError:
    version = 'unknown'


setup(
        name='Wikked',
        version=version,
        description=("A wiki engine entirely managed with text files "
            "stored in a revision control system."),
        author='Ludovic Chabant',
        author_email='ludovic@chabant.com',
        url="https://github.com/ludovicchabant/Wikked",
        license="Apache 2.0",
        keywords="wiki mercurial hg git",
        packages=find_packages(exclude=["tests"]),
        install_requires=[
            'Flask==0.10.1',
            'Flask-Bcrypt==0.5.2',
            'Flask-Login==0.2.10',
            'Flask-Script==0.5.1',
            'Jinja2==2.7.2',
            'Markdown==2.2.1',
            'Pygments==1.6',
            'SQLAlchemy==0.9.3',
            'Whoosh==2.5.5',
            'colorama==0.2.7',
            'py-bcrypt==0.4',
            'pysqlite==2.6.3',
            'pytest==2.5.2',
            'repoze.lru==0.6',
            'python-hglib'],
        include_package_data=True,
        package_data={
            'wikked': [
                'resources/defaults.cfg',
                'resources/hg_log.style',
                'resources/init/*',
                'templates/*.html',
                'static/bootstrap/fonts/*',
                'static/css/wikked.min.css',
                'static/img/*.png',
                'static/js/require.js',
                'static/js/wikked.min.js'
                ]
            },
        zip_safe=False,
        classifiers=[
                'Development Status :: 3 - Alpha',
                'License :: OSI Approved :: Apache Software License',
                'Environment :: Console',
                'Operating System :: MacOS :: MacOS X',
                'Operating System :: Unix',
                'Operating System :: POSIX',
                'Operating System :: Microsoft :: Windows',
                'Programming Language :: Python',
            ],
        entry_points={
                'console_scripts': [
                    'wk = wikked.witch:real_main'
                ]
            },
        )

