#!/usr/env python
"""XolNowPlaying

Usage:
    np.py -u <user>
    np.py -u <user> -r <refresh_rate>
    np.py (-h | --help)
    np.py --version

Options:
    -h --help           Show this help.
    --version           Show script version.
    -u <user>           User to look up.
    -r <refresh_rate>   Refresh every <refresh_rate> seconds.

"""
from docopt import docopt
from time import sleep
from xolnowplaying.scrobbler import Scrobbler
import os

class XolNowPlaying:

    def __init__(self, user):
        home = os.getenv('USERPROFILE') or os.getenv('HOME')
        self.filename = home + '/np.txt'
        self.refresh(user)

    def refresh(self, user):
        self.scrobbler = Scrobbler(user)

    def run(self):
        string = ''
        if self.is_playing:
            string = '%s - %s' % (self.artist, self.title)
            if self.write_to_file(self.filename, string):
                Scrobbler.log('info ', string)
        else:
            string = ''
            self.write_to_file(self.filename, string)

    def write_to_file(self, filename, string):
        with open(filename, 'w+') as f:
            read = f.readline()
        f.close()
        if not read == string:
            with open(filename, 'w') as f:
                try:
                    f.write(string)
                except UnicodeEncodeError:
                    pass
            f.close()
            return True
        else:
            return False

    def main(self, arguments):
        if not arguments['-r']:
            self.run()
        else:
            Scrobbler.log('debug', 'Refreshing every %s seconds' % (arguments['-r']))
            while True:
                self.run()
                sleep(float(arguments['-r']))
                self.refresh(arguments['-u'])
        return 0

    @property
    def is_playing(self):
        return self.scrobbler.np or False

    @property
    def artist(self):
        return self.scrobbler.artist or None

    @property
    def title(self):
        return self.scrobbler.title or None


if __name__ == '__main__':
    arguments = docopt(__doc__, version='xolnowplaying v0.1.7')
    np = XolNowPlaying(arguments['-u'])
    exit(np.main(arguments))
