# -*- coding: UTF-8
from __future__ import unicode_literals
import codecs
import os
import datetime
from yozuch import rstparser
from tests import YozuchTestCase

ROOT_DIR = os.path.dirname(__file__)


def load_document(filename):
    path = os.path.join(ROOT_DIR, 'data', filename)
    with codecs.open(path, 'r', 'utf-8-sig') as f:
        return path, f.read()


class RstParserTest(YozuchTestCase):

    def test_post(self):
        path, text = load_document(os.path.join('posts', 'cat1', 'post.rst'))
        doc = rstparser.parse(text, path)
        rstparser.publish(doc)
        tags = [tag.name for tag in doc.tags]
        self.assertEqual(doc.title, '测试')
        self.assertEqual(doc.summary, None)
        self.assertEqual(doc.date, datetime.datetime(2013, 10, 2))
        self.assertEqual(tags, ['tag1'])
        self.assertEqual(doc.author, 'author1')
        self.assertEqual(doc.metadata['author'], 'author1')
        self.assertFalse(self._is_logger_errors())

    def test_post_additional_meta(self):
        path, text = load_document(os.path.join('posts', '2013-09-20-post-file-metadata.rst'))
        doc = rstparser.parse(text, path,
                              additional_meta={'date': datetime.datetime(2013, 9, 20), 'slug': 'post-file-metadata'})
        rstparser.publish(doc)
        self.assertEqual(doc.title, 'Title')
        self.assertEqual(doc.slug, 'post-file-metadata')
        self.assertEqual(doc.content, '<p>Content.</p>\n')
        self.assertEqual(doc.summary, None)
        self.assertEqual(doc.date, datetime.datetime(2013, 9, 20))
        self.assertEqual(doc.category.name, 'cat2')
        self.assertEqual(str(doc.category), 'cat2')
        self.assertEqual([tag.name for tag in doc.tags], ['tag1', 'tag 2'])
        self.assertEqual([tag.slug for tag in doc.tags], ['tag1', 'tag-2'])
        self.assertFalse(self._is_logger_errors())

    def test_post_default_meta(self):
        path, text = load_document(os.path.join('posts', 'post-readmore.rst'))
        doc = rstparser.parse(text, path, default_meta={'category': 'default'})
        rstparser.publish(doc)
        self.assertEqual(doc.category.name, 'default')
        self.assertFalse(self._is_logger_errors())

    def test_post_additional_default_meta(self):
        path, text = load_document(os.path.join('posts', 'post-readmore.rst'))
        doc = rstparser.parse(text, path, default_meta={'category': 'default'},
                              additional_meta={'category': 'additional'})
        rstparser.publish(doc)
        self.assertEqual(doc.category.name, 'additional')
        self.assertFalse(self._is_logger_errors())

    def test_post_readmore(self):
        path, text = load_document(os.path.join('posts', 'post-readmore.rst'))
        doc = rstparser.parse(text, path)
        rstparser.publish(doc)
        self.assertEqual(doc.title, 'ReadMore')
        self.assertEqual(doc.content, '<p>foo</p>\n<div class="section" id="section">\n<h3>Section</h3>\n<p>bar</p>\n<p>baz</p>\n</div>\n')
        self.assertEqual(doc.summary, '<p>foo</p>\n<div class="section" id="section">\n<h3>Section</h3>\n<p>bar</p>\n</div>')
        self.assertEqual(doc.date, datetime.datetime(2011, 1, 1))
        self.assertEqual(doc.category, None)
        self.assertFalse(self._is_logger_errors())

    def test_post_readmore_addheading(self):
        path, text = load_document(os.path.join('posts', 'post-readmore.rst'))
        doc = rstparser.parse(text, path)
        rstparser.publish(doc, None, None, '&#x00b6;')
        self.assertEqual(doc.title, 'ReadMore')
        self.assertEqual(doc.content, '<p>foo</p>\n<div class="section" id="section">\n<h3>Section<a class="headerlink" href="#section" title="Permalink to this headline">&#x00b6;</a></h3>\n<p>bar</p>\n<p>baz</p>\n</div>\n')
        self.assertEqual(doc.summary, '<p>foo</p>\n<div class="section" id="section">\n<h3>Section<a class="headerlink" href="#section" title="Permalink to this headline">&#x00b6;</a></h3>\n<p>bar</p>\n</div>')
        self.assertEqual(doc.date, datetime.datetime(2011, 1, 1))
        self.assertFalse(self._is_logger_errors())

    def test_post_error(self):
        path, text = load_document(os.path.join('posts', 'post-error.rst'))
        doc = rstparser.parse(text, path)
        self.assertEqual(doc, None)
        self.assertEqual(len(self.logger_handler.messages['error']), 1)

    def test_page(self):
        path, text = load_document(os.path.join('documents', 'page.rst'))
        doc = rstparser.parse(text, path)
        rstparser.publish(doc)
        self.assertEqual(doc.title, 'Заголовок')
        self.assertEqual(doc.slug, None)
        self.assertEqual(doc.content, '<p>Содержимое</p>\n')
        self.assertEqual(doc.summary, None)
        self.assertFalse(self._is_logger_errors())