"""
Content generators.
"""

import os
import codecs
from yozuch import logger
from yozuch.ttt import iteritems
from yozuch.utils import format_url_from_object, makedirs, path_from_url


_registered_generators = {}


def content_generator(name):
    def wrapper(cls):
        _registered_generators[name] = cls
        return cls
    return wrapper


class Generator(object):

    def __init__(self):
        self._generators = []

    def url_template_for(self, id, url_source=None):
        for url, _, route_params in self._generators:
            if 'id' in route_params and route_params['id'] == id:
                if url_source is not None:
                    return route_params[url_source]
                else:
                    return url

    def url_for(self, id, entry=None, url_source=None, **kwargs):
        url_template = self.url_template_for(id, url_source)
        if url_template is None:
            raise LookupError('Unable to resolve URL for generator "{}": {} {} {}'.format(id, entry, url_source, kwargs))
        if entry is not None:
            return format_url_from_object(url_template, entry)
        else:
            return url_template.format(**kwargs)

    def url_exists_for(self, id):
        return self.url_template_for(id) is not None

    def _initialize_generator(self, route):
        url_template, generator_name, params = route
        if generator_name in _registered_generators:
            cls = _registered_generators[generator_name]
            generator = cls()
            self._generators.append((url_template, generator, params))
        else:
            raise LookupError('Unable to find generator with name "{}" for url "{}".'.format(generator_name, url_template))

    def _invoke_generators(self, method_name, context, project_dir):
        for url_template, generator, params in self._generators:
            entries = getattr(generator, method_name)(context, url_template, project_dir, **params)
            for entry, publisher, writer in entries:
                context['entries'][entry.url] = entry, publisher, writer

    def add_route(self, route):
        self._initialize_generator(route)

    def init(self, context):
        for route in context['config'].ROUTES:
            self._initialize_generator(route)

    def generate(self, env, context, project_dir, output_dir):
        self._invoke_generators('load', context, project_dir)
        self._invoke_generators('generate', context, project_dir)

        for _, (entry, publisher, _) in iteritems(context['entries']):
            if publisher is not None:
                publisher.publish(entry, context)

        logger.info('Writing content...')
        for _, (entry, _, writer) in iteritems(context['entries']):
            writer.write(entry, context, env, output_dir)


class ContentGeneratorBase(object):

    def load(self, context, url_template, project_dir,  **kwargs):
        return iter([])

    def generate(self, context, url_template, project_dir,  **kwargs):
        return iter([])


class WriterBase(object):

    def write(self, entry, context, env, output_dir):
        raise NotImplementedError()


class PublisherBase(object):

    def publish(self, entry, context):
        raise NotImplementedError()


class TemplatePageGenerator(ContentGeneratorBase):

    _template = None
    _entry = None

    def _register_document_reference(self, context, entry, project_dir):
        path = '/' + os.path.relpath(entry.source_path, project_dir).replace('\\', '/')
        context['references'] = context.get('references', {})
        context['references'][path] = entry.url

    def __init__(self, template=None, **kwargs):
        self._template = template


class TemplateWriter(WriterBase):

    def __init__(self, id=None):
        self._id = id

    def _write_file(self, content, path):
        with codecs.open(path, 'w', 'utf-8') as f:
            f.write(content)

    def _write_page(self, entry, env, context, output_dir, template_vars=None):
        path = os.path.join(output_dir, path_from_url(entry.url))
        template = env.get_template(entry.template)
        template_vars = template_vars or {}
        template_vars.update({
            'site': context['site'],
            'theme': context['theme'],
            'page_id': self._id,
            'page_url': entry.url
        })
        content = template.render(**template_vars)
        makedirs(os.path.dirname(path))
        self._write_file(content, path)

    def write(self, entry, context, env, output_dir):
        self._write_page(entry, env, context, output_dir)


class TemplateEntryWriter(TemplateWriter):

    def __init__(self, id, entry_name):
        super(TemplateEntryWriter, self).__init__(id)
        self._entry_name = entry_name

    def write(self, entry, context, env, output_dir):
        self._write_page(entry, env, context, output_dir, {self._entry_name: entry})