"""
reStructuredText parser.
"""

from __future__ import unicode_literals
import datetime
import docutils.readers.standalone
import docutils.readers.doctree
import docutils.core
import docutils.nodes
import docutils.utils
import docutils.io
from docutils.parsers.rst import Directive, directives
from docutils.writers.html4css1 import Writer, HTMLTranslator as BaseTranslator
from yozuch import config, logger
from yozuch.entries import RSTDocument
from yozuch.ttt import unicode_strip, urljoin


_PUBLISH_SETTINGS = {
    'strip_comments': True,
    'initial_header_level': 3,
    'report_level': 2,
    'syntax_highlight': 'short',
    'stylesheet_path': None
}


class _ReadMoreDirective(Directive):

    MARK = '~~YOZUCH_READMORE_MARK~~'

    has_content = False

    def run(self):
        return [docutils.nodes.raw('', self.MARK, format='html')]

directives.register_directive('read-more', _ReadMoreDirective)


def _get_reporter_observer(source):
    def reporter_observer(msg):
        if msg['level'] >= _PUBLISH_SETTINGS['report_level']:
            log_level = {
                0: logger.debug,
                1: logger.info,
                2: logger.warning,
                3: logger.error,
                4: logger.critical
            }
            text = docutils.nodes.Element.astext(msg)
            log_level[msg['level']]('{source}, line {line}: {text}'.format(source=source, line=msg['line'], text=text))

    return reporter_observer


class _Reader(docutils.readers.standalone.Reader):

    def __init__(self):
        docutils.readers.standalone.Reader.__init__(self)

    def new_document(self):
        doc = docutils.utils.new_document(self.source.source_path, self.settings)
        doc.reporter.stream = False
        doc.reporter.attach_observer(_get_reporter_observer(self.source.source_path))
        return doc


def get_html_translator(rooturl, permalink_text, documents):

    class _HTMLTranslator(BaseTranslator):

        def __init__(self, document):
            BaseTranslator.__init__(self, document)

        def depart_title(self, node):
            close_tag = self.context[-1]
            parent = node.parent
            if permalink_text is not None and isinstance(parent, docutils.nodes.section) and parent.hasattr('ids') and parent['ids']:
                anchor_name = parent['ids'][0]
                if close_tag.startswith('</h'):
                    self.body.append(
                        '<a class="headerlink" href="#{}" title="Permalink to this headline">{}</a>'.format(anchor_name, permalink_text)
                    )
            BaseTranslator.depart_title(self, node)

        def visit_image(self, node):
            if rooturl is not None:
                uri = node['uri']
                if uri.find('://') == -1:
                    node['uri'] = urljoin(rooturl, uri)
            node['alt'] = ''
            BaseTranslator.visit_image(self, node)

        def visit_reference(self, node):
            if documents is not None and 'refuri' in node:
                uri = node['refuri']
                if uri in documents:
                    node['refuri'] = documents[uri]
            BaseTranslator.visit_reference(self, node)

    return _HTMLTranslator


def _find_node(doctree, node_class, remove=True):
    for node in doctree.traverse(node_class):
        if remove:
            node.parent.remove(node)
        return node


def _find_summary_and_remove_readmore(content):
    idx = content.find(_ReadMoreDirective.MARK)
    if idx != -1:
        summary = content[:idx]
        # TODO: think about a better way to close sections? rebuild doctree once more?
        # FIXME: summary may contain heading permalink
        summary += '</div>' * (summary.count('<div') - summary.count('</div'))
        return summary, content[:idx]+content[idx+len(_ReadMoreDirective.MARK):]
    return None, content


def _parse_and_remove_title(doctree):
    node = _find_node(doctree, docutils.nodes.title)
    if node is not None:
        return node.astext()


def _parse_and_remove_fields(doctree):

    def list_parser(s):
        return map(unicode_strip, s.split(','))

    def date_parser(s):
        return datetime.datetime.strptime(s, config.RST_META_DATE_FORMAT)

    parsers = {
        'tags': list_parser,
        'date': date_parser
    }

    fields = _find_node(doctree, docutils.nodes.field_list)
    if fields is not None:
        for field in fields:
            name = field[0].astext().lower()
            value = field[1].astext()
            if name in parsers:
                value = parsers[name](value)
            yield name, value


def parse(source, source_path, additional_meta=None, default_meta=None):
    doctree = docutils.core.publish_doctree(
        source=source, source_path=source_path,
        reader=_Reader(),
        settings_overrides=_PUBLISH_SETTINGS
    )
    if doctree.reporter.max_level < 3:
        doc = RSTDocument(doctree, source_path)
        doc.title = _parse_and_remove_title(doctree)
        meta = default_meta or {}
        meta.update(dict(_parse_and_remove_fields(doctree)))
        if additional_meta:
            meta.update(additional_meta)
        doc.metadata = meta
        return doc


def _publish(doctree, documents, rooturl, permalink_text):
    html_writer = Writer()
    html_writer.translator_class = get_html_translator(rooturl, permalink_text, documents)

    pub = docutils.core.Publisher(
        reader=docutils.readers.doctree.Reader(parser_name='null'),
        source=docutils.io.DocTreeInput(doctree),
        destination_class=docutils.io.StringOutput,
        writer=html_writer
    )

    pub.process_programmatic_settings(None, _PUBLISH_SETTINGS, None)
    pub.publish()

    content = pub.writer.parts['fragment']
    return _find_summary_and_remove_readmore(content)


def publish(doc, documents=None, rooturl=None, permalink_text=None):
    doc.summary, doc.content = _publish(doc.doctree, documents, rooturl, permalink_text)