"""
Site builder.
"""

import os
import datetime
import sys
import jinja2
from yozuch import config, logger
from yozuch.generators import Generator
from yozuch.route import assets
from yozuch.utils import is_external_url, remove_dir_contents


def build(project_dir, config_overrides=None):

    if not os.path.isfile(os.path.join(project_dir, 'config.py')):
        logger.error('Unable to find config.py in {}. Wrong project directory?'.format(project_dir))
        sys.exit(1)

    from yozuch.generators import blog_index, template, page, post, document, asset, tag, category  # TODO

    config.merge_from_directory(project_dir)
    if config_overrides:
        config.merge_from_object(config_overrides)

    if config.THEME_NAME is None:
        logger.warning('Theme is not specified')

    context = {
        'site': {
            # config aliases
            'url': config.URL,
            'title': config.TITLE,
            'author': config.AUTHOR,
            'description': config.DESCRIPTION,

            # current time
            'time': datetime.datetime.now(),
        },

        'config': config,
        'theme': config.THEME_CONFIG,
        'entries': {}
    }

    theme_package_dir = os.path.join('themes', config.THEME_NAME) if config.THEME_NAME is not None else None
    theme_project_dir = os.path.join(project_dir, 'themes', config.THEME_NAME) if config.THEME_NAME is not None else None
    templates_project_dir = os.path.join(project_dir, 'templates')
    pages_dir = os.path.join(project_dir, 'pages')
    output_dir = os.path.join(project_dir, 'output')

    remove_dir_contents(output_dir)

    generator = Generator()
    generator.init(context)

    # add theme static routes and load theme config
    if theme_project_dir is not None and os.path.isdir(theme_project_dir):
        _import_theme(generator, theme_project_dir)
    elif theme_package_dir is not None:
        _import_theme(generator, os.path.join(config.PACKAGE_DIR, theme_package_dir))

    _set_default_templates()

    env = _create_template_env(generator, templates_project_dir, theme_package_dir, theme_project_dir, pages_dir)
    generator.generate(env, context, project_dir, output_dir)

    logger.info('Done!')

    return output_dir


def _set_default_templates():
    for _, name, params in config.ROUTES:
        if 'template' not in params:
            if 'id' in params and params['id'] in config.THEME_DEFAULT_TEMPLATES:
                params['template'] = config.THEME_DEFAULT_TEMPLATES[params['id']]
            elif name in config.THEME_DEFAULT_TEMPLATES:
                params['template'] = config.THEME_DEFAULT_TEMPLATES[name]


def _import_theme(generator, directory):
    config.merge_from_theme_directory(directory)
    assets_url = generator.url_template_for(id='assets') or '/{filename}'
    generator.add_route(assets(assets_url, id='theme_assets', directory=os.path.join(directory, 'assets')))


def _create_template_env(generator, templates_project_dir, theme_package_dir, theme_project_dir, pages_dir):
    theme_loader = None
    if theme_project_dir is not None:
        project_theme_templates_dir = os.path.join(theme_project_dir, 'templates')
        if os.path.isdir(project_theme_templates_dir):
            theme_loader = jinja2.FileSystemLoader(project_theme_templates_dir)

    if theme_loader is None and theme_package_dir is not None:
        theme_loader = jinja2.PackageLoader('yozuch', os.path.join(theme_package_dir, 'templates'))

    loaders = []

    if os.path.isdir(templates_project_dir):
        loaders.append(jinja2.FileSystemLoader(templates_project_dir))

    if os.path.isdir(pages_dir):
        loaders.append(jinja2.PrefixLoader({'!pages': jinja2.FileSystemLoader(pages_dir)}))

    if theme_loader is not None:
        loaders.append(jinja2.PrefixLoader({'!theme': theme_loader}))
        loaders.append(theme_loader)
    loaders.append(jinja2.PackageLoader('yozuch', os.path.join('themes', 'base', 'templates')))

    env = jinja2.Environment(loader=jinja2.ChoiceLoader(loaders))
    env.globals['url_for'] = generator.url_for
    env.globals['url_exists_for'] = generator.url_exists_for
    env.globals['is_external_url'] = is_external_url
    return env