"""
Helpers.
"""

import os
import shutil
import string
import glob
from collections import OrderedDict


class ObjectFormatter(string.Formatter):

    def get_value(self, key, args, kwargs):
        if len(args) < 1 or not isinstance(args[0], object):
            raise TypeError()
        try:
            val = getattr(args[0], key)
        except AttributeError:
            val = args[0][key]
        if val is None:
            raise TypeError("'{}' object attribute '{}' is None".format(args[0], key))
        return val


def format_object(fmt, obj):
    return ObjectFormatter().format(fmt, obj)


def format_url_from_object(fmt, obj):
    url = format_object(fmt, obj)
    if url.endswith('/index/'):
        url = url.replace('/index/', '/')
    return url


def copytree(src, dst, ignore=None):
    names = os.listdir(src)
    if ignore is not None:
        ignored_names = ignore(src, names)
    else:
        ignored_names = set()

    makedirs(dst)

    for name in names:
        if name in ignored_names:
            continue
        srcname = os.path.join(src, name)
        dstname = os.path.join(dst, name)

        if os.path.isdir(srcname):
            copytree(srcname, dstname, ignore)
        else:
            try:
                shutil.copy(srcname, dstname)
            except IOError:
                pass


def makedirs(path):
    try:
        os.makedirs(path)
    except OSError:
        pass


def remove_dir_contents(path):
    if os.path.isdir(path):
        for name in os.listdir(path):
            item_path = os.path.join(path, name)
            if os.path.isfile(item_path):
                os.remove(item_path)
            elif os.path.isdir(item_path):
                shutil.rmtree(item_path, True)


def path_from_url(url):
    path = url.lstrip('/').replace('/', os.path.sep)
    filename = os.path.basename(path)
    return os.path.join(path, 'index.html') if not filename else path


def normalize_slug(s):
    return s.replace(' ', '-').lower()


def unique_list_items(l):
    return list(OrderedDict.fromkeys(l))


def glob2(root, pattern):
    dirname, pattern = os.path.split(pattern)
    if not dirname:
        dirname = root
    else:
        dirname = os.path.join(root, dirname)
    return glob.glob(os.path.join(dirname, pattern))


def is_external_url(url):
    if url.startswith('//'):
        return True
    schemes = ('http', 'https', 'skype', 'mailto')
    return any([url.startswith(scheme + ':') for scheme in schemes])