"""
reStructuredText loader.
"""

import os
import codecs
import re
import datetime
from yozuch import config, rstparser, logger
from yozuch.utils import glob2

_filename_meta_regexs = [re.compile(rs) for rs in config.FILENAME_META_FORMATS]


def _parse_filename_meta(name):
    parsers = {
        'date': lambda s: datetime.datetime.strptime(s, config.FILENAME_META_DATE_FORMAT)
    }
    for regex in _filename_meta_regexs:
        result = regex.search(name)
        if result is not None:
            fields = result.groupdict()
            for name in parsers:
                if name in fields:
                    fields[name] = parsers[name](fields[name])
            return fields


def _load_documents(content_dir, source, validate_date, default_category):
    for path in glob2(content_dir, source):
        filename = os.path.basename(path)
        name, ext = os.path.splitext(filename)
        if ext in config.DOCUMENT_FILE_EXTENSIONS:
            with codecs.open(path, 'r', 'utf-8-sig') as f:
                default_meta = None
                if default_category is not None:
                    default_meta = {
                        'category': default_category
                    }
                doc = rstparser.parse(f.read(), path, additional_meta=_parse_filename_meta(name),
                                      default_meta=default_meta)
                if doc is not None:
                    doc.filename = filename
                    if validate_date and not doc.date:
                        logger.error("Post must contain 'date' metadata")
                        yield
                    yield doc


def load_documents(content_dir, source, validate_date=False, default_category=None, subdirectory_as_category=False):
    # Python 2 doesn't have yield from :(
    docs = []
    docs.extend(_load_documents(content_dir, source, validate_date, default_category))
    if subdirectory_as_category:
        for dirname in os.listdir(content_dir):
            subdir_path = os.path.join(content_dir, dirname)
            if os.path.isdir(subdir_path):
                docs.extend(_load_documents(subdir_path, source, validate_date, dirname))
    return docs