"""
Content generators.
"""

import os
import codecs
from yozuch import logger
from yozuch.ttt import iteritems
from yozuch.utils import makedirs, path_from_url, is_external_url, import_module_member


class GeneratorManager(object):

    def __init__(self, context):
        self._context = context
        self._generators = []

    def _url_template_for(self, name, url_source=None):
        for url, _, route_params in self._generators:
            if 'name' in route_params and route_params['name'] == name:
                if url_source is not None:
                    return route_params[url_source]
                else:
                    return url

    def _url_for(self, name, url_source=None, **kwargs):
        url_template = self._url_template_for(name, url_source)
        if url_template is None:
            raise LookupError('Unable to resolve URL for {} {} {}'.format(name, url_source, kwargs))
        else:
            return url_template.format(**kwargs)

    def url_for(self, name_or_url, url_source=None, **kwargs):
        references = self._context.get('references', {})
        if name_or_url in references:
            return references[name_or_url]
        if name_or_url.startswith('/') or is_external_url(name_or_url):
            return name_or_url
        return self._url_for(name_or_url, url_source, **kwargs)

    def url_exists_for(self, name):
        return self._url_template_for(name) is not None

    def _initialize_generator(self, route):
        url_template, generator, params = route
        cls = import_module_member(generator)
        if cls is not None:
            generator_cls = cls()
            self._generators.append((url_template, generator_cls, params))
        else:
            raise LookupError('Unable to find generator with name "{}" for url "{}".'.format(generator, url_template))

    def init(self):
        for route in self._context['config']['ROUTES']:
            self._initialize_generator(route)

    def generate(self, env, output_dir):
        context = self._context

        for url_template, generator, params in self._generators:
            entries = generator.generate(self._context, url_template, **params)
            for entry, publisher, writer in entries:
                self._context['entries'][entry.url] = entry, publisher, writer

        for _, (entry, publisher, _) in iteritems(context['entries']):
            if publisher is not None:
                publisher.publish(entry, context)

        logger.info('Writing content...')
        for _, (entry, _, writer) in iteritems(context['entries']):
            writer.write(entry, context, env, output_dir)


class ContentGeneratorBase(object):

    def generate(self, context, url_template, **kwargs):
        raise NotImplementedError()


class WriterBase(object):

    def write(self, entry, context, env, output_dir):
        raise NotImplementedError()


class PublisherBase(object):

    def publish(self, entry, context):
        raise NotImplementedError()


class TemplatePageGenerator(ContentGeneratorBase):

    _template = None
    _entry = None

    def _register_document_reference(self, context, entry):
        context['references'] = context.get('references', {})
        context['references'][entry.id] = entry.url

    def __init__(self, template=None, **kwargs):
        self._template = template


class TemplateWriter(WriterBase):

    def __init__(self, page_id=None):
        self._page_id = page_id

    def _write_file(self, content, path):
        with codecs.open(path, 'w', 'utf-8') as f:
            f.write(content)

    def _write_page(self, entry, env, context, output_dir, template_vars=None):
        if entry.template is None:
            raise LookupError('Template is not specified for {} {}'.format(entry, self._page_id))
        path = os.path.join(output_dir, path_from_url(entry.url))
        template = env.get_template(entry.template)
        template_vars = template_vars or {}
        template_vars.update({
            'site': context['site'],
            'theme': context['theme'],
            'config': context['config'],
            'page_id': self._page_id,
            'page_url': entry.url
        })
        content = template.render(**template_vars)
        makedirs(os.path.dirname(path))
        self._write_file(content, path)

    def write(self, entry, context, env, output_dir):
        self._write_page(entry, env, context, output_dir)


class TemplateEntryWriter(TemplateWriter):

    def __init__(self, page_id, entry_name):
        super(TemplateEntryWriter, self).__init__(page_id)
        self._entry_name = entry_name

    def write(self, entry, context, env, output_dir):
        self._write_page(entry, env, context, output_dir, {self._entry_name: entry})