"""
reStructuredText loader.
"""

import os
import codecs
import re
import datetime
from yozuch import logger
from yozuch.ttt import iteritems
from yozuch.rstparser import RstParser


class RstLoader(object):

    def __init__(self, options):
        self._options = options
        self._filename_meta_regexs = [re.compile(rs) for rs in options['filename_meta_formats']]

    def _parse_filename_meta(self, name):
        parsers = {
            'date': lambda s: datetime.datetime.strptime(s, self._options['filename_meta_date_format'])
        }
        for regex in self._filename_meta_regexs:
            result = regex.search(name)
            if result is not None:
                fields = result.groupdict()
                for name, parser in iteritems(parsers):
                    if name in fields:
                        fields[name] = parser(fields[name])
                return fields

    def _load_documents(self, content_dir, validate_date, default_category):
        for filename in os.listdir(content_dir):
            path = os.path.join(content_dir, filename)
            if os.path.isfile(path):
                name, ext = os.path.splitext(filename)
                if ext in self._options['file_extensions']:
                    with codecs.open(path, 'r', 'utf-8-sig') as f:
                        default_meta = None
                        if default_category is not None:
                            default_meta = {
                                'category': default_category
                            }
                        parser = RstParser(self._options)
                        doc = parser.parse(f.read(), path, additional_meta=self._parse_filename_meta(name),
                                           default_meta=default_meta)
                        if doc is not None:
                            doc.filename = filename
                            if validate_date and not doc.date:
                                logger.error("Post must contain 'date' metadata")
                                yield
                            yield doc

    def load_documents(self, content_dir, validate_date=False, default_category=None, subdirectory_as_category=False):
        # Python 2 doesn't have yield from :(
        docs = list(self._load_documents(content_dir, validate_date, default_category))
        if subdirectory_as_category:
            for dirname in os.listdir(content_dir):
                subdir_path = os.path.join(content_dir, dirname)
                if os.path.isdir(subdir_path):
                    docs.extend(self._load_documents(subdir_path, validate_date, dirname))
        return docs