"""
Site builder.
"""

import os
import datetime
import sys
import jinja2
from yozuch import logger
from yozuch.generators import GeneratorManager
from yozuch.loaders import LoaderManager
from yozuch.utils import remove_dir_contents
from yozuch.config import Config, PACKAGE_DIR
from yozuch.validator import validate as validate_site


def build(project_dir, config_overrides=None, validate=False, output_dir='output'):

    if not os.path.isfile(os.path.join(project_dir, 'config.py')):
        logger.error('Unable to find config.py in {}. Wrong project directory?'.format(project_dir))
        sys.exit(1)

    config = Config()

    config.update_from_directory(project_dir)
    if config_overrides:
        config.update_from_dict(config_overrides)

    if config['THEME_NAME'] is None:
        logger.warning('Theme is not specified')

    context = {
        'site': {
            # config aliases
            'url': config['URL'],
            'title': config['TITLE'],
            'author': config['AUTHOR'],
            'description': config['DESCRIPTION'],

            # current time
            'time': datetime.datetime.now(),
        },
        'project_dir': project_dir,
        'config': config,
        'theme': config['THEME_CONFIG'],
        'entries': {},
        'collections': {},
    }

    theme_package_dir = os.path.join('themes', config['THEME_NAME']) if config['THEME_NAME'] is not None else None
    theme_project_dir = os.path.join(project_dir, 'themes', config['THEME_NAME']) if config['THEME_NAME'] is not None else None
    templates_project_dir = os.path.join(project_dir, 'templates')
    pages_dir = os.path.join(project_dir, 'pages')
    output_dir = os.path.join(project_dir, output_dir)

    remove_dir_contents(output_dir)

    # add theme static to loader
    if theme_project_dir is not None and os.path.isdir(theme_project_dir):
        _import_theme(config, theme_project_dir)
    elif theme_package_dir is not None:
        _import_theme(config, os.path.join(PACKAGE_DIR, theme_package_dir))

    loader = LoaderManager()
    loader.load(context)

    generator = GeneratorManager(context)
    generator.init()

    _set_default_templates(config)

    env = _create_template_env(generator, templates_project_dir, theme_package_dir, theme_project_dir, pages_dir)
    generator.generate(env, output_dir)

    if validate:
        validate_site(config, output_dir)

    logger.info('Done!')

    return output_dir


def _set_default_templates(config):
    theme_default_templates = config['THEME_DEFAULT_TEMPLATES']
    for _, name, params in config['ROUTES']:
        if 'template' not in params:
            if 'name' in params and params['name'] in theme_default_templates:
                params['template'] = theme_default_templates[params['name']]
            elif name in theme_default_templates:
                params['template'] = theme_default_templates[name]


def _import_theme(config, directory):
    config.update_from_directory(directory, replace_duplicates=False)
    config['LOADERS'].append(('yozuch.loaders.asset.AssetLoader', {'directory': os.path.join(directory, 'assets')}))


def _create_template_env(generator, templates_project_dir, theme_package_dir, theme_project_dir, pages_dir):
    theme_loader = None
    if theme_project_dir is not None:
        project_theme_templates_dir = os.path.join(theme_project_dir, 'templates')
        if os.path.isdir(project_theme_templates_dir):
            theme_loader = jinja2.FileSystemLoader(project_theme_templates_dir)

    if theme_loader is None and theme_package_dir is not None:
        theme_loader = jinja2.PackageLoader('yozuch', os.path.join(theme_package_dir, 'templates'))

    loaders = []

    if os.path.isdir(templates_project_dir):
        loaders.append(jinja2.FileSystemLoader(templates_project_dir))

    if os.path.isdir(pages_dir):
        loaders.append(jinja2.PrefixLoader({'!pages': jinja2.FileSystemLoader(pages_dir)}))

    if theme_loader is not None:
        loaders.append(jinja2.PrefixLoader({'!theme': theme_loader}))
        loaders.append(theme_loader)
    loaders.append(jinja2.PackageLoader('yozuch', os.path.join('themes', 'base', 'templates')))

    env = jinja2.Environment(loader=jinja2.ChoiceLoader(loaders))
    env.globals['url_for'] = generator.url_for
    env.globals['url_exists_for'] = generator.url_exists_for
    return env