"""
Default configuration and configuration related helpers.
"""

import imp
import os
import sys
from yozuch.ttt import iteritems
from yozuch.route import route


PACKAGE_DIR = os.path.dirname(__file__)


class DefaultConfig(object):

    DEBUG = False

    TITLE = 'Site title'
    URL = 'http://localhost:8000'
    DESCRIPTION = 'Site description'
    AUTHOR = 'Name Surname'

    THEME_NAME = None
    THEME_CONFIG = {}
    THEME_DEFAULT_TEMPLATES = {
        'blog-index': 'blog_index.html',
        'posts': 'post.html',
        'tags-index': 'tags_index.html',
        'tags': 'tag.html',
        'categories-index': 'categories_index.html',
        'categories': 'category.html',
        'archive-index': 'archive_index.html',
        'atom-feed': 'atom.xml',
        'documents': 'document.html',
    }

    RST_OPTIONS = {
        'meta_date_format': '%Y-%m-%d',
        'filename_meta_formats': ['(?P<date>\d{4}-\d{2}-\d{2})-(?P<slug>.*)', '(?P<slug>.*)'],
        'filename_meta_date_format': '%Y-%m-%d',
        'file_extensions': ['.rst'],
        'add_heading_permalink': False,
        'permalink_text': '&#x00b6;',
    }

    PAGE_FILE_EXTENSIONS = ['.html', '.xml']

    TAXONOMIES = [
        ('tags', 'tags', 'tag'),
        ('categories', 'category', 'category'),
    ]

    LOADERS = [
        ('yozuch.loaders.post.PostLoader', {}),
        ('yozuch.loaders.document.DocumentLoader', {}),
        ('yozuch.loaders.page.PageLoader', {}),
        ('yozuch.loaders.asset.AssetLoader', {}),
    ]

    ROUTES = (
        route('/', 'blog-index'),
        route('/blog/{date:%Y}/{date:%m}/{date:%d}/{slug}/', 'posts'),
        route('/tags/', 'tags-index'),
        route('/tag/{slug}/', 'tags'),
        route('/categories/', 'categories-index'),
        route('/category/{slug}/', 'categories'),
        route('/archive/', 'archive-index'),
        route('/{slug}/', 'documents'),
        route('/atom.xml', 'atom-feed'),
        route('/{filename}', 'assets'),
    )


class Config(dict):

    KEYS_TO_MERGE = ['RST_OPTIONS', 'THEME_CONFIG', 'THEME_DEFAULT_TEMPLATES']

    def __init__(self, *args, **kwargs):
        super(Config, self).__init__(*args, **kwargs)
        self.update_from_object(DefaultConfig)

    def update_from_object(self, obj, replace_duplicates=True):
        self.update_from_dict({key: getattr(obj, key) for key in dir(obj) if key.isupper()}, replace_duplicates)

    def update_from_dict(self, other, replace_duplicates=True):
        for key, value in iteritems(other):
            if key.isupper():
                if key in self.KEYS_TO_MERGE and self.get(key):
                    if not replace_duplicates:
                        other[key].update(self[key])
                    self[key].update(other[key])
                else:
                    self[key] = other[key]

    def update_from_directory(self, source_dir, replace_duplicates=True):
        try:
            fp, pathname, description = imp.find_module('config', [source_dir])
        except ImportError:
            return
        try:
            if 'user_config' in sys.modules:
                del sys.modules['user_config']  # not sure if it's ok to do such hack
            module = imp.load_module('user_config', fp, pathname, description)
            self.update_from_object(module, replace_duplicates)
        finally:
            fp.close()