__title__ = 'admin_timeline.views'
__version__ = '0.5'
__build__ = 0x000005
__author__ = 'Artur Barseghyan <artur.barseghyan@gmail.com>'
__all__ = ('log',)

import datetime

from django.http import HttpResponse
from django.utils import simplejson as json
from django.views.decorators.cache import never_cache
from django.contrib.admin.views.decorators import staff_member_required
from django.contrib.admin.models import LogEntry
from django.shortcuts import render_to_response
from django.template import RequestContext
from django.template.loader import render_to_string
from django.template.defaultfilters import date as date_format
from django.utils.translation import ugettext_lazy as _

from admin_timeline.settings import NUMBER_OF_ENTRIES_PER_PAGE, SINGLE_LOG_ENTRY_DATE_FORMAT
from admin_timeline.settings import LOG_ENTRIES_DAY_HEADINGS_DATE_FORMAT


@never_cache
@staff_member_required
def log(request, template_name="admin_timeline/timeline.html", template_name_ajax="admin_timeline/timeline_ajax.html"):
    """
    Get number of log entires. Serves both non-AJAX and AJAX driven requests.

    Since we have a breakdown of entries per day per entry and we have an AJAX driven infinite scroll and we want to
    avoid having duplicated date headers, we always pass a variable named "last_date" when making another request
    to our main AJAX-driven view. So... this is our case scenario:

    Initial timeline rendered as a normal HTML (non AJAX request) (from a list of log entries). We send date of
    last element as "last_date" to the context too, which will be used an an initial value for a global JavaScript
    variable. Later on that date will be used to send it to the AJAX driven view and used in
    rendering ("render_to_string" method). After we have rendered the HTML to send back, we get the last date of
    the last element and send it along with the HTML rendered to our view in JSON response. When receiving the
    JSON response, we update the above mentioned global JavaScript variable with the value given.

    :param request: django.http.HttpRequest
    :param template_name: str
    :param template_name_ajax: str
    :return: django.http.HttpResponse

    This view accepts the following GET variables (all optional).
    :param page: int - Page number to get.
    :param user_id: int - If set, used to filter the user by.
    :param last_date: str - Example value "2012-05-24".
    :param start_date: str - If set, used as a start date to filter the actions with. Example value "2012-05-24".
    :param end_date: str - If set, used as an end date to filter the actions with. Example value "2012-05-24".

    NOTE: If it gets too complicatd with filtering, we need to have forms to validate and process the GET data.
    """
    def _get_date_from_string(s):
        """
        Gets date from a string given.

        :param s: str - date in string format
        :return: datetime.datetime
        """
        try:
            return datetime.date(*map(lambda x: int(x), s.split("-")))
        except Exception, e:
            return ""

    try:
        page = int(request.GET.get('page', 1))
        if page < 1:
            page = 1
    except Exception, e:
        page = 1

    # Some kind of a pagination
    start = (page - 1) * NUMBER_OF_ENTRIES_PER_PAGE
    end = page * NUMBER_OF_ENTRIES_PER_PAGE

    # Getting admin log entires taking page number into consideration.
    log_entries = LogEntry.objects.all().select_related('content_type', 'user')

    # Getting user
    try:
        user_id = int(request.GET.get('user_id', None))
    except Exception, e:
        user_id = ""

    start_date = _get_date_from_string(request.GET.get('start_date'))
    end_date = _get_date_from_string(request.GET.get('end_date'))

    if start_date:
        log_entries = log_entries.filter(action_time__gte=start_date) # TODO

    if end_date:
        log_entries = log_entries.filter(action_time__lte=end_date) # TODO

    # If user provided, filtering by user
    if user_id:
        log_entries = log_entries.filter(user__id=user_id)

    # Applying limits / freezing the queryset
    log_entries = log_entries[start:end]

    if log_entries:
        last_date = date_format(log_entries[len(log_entries) - 1].action_time, "Y-m-d")
    else:
        last_date = request.GET.get('last_date', None)

    # Using different template for AJAX driven requests
    if request.is_ajax():
        # Context to render the AJAX driven HTML with
        context = {
            'admin_log': log_entries,
            'number_of_entries_per_page': NUMBER_OF_ENTRIES_PER_PAGE,
            'page': page,
            'last_date': request.GET.get('last_date', None),
            'SINGLE_LOG_ENTRY_DATE_FORMAT': SINGLE_LOG_ENTRY_DATE_FORMAT,
            'LOG_ENTRIES_DAY_HEADINGS_DATE_FORMAT': LOG_ENTRIES_DAY_HEADINGS_DATE_FORMAT
        }

        # Rendering HTML for an AJAX driven request
        html = render_to_string(
            template_name_ajax,
            context,
            context_instance=RequestContext(request)
        )

        # Context to send back to user in a JSON response
        context = {
            'html': html,
            'last_date': last_date,
            'success': 1 if len(log_entries) else 0
        }
        return HttpResponse(json.dumps(context))

    # Getting a plain list of all users who have done anything.
    users = LogEntry.objects.all().select_related('user').distinct().order_by('user') \
                    .values_list('user_id', 'user__username')

    # Context for a non-AJAX request
    context = {
        'admin_log': log_entries,
        'number_of_entries_per_page': NUMBER_OF_ENTRIES_PER_PAGE,
        'page': page,
        'last_date': last_date,
        'user_id': user_id,
        'start_date': date_format(start_date, "Y-m-d") if start_date else "",
        'end_date': date_format(end_date, "Y-m-d") if end_date else "",
        'admin_log_users': users,
        'SINGLE_LOG_ENTRY_DATE_FORMAT': SINGLE_LOG_ENTRY_DATE_FORMAT,
        'LOG_ENTRIES_DAY_HEADINGS_DATE_FORMAT': LOG_ENTRIES_DAY_HEADINGS_DATE_FORMAT,
        'title': _("Timeline") # For template breadcrumbs, etc.
    }

    return render_to_response(template_name, context, context_instance=RequestContext(request))
