#!/usr/bin/env python
"""Run Tulip unittests.

Usage:
  python3 runtests.py [flags] [pattern] ...

Patterns are matched against the fully qualified name of the test,
including package, module, class and method,
e.g. 'aiotest.test_events.PolicyTests.testPolicy'.

For full help, try --help.

runtests.py --coverage is equivalent of:

  $(COVERAGE) run --branch runtests.py -v
  $(COVERAGE) html $(list of files)
  $(COVERAGE) report -m $(list of files)

"""

# Originally written by Beech Horn (for NDB).

from __future__ import print_function
import aiotest
import gc
import logging
import optparse
import os
import random
import re
import sys
import textwrap
import traceback
PY33 = (sys.version_info >= (3, 3))
if PY33:
    import importlib.machinery
else:
    import imp
try:
    import coverage
except ImportError:
    coverage = None
if sys.version_info < (3,):
    sys.exc_clear()

try:
    import unittest
    from unittest.signals import installHandler
except ImportError:
    import unittest2 as unittest
    from unittest2.signals import installHandler

if PY33:
    def load_module(modname, sourcefile):
        loader = importlib.machinery.SourceFileLoader(modname, sourcefile)
        return loader.load_module()
else:
    def load_module(modname, sourcefile):
        return imp.load_source(modname, sourcefile)


def load_modules(basedir, suffix='.py'):
    def list_dir(prefix, dir):
        files = []

        modpath = os.path.join(dir, '__init__.py')
        if os.path.isfile(modpath):
            mod = os.path.split(dir)[-1]
            files.append(('{0}{1}'.format(prefix, mod), modpath))

            prefix = '{0}{1}.'.format(prefix, mod)

        for name in os.listdir(dir):
            path = os.path.join(dir, name)

            if os.path.isdir(path):
                files.extend(list_dir('{0}{1}.'.format(prefix, name), path))
            else:
                if (name != '__init__.py' and
                    name.endswith(suffix) and
                    not name.startswith(('.', '_'))):
                    files.append(('{0}{1}'.format(prefix, name[:-3]), path))

        return files

    mods = []
    for modname, sourcefile in list_dir('', basedir):
        if modname == 'runtests':
            continue
        try:
            mod = load_module(modname, sourcefile)
            mods.append((mod, sourcefile))
        except SyntaxError:
            raise
        #except Exception as err:
        #    print("Skipping '{0}': {1}".format(modname, err), file=sys.stderr)

    return mods


def randomize_tests(tests, seed):
    if seed is None:
        seed = random.randrange(10000000)
    random.seed(seed)
    print("Using random seed", seed)
    random.shuffle(tests._tests)


class TestsFinder:

    def __init__(self, testsdir, includes=(), excludes=()):
        self._testsdir = testsdir
        self._includes = includes
        self._excludes = excludes
        self.find_available_tests()

    def find_available_tests(self):
        """
        Find available test classes without instantiating them.
        """
        self._test_factories = []
        mods = [mod for mod, _ in load_modules(self._testsdir)]
        for mod in mods:
            for name in set(dir(mod)):
                if name.endswith('Tests'):
                    self._test_factories.append(getattr(mod, name))

    def load_tests(self):
        """
        Load test cases from the available test classes and apply
        optional include / exclude filters.
        """
        loader = unittest.TestLoader()
        suite = unittest.TestSuite()
        for test_factory in self._test_factories:
            tests = loader.loadTestsFromTestCase(test_factory)
            if self._includes:
                tests = [test
                         for test in tests
                         if any(re.search(pat, test.id())
                                for pat in self._includes)]
            if self._excludes:
                tests = [test
                         for test in tests
                         if not any(re.search(pat, test.id())
                                    for pat in self._excludes)]
            suite.addTests(tests)
        return suite


class TestResult(unittest.TextTestResult):

    def __init__(self, stream, descriptions, verbosity):
        super().__init__(stream, descriptions, verbosity)
        self.leaks = []

    def startTest(self, test):
        super().startTest(test)
        gc.collect()

    def addSuccess(self, test):
        super().addSuccess(test)
        gc.collect()
        if gc.garbage:
            if self.showAll:
                self.stream.writeln(
                    "    Warning: test created {} uncollectable "
                    "object(s).".format(len(gc.garbage)))
            # move the uncollectable objects somewhere so we don't see
            # them again
            self.leaks.append((self.getDescription(test), gc.garbage[:]))
            del gc.garbage[:]


class TestRunner(unittest.TextTestRunner):
    resultclass = TestResult

    def run(self, test):
        result = super().run(test)
        if result.leaks:
            self.stream.writeln("{0} tests leaks:".format(len(result.leaks)))
            for name, leaks in result.leaks:
                self.stream.writeln(' '*4 + name + ':')
                for leak in leaks:
                    self.stream.writeln(' '*8 + repr(leak))
        return result


def _runtests(config, tests):
    # Set the global variable
    aiotest.config = config

    if config.find_leaks:
        runner_factory = TestRunner
    else:
        runner_factory = unittest.TextTestRunner

    if config.randomize:
        randomize_tests(tests, config.random_seed)

    runner = runner_factory(verbosity=config.verbosity,
                            failfast=config.fail_fast)
    return runner.run(tests)

def runtests(config):
    if config.verbosity == 0:
        level = logging.CRITICAL
    elif config.verbosity == 1:
        level = logging.ERROR
    elif config.verbosity == 2:
        level = logging.WARNING
    elif config.verbosity == 3:
        level = logging.INFO
    elif config.verbosity >= 4:
        level = logging.DEBUG
    logging.basicConfig(level=level)

    testsdir = os.path.realpath(os.path.dirname(__file__))
    finder = TestsFinder(testsdir, config.includes, config.excludes)

    if hasattr(config.asyncio, 'coroutines'):
        debug = config.asyncio.coroutines._DEBUG
    else:
        # Tulip <= 3.4.1
        debug = config.asyncio.tasks._DEBUG
    if debug:
        print("Run tests in debug mode")
    else:
        print("Run tests in release mode")
    sys.stdout.flush()

    if config.catch_break:
        installHandler()

    tests = finder.load_tests()
    if config.forever:
        while True:
            result = _runtests(config, tests)
            if not result.wasSuccessful():
                sys.exit(1)
    else:
        result = _runtests(config, tests)
        sys.exit(not result.wasSuccessful())

def _load_event_loop_policy(policy_path):
    parts = policy_path.strip().split('.')
    attr = None
    mod = None
    modname = []
    use_import = True
    for part in parts:
        modname.append(part)
        if use_import:
            try:
                mod = __import__('.'.join(modname))
            except ImportError:
                if not hasattr(attr, part):
                    print("ERROR: Unable to get the event loop policy "
                          "from: %r" % policy_path)
                    traceback.print_exc()
                    sys.exit(1)
                use_import = False
            else:
                for part in reversed(modname[1:]):
                    mod = getattr(mod, part)
                attr = mod
        if not use_import:
            try:
                attr = getattr(attr, part)
            except AttributeError:
                print("ERROR: Unable to get the event loop policy "
                      "from: %r" % policy_path)
                if attr is not None:
                    traceback.print_exc()
                sys.exit(1)
    return attr


def _process_options(config, options, patterns):
    config.fail_fast = options.failfast
    config.forever = options.forever
    config.randomize = options.randomize
    config.random_seed = options.seed
    config.find_leaks = options.findleaks
    config.catch_break = options.catchbreak

    if options.quiet:
        config.verbosity = 0
    else:
        config.verbosity = options.verbose + 1

    if options.exclude:
        config.excludes = patterns
    else:
        config.includes = patterns


def _parse_options():
    usage = "%prog [options] [pattern] [pattern2 ...]"
    parser = optparse.OptionParser(description="Run all unittests.",
        usage=usage)
    parser.add_option(
        '-v', '--verbose', action="store_true", dest='verbose',
        default=0, help='verbose')
    parser.add_option(
        '-x', action="store_true", dest='exclude', help='exclude tests')
    parser.add_option(
        '-f', '--failfast', action="store_true", default=False,
        dest='failfast', help='Stop on first fail or error')
    parser.add_option(
        '-c', '--catch', action="store_true", default=False,
        dest='catchbreak', help='Catch control-C and display results')
    parser.add_option(
        '--forever', action="store_true", dest='forever', default=False,
        help='run tests forever to catch sporadic errors')
    parser.add_option(
        '--findleaks', action='store_true', dest='findleaks',
        help='detect tests that leak memory')
    parser.add_option(
        '-r', '--randomize', action='store_true',
        help='randomize test execution order.')
    parser.add_option(
        '--seed', type=int,
        help='random seed to reproduce a previous random run')
    parser.add_option(
        '-q', action="store_true", dest='quiet', help='quiet')
    parser.add_option(
        '--coverage', action="store_true", dest='coverage',
        help='enable html coverage report')

    options, args = parser.parse_args()
    return options, args


def main(config):
    options, args = _parse_options()
    patterns = args
    _process_options(config, options, patterns)

    if options.coverage and coverage is None:
        URL = "bitbucket.org/pypa/setuptools/raw/bootstrap/ez_setup.py"
        print(textwrap.dedent("""
            coverage package is not installed.

            To install coverage3 for Python 3, you need:
              - Setuptools (https://pypi.python.org/pypi/setuptools)

              What worked for me:
              - download {0}
                 * curl -O https://{0}
              - python3 ez_setup.py
              - python3 -m easy_install coverage
        """.format(URL)).strip())
        sys.exit(1)

    if options.coverage:
        cov = coverage.coverage(branch=True,
                                source=['asyncio'],
                                )
        cov.start()

    try:
        runtests(config)
    finally:
        if options.coverage:
            cov.stop()
            cov.save()
            cov.html_report(directory='htmlcov')
            print("\nCoverage report:")
            cov.report(show_missing=False)
            here = os.path.dirname(os.path.abspath(__file__))
            print("\nFor html report:")
            print("open file://{0}/htmlcov/index.html".format(here))
