'''This is a calibration file for data collected at PAPER in Green Bank
on JD 2454564.'''

import aipy as a, numpy as n

class AntennaArray(a.fit.AntennaArray):
    def __init__(self, *args, **kwargs):
        self.blpolys = {
            # ---------------CYG-----------------------------------------
            a.miriad.ij2bl(0,1): [-232584217.07434878, 286521647.31394947, -153946492.32064945, 47122995.490021512, -8988716.1442265864, 1094205.41369016, -83018.310352084038, 3589.593617866537, -67.728697699535218] ,
            a.miriad.ij2bl(1,2): [498917273.41225547, -605571515.18029666, 321027254.3157112, -97078828.775091439, 18315294.427484743, -2207457.326350383, 165974.67853106419, -7117.4760754829413, 133.27306917564334] ,
            a.miriad.ij2bl(1,3): [-101675405.81259295, 123964923.94303529, -65950600.200350694, 19999341.011060353, -3781528.2496226514, 456601.91445233446, -34387.158552757493, 1477.06896436843, -27.71076554440069] ,
            a.miriad.ij2bl(0,4): [-119765698.97738941, 146673216.44969541, -78329267.705297127, 23827170.041310035, -4515998.2562410096, 546150.15540957765, -41162.07497045747, 1767.8609369733979, -33.131730814909552] ,
            a.miriad.ij2bl(1,5): [165072286.45539233, -196906256.84772852, 102634681.41546345, -30531213.961183086, 5668974.829190203, -672739.64002902003, 49823.798448355265, -2105.3008044015637, 38.8549848024998] ,
            a.miriad.ij2bl(4,6): [89304307.085157663, -105218855.5249964, 54166810.691475131, -15915886.138127174, 2919827.0943721645, -342498.92621388403, 25088.489738093027, -1049.3213582290305, 19.186028371042116] ,
            a.miriad.ij2bl(5,7): [-370013124.48666418, 447411487.07525665, -236172173.76994443, 71081817.349527732, -13341676.514251227, 1599130.0825915278, -119531.50652952616, 5094.4409022333293, -94.788442747546668] ,
            a.miriad.ij2bl(1,7): [-212541478.33927828, 254599650.73375177, -133123162.0786358, 39683170.209091507, -7376116.8205082472, 875416.22512798384, -64783.936723787068, 2733.2104435030033, -50.333582241124887] ,
            a.miriad.ij2bl(2,7): [21209682.729913376, -23694709.675168406, 11517060.410565007, -3180343.8820138657, 545528.58051407663, -59493.164514871976, 4025.5862253503319, -154.3662372828004, 2.5645648837536448] ,
            a.miriad.ij2bl(0,2): [-316882987.22250527, 385973367.55068403, -205165807.05492854, 62161660.68452727, -11741617.379300762, 1415871.0590410188, -106442.83546191921, 4561.403936059005, -85.310102627258402] ,
            a.miriad.ij2bl(3,7): [140202724.68763375, -169837388.31224203, 89836564.476131603, -27099885.486190148, 5098689.6101386854, -612613.4877893778, 45899.331077184659, -1960.4175025838642, 36.540292270006823] ,
            a.miriad.ij2bl(2,3): [316674342.22960532, -379914552.72872633, 199083994.88502744, -59516191.254261464, 11101607.441614931, -1323037.5370389565, 98372.900830314728, -4172.1169941876424, 77.269404631135288] ,
            a.miriad.ij2bl(0,3): [-574165981.09426689, 694539316.6672101, -366797132.06422991, 110460228.69652407, -20746948.847212963, 2488698.8845048784, -186193.47092168205, 7943.637045901125, -147.96635410693722] ,
            a.miriad.ij2bl(4,7): [30740994.387132924, -35463608.853921279, 17925298.08596639, -5187863.3400075575, 940756.26495586871, -109495.4142534686, 7989.9160711490476, -334.20250118050109, 6.1338632785247658] ,
            a.miriad.ij2bl(2,4): [367170247.60192627, -439740294.68682104, 230058166.33741817, -68670954.937501013, 12791371.663860355, -1522526.2333426217, 113086.12792035965, -4792.0742252187656, 88.697622940190598] ,
            a.miriad.ij2bl(4,5): [216085330.15918624, -259682000.36508548, 136319110.21745592, -40826379.797164828, 7629521.7264758227, -910979.28731902444, 67866.21764962186, -2883.9628883796963, 53.518432508090868] ,
            a.miriad.ij2bl(5,6): [253537325.9746936, -305721940.27161717, 161058205.08503592, -48414727.602764219, 9082581.9935962409, -1088821.25126255, 81450.691526758121, -3475.9867552015835, 64.787641312562371] ,
            a.miriad.ij2bl(2,6): [325100112.80448359, -387434795.53647, 201655523.36677119, -59874092.491589911, 11091890.056437992, -1312836.8174808323, 96951.195812372127, -4084.232534623729, 75.143509921861195] ,
            a.miriad.ij2bl(1,4): [247436590.0802632, -296979006.18987298, 155703643.07270351, -46574987.655231774, 8693417.6757932846, -1036801.4312347455, 77152.421326280441, -3275.0176623232965, 60.712894734629906] ,
            a.miriad.ij2bl(6,7): [-333343844.69829029, 401242086.20069671, -210947708.10856882, 63265803.161510982, -11838518.047250563, 1415314.8981494515, -105568.35305223959, 4491.8339792535508, -83.473094259473996] ,
            a.miriad.ij2bl(3,4): [-131065378.73794083, 161421726.77114159, -86765424.283737689, 26587529.100159075, -5080737.0423088195, 620079.23951927794, -47205.878647848818, 2049.8164680618356, -38.876017967663039] ,
            a.miriad.ij2bl(0,5): [-234320539.56547371, 282000473.80365056, -148140803.83867955, 44367524.18745558, -8285889.5115611237, 988098.01006042131, -73477.727598441037, 3115.3058102077871, -57.659146095486207] ,
            a.miriad.ij2bl(2,5): [327455003.13805789, -392649111.02681702, 205680455.29183909, -61474784.58589635, 11466444.748440571, -1366720.9783922003, 101657.37095904065, -4313.92841140768, 79.961343793072601] ,
            a.miriad.ij2bl(3,5): [262605825.15454268, -312999877.03161752, 163044472.52631101, -48480351.54887633, 8999670.0776158869, -1067988.8454526784, 79114.602751877144, -3344.5727174002473, 61.77168751229221] ,
            a.miriad.ij2bl(0,6): [-383577960.3218025, 466584211.06198978, -247814759.84526321, 75061859.745689929, -14181331.172546795, 1711255.7496460318, -128798.40406160812, 5528.2262917437474, -103.60004275012189] ,
            a.miriad.ij2bl(1,6): [-34754574.289955489, 42075096.424042583, -22231350.561624456, 6696106.2356061228, -1257551.2344184883, 150799.40400946641, -11276.681054831457, 480.84578605557925, -8.9527727868688594] ,
            a.miriad.ij2bl(3,6): [-104722933.17429468, 122200619.00072588, -62149055.679439075, 17993418.934354682, -3243736.2305681072, 372873.89069971372, -26694.080102977765, 1088.3381269398801, -19.352264302225478] ,
            a.miriad.ij2bl(0,7): [-1004667992.9333287, 1202665749.5489261, -628617975.76848447, 187382292.47500902, -34840670.771535136, 4137716.6466669822, -306516.21576277568, 12949.436875333462, -238.87769228424057] ,
        }
        a.fit.AntennaArray.__init__(self, *args, **kwargs)
    def update(self):
        a.fit.AntennaArray.update(self)
        afreqs = self.ants[0].beam.afreqs
        self.blgain = {}
        for bl in self.blpolys:
            self.blgain[bl] = n.polyval(self.blpolys[bl], afreqs)
            self.blgain[bl] = n.clip(self.blgain[bl], 0, n.Inf)
            self.blgain[bl] = n.where(afreqs > .174, 0, self.blgain[bl])
        self.bmgain = {}
        for bl in self.blpolys:
            i,j = self.bl2ij(bl)
            self.bmgain[bl] = self.ants[i].bm_response((0,0,1)) * \
                n.conjugate(self.ants[j].bm_response((0,0,1)))
            self.bmgain[bl] = self.bmgain[bl].squeeze()
    def passband(self, i, j):
        bl = self.ij2bl(i,j)
        gain = self.blgain[bl]
        return a.fit.AntennaArray.passband(self, i, j) * gain
    def bm_response(self, i, j, pol='xx'):
        bl = self.ij2bl(i,j)
        return a.fit.AntennaArray.bm_response(self,i,j,pol=pol)/self.bmgain[bl]

prms = {
    'loc': ('38:25:59.24',  '-79:51:02.1'), # Green Bank, WV
    'antpos':
        [[  -0.029,    0.095,    0.032],
         [ 214.065, -136.130, -260.619],
         [ 197.111, -809.619, -252.626],
         [-254.518, -673.669,  308.515],
         [-284.658, -462.542,  350.485],
         [-277.411, -361.399,  342.699],
         [-174.698, -102.248,  218.335],
         [ -75.025,  -21.154,   97.094]],
    'delays': [ 0.1908,-10.5282,  4.3672,  5.5459,  
                3.8875,  1.5015,  1.6312,  0.0363],
    'offsets': [
        [-0.55258110136188088, 0.055599382493482577, 0.0021762780731818406],
        [0.035082976085547238, -0.013311711075318297, 0.48025783930040622],
        [0.004642352836130393, 0.0083555442407614137, 0.52475856679060495],
        [0.011286076396735688, 0.00088092535537456903, 0.52331469281702336],
        [7.3451453877729411e-07, 0.021834972991227178, 0.53002138023218692],
        [0.27788937605745268, 0.059055756151447356, -0.018919380809884867],
        [0.0040007476621804379, 0.0024474331128108116, 0.49317412039154324],
        [0.20370951329092582, -0.01306497667758439, 0.43683103702332504],
        #[0.002], 
        #[0.475], 
        #[0.536], 
        #[0.527], 
        #[0.539],
        #[0.0], 
        #[0.498], 
        #[0.445],
    ],
    'amps': [1.00,1.00,1.00,1.00,1.00,1.00,1.00,1.00,],
    'bp_r': n.array([
        [1.],
        [1.],
        [1.],
        [1.],
        [1.],
        [1.],
        [1.],
        [1.],
    ]),
    'bp_i': n.array([
        [0.000],
        [0.000],
        [0.000],
        [0.000],
        [0.000],
        [0.000],
        [0.000],
        [0.000],
    ]),
    'beam': a.fit.BeamAlm,
    'bm_prms': {
        'alm7': n.array(
[4611985811.4778929, 0.0, 2660661497.332839, 0.0, 1161684.181439463, 2354124.2922778274, -12784608556.758402, 0.0, 3700154.6975359558, -44051518.998377167, 13123628809.174549, 386099.55688131653, -2382423132.7169805, 0.0, -1081545.505707864, -29373602.722160414, 14777949279.781567, -1683401.8125857676, 813116.6043629318, -3812651.7148833866, 3466759137.199594, 0.0, -2563552.9410186801, -23139262.802033395, 25392282.092893571, -1965503.0138684819, 3513117.4756964981, -11570077.243654389, 2161050343.2314053, -439571.41550456954, -4952215127.7080574, 0.0, -1032405.0793414298, 3328758.6370827751, 8081580548.5924425, -1652705.1370973999, -1865385.6858704605, 9460516.1789629385, -9003744061.3970089, -440683.83033249516, 5995659.440223611, -2758772.4421860543, -2242539710.1916742, 0.0, 35822.753338495328, -3834306.8410776504, -4154425040.89606, -405331.8394943634, -3127073.5293740062, 23566426.32039104, -3040268986.2301226, 1867354.8252214531, 8301007.3877734169, 6375031.9025108283, 22538293904.08601, 1727887.019201526, -9333187864.2987041, 0.0, 4515559.1980632711, -15330723.299259013, 697579270.46958423, 759404.9932369173, -1597884.4653839464, 6695441.3151492076, 7530092067.1936522, -1695306.2769341765, 6876785.4138246849, -30016400.97517173, 10838930493.839764, -1368664.0340590996, -2211484.3020582329, 16363887.764421593, 2253711333.906599, 0.0, -844126.07871851511, -6318545.7948333658, -4775781341.9252472, 2105897.9371543489, 1048796.0530797702, 392150.03348757443, -840122112.02322769, 315230.49375097384, 3070199.7155984561, 2977901.844212072, 6091190729.7350578, -4849392.4916701755, -2260932.7544293264, 2056849.1002333786, 6261257109.1774731, 1524900.7628346651, -1518768907.0558486, 0.0, 2777964.1198691949, -5894786.967444811, -1397124380.5216975, 2363993.9766160166, 553336.47123466979, 3623216.0804315004, 4821993268.3951569, -2004604.0840042909, -1967868.0351001341, -7636543.1123500178, -779372152.42068744, -4978300.5312785935, 226828.929573167, 8294479.7546154037, 1656859641.6065073, -957601.87508812803, -1809185.4608415843, 586765.25698671234, -1306300160.8376188, 0.0, 871434.78962225269, -6327850.6900228104, -476919963.69395149, 1908544.0218539392, -136993.4880822585, 6463209.6213090634, -3029889922.7930927, 914449.27866353898, 2651122.7357223816, 4356520.1409043754, -397566752.74245155, -1862812.8776872838, 58107.484234671203, -5529946.7219171897, -1965417175.1103096, -128243.00152259972, 385592.92964855517, -457392.97022164427, 2382869174.9230151, 1962485.6379324049]
        ),
        'alm6': n.array(
[-4660440817.0325651, 0.0, -2524747587.4077301, 0.0, -1288015.7639206834, -5083648.5964583084, 13074309800.088018, 0.0, -3843713.3158637634, 41838377.653908908, -13287607320.476055, -374868.88982848369, 1779821188.3149352, 0.0, 1232773.9929147866, 27721902.415636718, -14769741202.472086, 1709261.8700547724, -829771.12014169095, 3883615.4413207932, -3934662156.1555352, 0.0, 2666092.6112787239, 22388788.488832884, -383328081.57203722, 2041865.7567146884, -3614944.5873596817, 11906132.317594584, -1895952941.7102704, 414279.04213376844, 4805870491.5618668, 0.0, 1115285.9439498219, -4098796.6816552044, -8454896818.3043442, 1682089.1923746828, 1903357.853637039, -9210270.1371825766, 9429541112.4519978, 371519.7054062094, -6171832.2490067836, 2554572.6181881721, 2222749756.2007051, 0.0, -28053.668632029556, 3357940.5541918227, 4182209694.536706, 426939.88750392734, 3051510.7954315976, -23576800.628423113, 3029948523.4987431, -1930316.432950052, -8409425.0000792574, -6380242.396455341, -23024522988.440693, -1740348.2991658312, 9509357659.5945759, 0.0, -4589056.5253075631, 15093202.934578849, -809461276.77322662, -765856.55479350523, 1610420.9181540865, -6578128.7256362671, -7591710922.1586523, 1641121.3909650473, -6962827.6394143943, 29803861.949707672, -10923040033.505558, 1380437.9700561177, 2200337.4183063959, -16880008.694912128, -2330867318.0235648, 0.0, 901146.30122510844, 6271580.9854984349, 4870527043.5510321, -2169333.9574615471, -1112710.0309672123, -273381.09217741672, 889545314.97025847, -360677.59358309768, -3110081.7978469818, -2888238.2134599043, -6225424428.5081224, 4998069.2806683043, 2238431.2425951185, -1514077.2264344564, -6450158447.4513016, -1544227.2343291694, 1589533537.2414124, 0.0, -2849590.4534891658, 5848957.5973186651, 1402350557.9784439, -2406793.2363643702, -559747.91029137094, -3607592.8341452475, -4946405456.8122149, 2000916.1250513815, 2003625.729428973, 7624001.8443190083, 814103021.2593931, 5042637.6743036993, -232550.16672018601, -8458045.838147644, -1552441513.6501589, 934808.89786321076, 1795219.7913953224, -543780.33693347056, 1300125197.6853461, 0.0, -878693.44198185124, 6431605.6063784314, 483106317.2754392, -1968299.9382686331, 160668.10415371601, -6589997.6366426758, 3147847116.4654188, -941792.93136318831, -2712305.3653369793, -4389774.5453290576, 384064278.78552872, 2008052.7921253969, -89734.627356168479, 5535411.1674011359, 1957276706.0453017, 110841.15266485221, -392364.61775274243, 439644.38718298491, -2446836950.0839534, -1981196.0119920783]
        ),
        'alm5': n.array(
[2000576951.6567385, 0.0, 1018174986.0901886, 0.0, 605918.62563263508, 3319308.8354331963, -5693750244.0869961, 0.0, 1701785.1047911406, -16920963.61056741, 5738155188.6359119, 154696.68328332825, -484673578.27879608, 0.0, -593002.57655283797, -11140100.114677167, 6290801715.422534, -740213.37482546736, 360431.78030755633, -1686714.2477207626, 1888161835.7256896, 0.0, -1180765.7620479362, -9245590.4610685185, 325361076.80476499, -904843.01082618581, 1585381.8520197107, -5224123.7543742713, 691177098.11475503, -165571.94371282117, -1981017353.6406167, 0.0, -511275.7999708248, 2069264.9790669759, 3770796235.1607347, -729874.24819612456, -828494.92326426215, 3827088.3107252019, -4207348299.9590492, -127671.14451857278, 2709287.8695738222, -1009545.7622988597, -937548169.50474024, 0.0, 9136.2731055648474, -1233437.1189854685, -1793968474.7897377, -191928.7399844288, -1267547.424492032, 10061056.084055521, -1284540587.5502262, 850754.45539897738, 3633323.5803311523, 2721443.8966966923, 10028797384.452114, 747581.3455337662, -4132464691.644228, 0.0, 1989349.2677438257, -6340081.2849623375, 394970615.25654101, 329507.87014517322, -693089.34879322199, 2759415.0593424304, 3263320431.1873245, -675519.54737822909, 3007153.9330359772, -12623122.164965319, 4692507938.8226728, -593500.16442786623, -932786.96099904994, 7415015.6333928639, 1027371073.6929944, 0.0, -409764.21407068346, -2656019.55938295, -2118602114.3743339, 953562.51298512227, 503027.47333889524, 63802.912034502646, -399863331.93641698, 173461.17821895628, 1343504.7557573256, 1190852.886973389, 2714208171.0180831, -2197599.7361137979, -944919.79479770293, 417635.36423222936, 2833733074.3894854, 667018.45160380984, -709529720.5740186, 0.0, 1246714.4759992538, -2475784.6897057607, -599844678.84892166, 1045086.7639113223, 241234.78760753473, 1534204.6565376476, 2164022007.9163022, -851603.27049885306, -870138.95390509232, -3246266.3960603969, -363192479.61007524, -2177658.5394275784, 101893.84743725008, 3675977.0003051665, 616517316.57388175, -388351.96002852329, -759455.94632553787, 215155.92785969403, -552130042.82784796, 0.0, 377690.67358282424, -2787084.9135859003, -208625206.42402333, 866233.73855700321, -78939.020526992332, 2862865.9835425392, -1394231913.8516233, 413630.97968383727, 1183778.3556235847, 1886999.8081495594, -157225245.31542793, -921815.00277177931, 51980.188969233976, -2362431.8939862116, -830876721.52216899, -39762.241847119731, 170458.99673859176, -179392.75231958073, 1071364323.0192623, 852977.1775799687]
        ),
        'alm4': n.array(
[-472696679.51073062, 0.0, -226261844.58283606, 0.0, -156819.89921196242, -1050747.1308088675, 1368492651.0813763, 0.0, -416205.59526278509, 3777527.0514986617, -1369963215.8704579, -35160.508360957785, 45555360.161302917, 0.0, 156295.85175899032, 2470935.6390057974, -1480107081.7698109, 177218.27933337886, -86388.587333996344, 404905.17258998874, -497091658.38848025, 0.0, 288648.15704091132, 2112663.0629795962, -116966484.8785509, 221684.7080159192, -384115.46634214581, 1266873.9778075432, -134318111.40297198, 36354.537441580811, 449420506.91056859, 0.0, 128971.61075013063, -561866.05314258719, -929178294.66281772, 175006.31877267605, 199383.70908056918, -879960.45131450146, 1036616243.415095, 22564.302107389492, -657368.5084218384, 220788.52768132801, 218052033.71763009, 0.0, -1627.5529223806225, 245188.60936947438, 425021490.25894392, 47720.913402910883, 290513.63452360401, -2374072.7344934396, 300239213.95300317, -207200.42696461914, -867772.86466036004, -641472.75381827517, -2414101572.7586575, -177529.54899106993, 992808937.03916669, 0.0, -476836.36140141869, 1473105.099849456, -105302648.29655592, -78397.999300193827, 165124.27523276984, -640799.39437259815, -775220331.10810554, 153232.50653722594, -718073.42385880044, 2956351.7322162841, -1113895752.0227387, 140988.25895482666, 218398.7779983576, -1798135.3846395621, -250187320.44826138, 0.0, 102849.38694731318, 622170.73167650751, 509486780.99315375, -231815.01202725468, -125587.9820727725, -2146.6474220444316, 98918126.331533894, -45565.769504060387, -320809.40960413183, -270520.13905222621, -654315387.61157548, 534279.69654830417, 220467.94851762458, -45683.080787360894, -688130942.421422, -159291.91584945272, 175066570.59439647, 0.0, -301547.09544404794, 579579.27069143381, 141729665.71351552, -250873.20918539472, -57403.934470674867, -361242.86151522771, -523345807.54437077, 200334.37502035761, 208924.35693491568, 764182.25156948378, 89674127.531611606, 519695.12402868905, -24723.10236118584, -882648.00083006057, -134428925.91831422, 88990.420766547657, 177559.05981595005, -47132.544779793549, 129686610.19492728, 0.0, -89707.546301326336, 667475.26727657777, 49776700.265661567, -210843.73217589484, 21128.666557236858, -686948.17998139351, 341205780.45880115, -100410.69882264256, -285681.55866082152, -448545.47213855234, 35325865.376718849, 233549.78162989195, -15524.662163431036, 557235.82820948947, 194888462.99005207, 7591.9011104529054, -40980.054387986936, 40223.606903436368, -259267147.62744895, -203009.0552224837]
        ),
        'alm3': n.array(
[66365304.895441972, 0.0, 29935552.939574994, 0.0, 24121.876244309857, 185183.79060861631, -196017613.015596, 0.0, 60717.782346892192, -502721.20913747977, 195279202.98641238, 4751.9672938265476, 3804566.8235258134, 0.0, -24404.93776504814, -326701.99589534744, 207751565.71544495, -25331.734895248519, 12339.617490534891, -58022.219908578358, 77594683.796933234, 0.0, -42062.35847573333, -288535.06510947237, 22406650.701983605, -32423.418369334056, 55524.770544749037, -183370.66726457846, 14753100.086506855, -4733.1415556255888, -60575405.745413519, 0.0, -19340.78235965782, 89263.160874093955, 136603594.11628419, -25041.235576498224, -28645.06211227148, 120927.22613531096, -152295320.88408348, -2079.602011700254, 95202.548449885609, -28867.030143253116, -30188773.536950249, 0.0, 177.26593152105761, -28281.91587649774, -60060797.70541326, -7085.4522519514412, -39676.557531085578, 334565.13682400895, -41778147.424835734, 30112.537776100351, 123725.85424156538, 90261.460161399926, 346814490.79463738, 25170.316185991407, -142400287.71425736, 0.0, 68251.245615570602, -204478.06686854284, 16594946.50769872, 11139.457378592722, -23519.361240393569, 88987.309390729788, 109907526.69723321, -20683.485392529587, 102381.81813651015, -413501.31984255678, 157784904.01132619, -19985.237109254296, -30500.590413289385, 259987.77499878613, 36356547.80741217, 0.0, -15385.724765428571, -87068.683435915358, -73150554.994824663, 33655.666541789149, 18698.16179801671, -1568.9640831565591, -14544327.117713498, 7073.2202901811706, 45729.457537876537, 36559.609375034866, 94185586.7554387, -77557.84007077133, -30706.525962777094, -836.14946592545039, 99741213.910499081, 22712.927813673199, -25781073.659567423, 0.0, 43545.749609007151, -81047.305932737887, -19977273.311153434, 35953.241001932933, 8143.9641628306799, 50860.982255357434, 75551068.617381766, -28132.900401197432, -29951.328326805997, -107409.5072268184, -13229165.520450925, -74017.556484796412, 3586.0469340095447, 126453.16053511089, 17368405.757738788, -12146.757199880918, -24778.540952883515, 6175.9473687333902, -18194639.226311762, 0.0, 12716.037771885833, -95408.533833514884, -7085536.7467317162, 30648.912879294974, -3335.0803767408706, 98334.748898240126, -49820494.439908288, 14548.704266338442, 41167.307195090107, 63671.943059505153, -4700031.656345739, -35262.07645547537, 2648.4621152716381, -78451.774535216551, -27277615.67601084, -817.1860164069567, 5887.3594324257774, -5343.9542476281931, 37446409.25380192, 28844.737724172755]
        ),
        'alm2': n.array(
[-5534011.8015798181, 0.0, -2359432.8883475899, 0.0, -2205.7032917965171, -18656.915209584891, 16729669.443972664, 0.0, -5283.0118220309514, 39880.026770071105, -16618433.991254264, -381.76308702168882, -1221449.7176860846, 0.0, 2259.4301440481722, 25747.489901858502, -17396314.909020782, 2161.7758419850061, -1050.500935269064, 4963.1673181912101, -7185790.2325648097, 0.0, 3653.6687309462127, 23555.463835584389, -2403393.7286230233, 2830.5123496668634, -4788.4647412267095, 15841.26919192817, -881776.17483379808, 365.11817736313048, 4848722.3979713721, 0.0, 1724.8009168402734, -8336.6446198490885, -11980211.178861016, 2138.1116234842762, 2456.2978438598693, -9933.0576071072392, 13340339.956709825, 79.994813592674376, -8228.6934185026785, 2256.8626775597049, 2487162.6297275615, 0.0, -12.545914565040562, 1870.2140047666539, 5062139.5340395924, 628.03798088007068, 3228.9869165063255, -28159.386524220379, 3460520.7474105731, -2611.0626714777095, -10530.288406717602, -7583.4628450089604, -29733490.498504147, -2130.6576159181773, 12193285.382816285, 0.0, -5833.386303830519, 16957.28063511377, -1547792.9765405189, -945.19624889672502, 2002.6756486685813, -7391.1251874060436, -9299157.960524248, 1661.0789769068747, -8715.5968754359274, 34541.703546486031, -13337007.467215525, 1690.4250979081953, 2540.7372654811738, -22415.013253589001, -3152813.9886766411, 0.0, 1371.5241380186339, 7279.7141561471326, 6270081.3690575873, -2917.6284210562258, -1659.8699777861277, 293.18996337762997, 1271137.2461998067, -649.71102858650329, -3891.0451468076108, -2938.931008772206, -8094631.2483585682, 6721.425836522154, 2553.1361694925363, 665.63579692502776, -8628205.0821410213, -1933.5248062114506, 2265610.421221653, 0.0, -3754.1946648550997, 6770.5447687286942, 1679847.526713683, -3075.9611318107454, -688.74272812172455, -4281.8205741770171, -6509814.7958796294, 2358.4752964492282, 2563.5705549162417, 9014.1474164304091, 1165649.1907468438, 6291.2235560564804, -310.82464811214334, -10809.263109196316, -1328535.5130432413, 987.53420702343078, 2064.0923943160942, -484.23397723288099, 1524599.908664112, 0.0, -1075.8019401111007, 8139.5790083668044, 601665.49089752848, -2660.2582070557646, 311.00153238926453, -8398.1256482474873, 4339551.3122292235, -1258.0349520468158, -3541.9168574223513, -5397.3458115035601, 369848.3808616256, 3172.005785004721, -261.56146908417941, 6592.4213933094888, 2278181.5710720317, 47.5922564439079, -505.37715424199826, 419.93292205764101, -3227613.1062257914, -2446.6980108345288]
        ),
        'alm1': n.array(
[253662.04140785141, 0.0, 102657.72228924923, 0.0, 111.04033644719563, 1008.7846910489577, -787659.75447146501, 0.0, 253.83721031709877, -1745.9466152081013, 781746.19866752857, 16.876040613167589, 100485.42691992909, 0.0, -114.90509185181016, -1119.8232597886563, 804670.24177662027, -101.97960345332903, 49.357956582494552, -234.64557985902357, 365729.36890787206, 0.0, -175.16570299363616, -1064.4692007773654, 136675.4985124871, -136.53931780133161, 228.11956195951458, -756.28796586768124, 23986.029472398684, -15.43785897286358, -213314.92245670213, 0.0, -84.721278934776166, 425.19310602843422, 580278.31001027778, -100.86521389144443, -116.3818572861553, 451.58170782863988, -645059.11295880133, 0.78758260800782942, 393.00505208224229, -97.718213340383684, -112869.98188980426, 0.0, 0.57031575493051889, -64.184312196168946, -235615.97017357495, -30.761704307079388, -144.99389320627228, 1310.7371416012445, -157972.55413632019, 125.06354225234149, 495.34099707214529, 352.31593235625587, 1408511.1968934615, 99.708519716686766, -577106.5538782801, 0.0, 275.65204832896461, -777.96197277325371, 79183.845039763517, 44.344394731179776, -94.384617117138092, 340.02232859379052, 434748.47675435641, -73.480658327567994, 410.16027830938464, -1595.6922660702598, 622875.10375922977, -79.000367906847956, -116.89532087567019, 1067.0857478010957, 151077.51254187227, 0.0, -67.449308156213661, -336.71871228661206, -297062.53838645853, 139.81878422654444, 81.337051421797781, -21.279088280382375, -61150.351873002095, 32.733654194884437, 182.98620245008064, 130.09601954414134, 384562.063220846, -321.95654584569508, -117.35047072397772, -57.812536152039158, 412425.46984646772, 90.98663317195188, -109993.56439037478, 0.0, 178.90578327834703, -312.87713913401274, -78029.353924545809, 145.46011241885145, 32.147267409860376, 199.56719820666726, 309961.40884985449, -109.2968302747814, -121.29128522457248, -418.24163116045236, -56779.003676337488, -295.47712387141928, 14.901579497823271, 510.46388340493525, 55647.981840776425, -44.277645758895481, -95.042993028736674, 21.043026652158588, -70644.253436452549, 0.0, 50.301996412430853, -383.75748887164804, -28217.128723545535, 127.64174985779439, -15.886263830606921, 396.24196473415952, -208760.61817931911, 60.10522284219131, 168.45215580203552, 252.96919293605984, -15914.71370523221, -157.37836544319813, 13.957185886014937, -306.15495946132847, -105125.58673489391, -1.2370918443699921, 23.99869089302328, -18.035038122563744, 153709.50900810928, 114.71789293186426]
        ),
        'alm0': n.array(
[-4925.7046238955681, 0.0, -1902.0375713250537, 0.0, -2.3745766844618617, -22.768024803072443, 15781.267354535015, 0.0, -5.1954037214577546, 32.538917685207167, -15680.523310490957, -0.31658944374066367, -2889.8255199254272, 0.0, 2.4773875059343107, 20.73203551417425, -15861.527480135486, 2.0514836186756664, -0.98745063826312252, 4.7297280454843547, -7895.3397818852472, 0.0, 3.5756477287113841, 20.542915708895677, -3217.3114851305872, 2.8071597665404453, -4.6311277937582682, 15.392961483030195, -133.6137203687999, 0.27562908963479466, 3977.0421883781419, 0.0, 1.7686389628848049, -9.1574367247491075, -11974.092800608571, 2.0280799534192475, 2.350198876926282, -8.7657871765960191, 13281.799054198817, -0.10646305267238632, -8.0006225312864103, 1.8081851100590824, 2175.867251134619, 0.0, -0.012998676146555144, 0.83887899439501346, 4671.9814535312789, 0.64211658600560295, 2.7713910386759699, -26.029719176206953, 3065.752153366444, -2.5525627775100781, -9.9343632007206004, -6.9832734519677793, -28439.935388950271, -1.9900254038995073, 11646.699383863624, 0.0, -5.5553806273656123, 15.232345262306115, -1715.4260220982721, -0.88735984714614191, 1.8990584233581722, -6.6843698501161191, -8663.7521848507076, 1.3809527234207628, -8.2315112012670468, 31.448956931442353, -12399.356213275894, 1.5737268335875474, 2.2916359505148161, -21.640969799270053, -3085.9284853166469, 0.0, 1.4115551288102317, 6.6471982198184598, 6000.8509835635987, -2.8569142059755319, -1.6970205548610837, 0.57602673963767281, 1249.3449408804927, -0.69863812326084684, -3.668910402843049, -2.4458985183219131, -7790.3726344852275, 6.5747577935938386, 2.3003784922607782, 1.6619535862540191, -8402.7138291012852, -1.8257404508197315, 2275.5677574324391, 0.0, -3.6353469639007123, 6.1703582699524802, 1544.6550946906325, -2.9330294760417273, -0.63883166405210456, -3.9722745643864341, -6291.0115285828288, 2.159978449309512, 2.4470510740252136, 8.2767275066780908, 1179.0642658555153, 5.9157043889713989, -0.30475328076004449, -10.273991209262309, -983.30528079683938, 0.84456827791361355, 1.8663561083942408, -0.3911210028840445, 1396.2914945122375, 0.0, -1.0028855404333623, 7.7135412266520005, 563.78897010968626, -2.6112802563639699, 0.34329750666156889, -7.9684218372149456, 4278.3154990848925, -1.2239243073622779, -3.4160724547571513, -5.0570662223631349, 288.34553169111047, 3.3217067738741073, -0.31203485803049957, 6.0616687375033811, 2067.6122558988372, 0.0052391271544491858, -0.48628047077818243, 0.32574162122794237, -3119.8458593128425, -2.2936162817706971]
        ),
    }
}

def get_aa(freqs):
    '''Return the AntennaArray to be used fro simulation.'''
    location = prms['loc']
    antennas = []
    nants = len(prms['antpos'])
    assert(len(prms['delays']) == nants and len(prms['amps']) == nants \
        and len(prms['bp_r']) == nants and len(prms['bp_i']) == nants)
    for i in range(len(prms['antpos'])):
        beam = prms['beam'](freqs, nside=128, lmax=10, mmax=10, deg=7)
        try: beam.set_params(prms['bm_prms'])
        except(AttributeError): pass
        pos = prms['antpos'][i]
        dly = prms['delays'][i]
        amp = prms['amps'][i]
        bp_r = prms['bp_r'][i]
        bp_i = prms['bp_i'][i]
        off = prms['offsets'][i]
        antennas.append(
            a.fit.Antenna(pos[0],pos[1],pos[2], beam, delay=dly, offset=off,
                amp=amp, bp_r=bp_r, bp_i=bp_i)
        )
    #aa = a.fit.AntennaArray(prms['loc'], antennas)
    aa = AntennaArray(prms['loc'], antennas)
    return aa

src_prms = {
        'cyg': { 'str': 10760, 'index':-0.15, },
        'cas': { 'str': 12000, 'index':-0.47, 'angsize':0.000614, },
        'Sun': { 'str': 50670, 'index':1.67, 'angsize':0.00517, },
        'crab': { 'str':1450, 'index':-0.34, 'angsize':0.00118, },
        'vir': { 'str':1650, 'index':-1.73, 'angsize':0.00146, },
        '3c392': { 'str':750, 'index':-3.91, 'angsize':0.00498, },
        '3c409': { 'str':170, 'index':-2.24, 'angsize':0.00277, },
        'her': { 'str':640, 'index':-2.13, 'angsize':0.00210, },
        '3c295': { 'str': 100, 'index': -0.18, 'angsize': 0.000472, },
        '3c196': { 'str': 100, 'index': -1.31, 'angsize': 0.000545, },
        '3c123': { 'str': 260, 'index': -1.00, 'angsize': 0.00137, },
        '3c010': { 'str': 240, 'index': -0.22, 'angsize': 0.00192, },
        '3c157': { 'str': 280, 'index': -2.85, 'angsize': 0.00665, },
        '3c134': { 'str': 90, 'index': -1.64, 'angsize': 0.00173, },
        '3c111': { 'str': 80, 'index': -0.24, 'angsize': 0.00062, },
        '3c380': { 'str': 90, 'index': -2.03, 'angsize': 0.00153, },
        '3c310': { 'str': 80, 'index': -1.52, 'angsize': 0.00028, },
        '3c433': { 'str': 70, 'index': -2.24, 'angsize': 0.00027, },
}
