#! /usr/bin/env python

"""
Create _version.py, based upon the latest darcs release tag.

If your source tree is coming from darcs (i.e. it is in a darcs repository),
this tool will determine the most recent release tag, count the patches that
have been applied since then, and compute a version number to be written into
_version.py . This version number will be available by doing:

 from your_package_name import __version__

Source trees that do not come from darcs (e.g. release tarballs, nightly
tarballs) and are not within a darcs repository should instead, come with a
_version.py that was generated before the tarball was produced. In this case,
this script will quietly exit without modifying the existing _version.py .

'release tags' are tags in the source repository that match the following
regexp:

 ^your_package_name-\d+\.\d+(\.\d+)?((a|b|c)(\d+)?)?\w*$

"""

import os, string, sys, re
import xml.dom.minidom
from subprocess import Popen, PIPE

try:
    EXE_NAME=os.path.basename(sys.argv[0])
except:
    EXE_NAME="darcsver"

try:
    # If we can import pyutil.version_class then use its regex.
    from pyutil import version_class
    VERSION_BASE_RE_STR = version_class.VERSION_BASE_RE_STR
except ImportError:
    # Else (perhaps a bootstrapping problem),then we'll use this
    # regex, which was copied from the pyutil source code on
    # 2007-10-30.
    VERSION_BASE_RE_STR="(\d+)(\.(\d+)(\.(\d+))?)?((a|b|c)(\d+))?"

def get_text(nodelist):
    rc = ""
    for node in nodelist:
        if node.nodeType == node.TEXT_NODE:
            rc = rc + node.data
    return rc

VERSION_BODY = '''
try:
    from pyutil.version_class import Version as pyutil_Version
    Version = pyutil_Version
except ImportError:
    from distutils.version import LooseVersion as distutils_Version
    Version = distutils_Version

# This is the version of this tree, as created by %s (from the pyutil
# library) from the Darcs patch information: the main version number is taken
# from the most recent release tag. If some patches have been added since the
# last release, this will have a -NN "build number" suffix. Please see
# pyutil.version_class for a description of what the different fields mean.

verstr = "%%s"
__version__ = Version(verstr)
''' % (EXE_NAME,)

def write_version_py(verstr, outfname):
    f = open(outfname, "wt+")
    f.write(VERSION_BODY % (verstr,))
    f.close()

def update(pkgname, verfilename, quiet=False):
    rc = -1
    cmd = ["darcs", "changes", "--from-tag=^%s" % (pkgname,), "--xml-output"]
    try:
        p = Popen(cmd, stdout=PIPE, universal_newlines=True)
    except:
        pass
    else:
        output = p.communicate()[0]
        rc = p.returncode
    if rc != 0:
        cmd = ["realdarcs.exe", "changes", "--from-tag=^%s" % (pkgname,), "--xml-output"]
        p = Popen(cmd, stdout=PIPE, universal_newlines=True)
        output = p.communicate()[0]
        rc = p.returncode
        if rc != 0:
            if os.path.exists(verfilename):
                if not quiet:
                    print "%s: Failure from attempt to find version tags with 'darcs changes', and %s already exists, so leaving it alone." % (EXE_NAME, verfilename,)
                return 0
            else:
                if not quiet:
                    print "%s: Failure from attempt to find version tags with 'darcs changes', and %s doesn't exist." % (EXE_NAME, verfilename,)
                return rc

    # Filter out bad chars that can cause the XML parser to give up in despair.
    # (Thanks to lelit of the tailor project for this hack.)
    allbadchars = "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0B\x0C\x0E\x0F\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1A\x1B\x1C\x1D\x1E\x1F\x7f"
    tt = string.maketrans(allbadchars, "?"*len(allbadchars))
    output = output.translate(tt)

    doc = xml.dom.minidom.parseString(output)
    changelog = doc.getElementsByTagName("changelog")[0]
    patches = changelog.getElementsByTagName("patch")
    count = 0
    regexstr = "^TAG %s-(%s)" % (pkgname, VERSION_BASE_RE_STR,)
    version_re = re.compile(regexstr)
    for patch in patches:
        name = get_text(patch.getElementsByTagName("name")[0].childNodes)
        m = version_re.match(name)
        if m:
            last_tag = m.group(1)
            last_tag = last_tag.encode("ascii")
            break
        count += 1
    else:
        if not quiet:
            print "%s: I'm unable to find a tag in the darcs history matching \"%s\", so I'm leaving %s alone." % (EXE_NAME, regexstr, verfilename,)
        return 0

    if count:
        # this is an interim version
        verstr = "%s-%d" % (last_tag, count)
    else:
        # this is a release
        verstr = last_tag

    write_version_py(verstr, verfilename)
    if not quiet:
        print "%s: wrote '%s' into %s" % (EXE_NAME, verstr, verfilename,)
    return 0

def main():
    quiet = False
    if "--quiet" in sys.argv:
        quiet = True
        sys.argv.remove("--quiet")
    if "-q" in sys.argv:
        quiet = True
        sys.argv.remove("-q")
        
    if len(sys.argv) >= 2:
        pkgname = sys.argv[1]
    else:
        pkgname = os.path.basename(os.getcwd())
        if not quiet:
            print "%s: You didn't pass a pkg-name on the command-line, so I'm going to take the name of the current working directory: \"%s\"" % (EXE_NAME, pkgname,)

    if len(sys.argv) >= 3:
        verfilename = sys.argv[2]
    else:
        verfilename = os.path.join(pkgname, "_version.py")
        if not quiet:
            print "%s: You didn't pass a verfilename on the command-line, so I'm going to build one from the name of the package: \"%s\"" % (EXE_NAME, verfilename,)

    rc = update(pkgname=pkgname, verfilename=verfilename, quiet=quiet)
    return rc

if __name__ == "__main__":
    rc = main()
    sys.exit(rc)
