'''
Created on May 23, 2012

@package: ally core
@copyright: 2012 Sourcefabric o.p.s.
@license: http://www.gnu.org/licenses/gpl-3.0.txt
@author: Gabriel Nistor

Provides basic meta decode implementations. 
'''

from ally.support.util_sys import locationStack
from collections import deque, Iterable
from inspect import isfunction, ismethod
import abc

# --------------------------------------------------------------------

class ResolveError(Exception):
    '''
    Exception raise when there is a problem with resolving.
    '''

class IResolve(metaclass=abc.ABCMeta):
    '''
    Specification for exploits resolving.
    '''
    __slots__ = ()

    @abc.abstractclassmethod
    def request(self, **data):
        '''
        Request a resolve with the contained exploits the data. The resolve is done asynchronous, in order to process it 
        you need to make the do call.
        
        @param data: key arguments
            The data to resolve with.
        @return: IResolve
            This resolve for chaining purposes.
        '''

    @abc.abstractclassmethod
    def queue(self, exploit, **data):
        '''
        Push a raw exploit in requests queue.
        
        @param expoit: callable(**data) -> boolean
            The exploit attached to the locator.
            @param data: key arguments
                The data used for processing the exploit.
            @return: boolean|None
                True or None for process success, False otherwise.
        @param data: key arguments
            The data to resolve the exploit with.
        @return: IResolve
            This resolve for chaining purposes.
        '''

    @abc.abstractclassmethod
    def queueBatch(self, exploit, data):
        '''
        Makes a batch of requests with the same exploit using data provided by the data iterable.
        
        @param expoit: callable(**data) -> boolean
            The exploit attached to the locator.
            @param data: key arguments
                The data used for processing the exploit.
            @return: boolean
                True or None for process success, False otherwise.
        @param data: Iterable(dictionary{string, object})
            The data to resolve the path with.
        @return: IResolve
            This resolve for chaining purposes.
        '''

    @abc.abstractclassmethod
    def has(self):
        '''
        Checks if there are any more request to be resolved.
        
        @return: boolean
            True if there are requests queued, False otherwise.
        '''

    @abc.abstractclassmethod
    def do(self):
        '''
        Proceeds with the resolve requests.
        
        @return: boolean
            True if the current resolve request has been finalized successful, False otherwise.
        '''

    @abc.abstractclassmethod
    def doAll(self):
        '''
        Process all requests. Only call this method if there are requests queued.
        
        @return: boolean
            True if all the requests have been finalized successfully, False otherwise.
        '''

# --------------------------------------------------------------------

class Resolve(IResolve):
    '''
    The default resolve implementation.
    '''
    __slots__ = ('root', 'requests', 'counts', 'count')

    def __init__(self, root):
        '''
        Construct the resolve for the provided exploit.
        
        @param root: callable(**data) -> boolean
            The root exploit of the resolve.
        '''
        assert callable(root), 'Invalid root exploit %s' % root

        self.root = root
        self.requests = deque()
        self.counts = deque()
        self.count = 0

    def request(self, **data):
        '''
        @see: IResolve.request
        '''
        return self.queue(self.root, **data)

    def queue(self, exploit, **data):
        '''
        @see: IResolve.queue
        '''
        assert callable(exploit), 'Invalid exploit call %s' % exploit

        self.requests.append((exploit, data, None))
        self.count += 1
        return self

    def queueBatch(self, exploit, data):
        '''
        @see: IResolve.queueBatch
        '''
        assert callable(exploit), 'Invalid exploit call %s' % exploit
        assert isinstance(data, Iterable), 'Invalid data %s' % data

        batch = iter(data)
        try: data = next(batch)
        except StopIteration: return False
        assert isinstance(data, dict), 'Invalid batch data %s' % data

        self.requests.append((exploit, data, batch))
        self.count += 1

        return self

    def has(self):
        '''
        @see: IResolve.has
        '''
        return self.count > 0

    def do(self):
        '''
        @see: IResolve.do
        '''
        if self.count == 0: return False

        exploit, data, batch = self.requests.popleft()
        assert callable(exploit), 'Invalid exploit call %s' % exploit

        if batch is None: self.count -= 1
        else:
            try: nextData = next(batch)
            except StopIteration: self.count -= 1
            else:
                assert isinstance(nextData, dict), 'Invalid next batch data %s' % nextData
                self.requests.appendleft((exploit, nextData, batch))

        self.counts.appendleft(self.count)
        self.count = 0

        data.update(resolve=self)  # Set the current resolve
        try: exploit(**data)
        except:
            self.requests.clear()
            self.counts.clear()
            self.count = 0

            if isfunction(exploit) or ismethod(exploit):
                cd = exploit.__code__
                raise ResolveError('Problems with exploit at:\n  File "%s", line %i, in %s' % 
                                   (cd.co_filename, cd.co_firstlineno, exploit.__name__))
            else: raise ResolveError('Problems with exploit call %s' % exploit)

        count = self.counts.popleft()
        if self.count > 0 and count > 0: self.requests.rotate(self.count)
        self.count += count
        return True

    def doAll(self):
        '''
        @see: IResolve.doAll
        '''
        valid = False
        while self.has():
            if self.do(): valid = True
            else: return False

        return valid

# --------------------------------------------------------------------

def handleExploitError(exploit):
    '''
    Handles the error for the provided exploit.
    '''
    if isfunction(exploit) or ismethod(exploit):
        raise ResolveError('Problems with exploit at:%s' % locationStack(exploit))
    else: raise ResolveError('Problems with exploit call %s' % exploit)
