"""Run workflow for Python-based projects"""
from re import sub
from subprocess import call, check_output
import os
import urllib

from almost_empty.all import meta
from almost_empty.helper import get_random_string, get_readable_name, modify_file, store_gitignore


def doc(readable_name, name, directory, additional=None):
	"""Set up documentation

	Enable auto-generation of Sphinx HTML docs.
	"""
	call(['pip', 'install', 'sphinx'])

	call(['sphinx-apidoc', '-F', '-o', 'docs', name], cwd=directory)
	docs_directory = os.path.join(directory, 'docs')
	os.remove(os.path.join(docs_directory, 'conf.py'))
	modify_file('conf.py', docs_directory, ['# -*- coding: utf-8 -*-', '# -- General configuration ------------------------------------------------', 'extensions = [', '    \'sphinx.ext.autodoc\',', '    \'sphinx.ext.viewcode\',', ']', '', 'master_doc = \'index\'', 'source_suffix = \'.rst\'', 'pygments_style = \'sphinx\'', '', 'exclude_patterns = [\'_build\']', '', 'project = u\'' + name + '\'', 'copyright = u\'2014, @TheKevJames (auto-generated)\'', 'show_authors = True', '', 'version = \'0\'', 'release = \'0\'', '', 'add_module_names = False', '', '# -- Options for HTML output ----------------------------------------------', 'html_theme = \'agogo\'', '', 'htmlhelp_basename = \'' + readable_name + ' Docs\''])
	if additional:
		modify_file('conf.py', docs_directory, '')
		modify_file('conf.py', docs_directory, additional)


def git(directory, repo, travis):
	"""Set up git repo, if provided

	Enable use of TravisCI, Coveralls, and Gemnasium. Provide useful git
	hooks.
	"""
	if repo:
		modify_file('.travis.yml', directory, ['language: python', 'python:', '    - "2.7"', 'install:', '    - "pip install -r requirements.txt"', '    - "pip install coveralls"', 'script:', '    - "' + travis + '"', 'after_success:', '    - "coveralls"'])

		call(['git', 'init'], cwd=directory)

		git_hooks = os.path.join(directory, '.git', 'hooks')
		urllib.urlretrieve('https://raw.githubusercontent.com/TheKevJames/util/master/.gittemplates/hooks/pre-commit', os.path.join(git_hooks, 'pre-commit'))
		urllib.urlretrieve('https://raw.githubusercontent.com/TheKevJames/util/master/.gittemplates/hooks/post-commit', os.path.join(git_hooks, 'post-commit'))

		git_user = sub(r'.*\.com/([\w]+)/.*\.git', r'\1', repo)
		git_name = sub(r'.*' + git_user + r'/([-\w]+)\.git', r'\1', repo)

		modify_file('README.md', directory, ['[![Build Status](https://travis-ci.org/' + git_user + '/' + git_name + '.svg?branch=master)](https://travis-ci.org/' + git_user + '/' + git_name + ')', '[![Coverage Status](https://coveralls.io/repos/' + git_user + '/' + git_name + '/badge.png?branch=master)](https://coveralls.io/r/' + git_user + '/' + git_name + '?branch=master)', '[![Dependency Status](https://gemnasium.com/' + git_user + '/' + git_name + '.svg)](https://gemnasium.com/' + git_user + '/' + git_name + ')', ''])

	modify_file('README.md', directory, ['Empty project auto-generated by [AlmostEmpty](https://github.com/TheKevJames/almost-empty)', ''])

	if repo:
		call(['git', 'add', '.'], cwd=directory)
		call(['git', 'commit', '-m', 'Auto-generated AlmostEmpty project', '-n'], cwd=directory)
		call(['git', 'remote', 'add', 'origin', repo], cwd=directory)
		call(['git', 'push', '-u', '--force', 'origin', 'master'], cwd=directory)


def test(readable_name, name, directory):
	"""Set up test suite

	Enable usage of pytest and coverage testing.
	"""
	call(['pip', 'install', 'pytest'])
	call(['pip', 'install', 'pytest-cov'])

	test_directory = os.path.join(directory, 'test')
	os.mkdir(test_directory)
	modify_file('__init__.py', test_directory)
	modify_file('test_' + name + '.py', test_directory, ['import unittest', '', '', 'class Test' + readable_name + '(unittest.TestCase):', '\tdef setUp(self):', '\t\tpass', '', '\tdef test_something(self):', '\t\tassert 1 == 1', '', '\tdef tearDown(self):', '\t\tpass', '', 'if __name__ == \'__main\':', '\tunittest.main()'])


def standard(directory, ignore=None):
	"""Set up standard features common to all Python-based projects

	Write .gitignore, pylintrc, pip requirements
	"""
	store_gitignore('Python', directory)
	if ignore:
		modify_file('.gitignore', directory, ['# Project specific'])
		for line in ignore:
			modify_file('.gitignore', directory, [line])
	urllib.urlretrieve('https://raw.githubusercontent.com/TheKevJames/util/master/.pylintrc', os.path.join(directory, 'pylintrc'))

	requirements = check_output(['pip', 'freeze'])
	requirements = filter(lambda req: 'almost-empty' not in req and not req.startswith('-e'), requirements.split())
	common = {'Django', 'Flask', 'Jinja2', 'Mako', 'MarkupSafe', 'SQLAlchemy', 'Werkzeug', 'alembic', 'argparse', 'itsdangerous', 'wsgiref'}
	dev_requirements = ['-r common.txt', '']
	for req in list(requirements):
		dev = True
		for c in common:
			if c in req:
				dev = False
		if dev:
			dev_requirements.append(req)
			requirements.remove(req)
	prod_requirements = ['-r common.txt', '']

	req_directory = os.path.join(directory, 'requirements')
	os.mkdir(req_directory)
	modify_file('common.txt', req_directory, requirements)
	modify_file('dev.txt', req_directory, dev_requirements)
	modify_file('prod.txt', req_directory, prod_requirements)
	modify_file('requirements.txt', directory, ['-r requirements/dev.txt'])


####################
# Workflow Runners #
####################
def pure(name, directory, repo):
	"""Run workflow for a pure Python project

	Install required packages, set up standard project structure, generate
	docs, enable testing, create setup.py file. If the user provided a
	repository, call git setup.
	"""
	readable_name = get_readable_name(name)

	# call(['source', '/usr/local/bin/virtualenvwrapper.sh'])
	# call(['mkvirtualenv', '-a', directory, '--no-site-packages', name])

	project_directory = os.path.join(directory, name)
	os.mkdir(project_directory)
	modify_file('__init__.py', project_directory, ['__version__ = \'alpha\''])

	call(['pip', 'install', 'pyandoc'])
	call(['pip', 'install', 'wheel'])
	disp_repo = repo or 'thekev.in (auto-generated)'
	modify_file('setup.py', directory, ['#!/usr/bin/env python', 'import io', 'import os', 'import pandoc', 'from pip.req import parse_requirements', 'import re', 'from setuptools import find_packages, setup', 'from setuptools.command.test import test as TestCommand', 'import sys', '', '', 'if sys.argv[-1] == \'publish\':', '\tos.system(\'python setup.py sdist bdist_wheel upload\')', '\tsys.exit()', '', '', 'def read(*filenames, **kwargs):', '\tencoding = kwargs.get(\'encoding\', \'utf-8\')', '\tsep = kwargs.get(\'sep\', \'\\n\')', '\tbuf = []', '\tfor filename in filenames:', '\t\twith io.open(filename, encoding=encoding) as f:', '\t\t\tbuf.append(f.read())', '\treturn sep.join(buf)', '', 'def find_version(*file_paths):', '\tversion_file = read(os.path.join(*file_paths))', '\tversion_match = re.search(r"^__version__ = [\'\\"]([^\'\\"]*)[\'\\"]", version_file, re.M)', '\tif version_match:', '\t\treturn version_match.group(1)', '\traise RuntimeError(\'Unable to find version string.\')', '', 'try:', '\tpandoc.core.PANDOC_PATH = \'/usr/bin/pandoc\'', '', '\tdoc = pandoc.Document()', '\tdoc.markdown = open(\'README.md\').read()', '\twith open(\'README.rst\', \'w+\') as rst:', '\t\trst.write(doc.rst)', '\tlong_description = read(\'README.rst\')', 'except:', '\tlong_description = read(\'README.md\')', '', 'requires = [str(ir.req) for ir in parse_requirements(\'requirements/prod.txt\')]', '', 'class PyTest(TestCommand):', '\tdef finalize_options(self):', '\t\tTestCommand.finalize_options(self)', '\t\tself.test_args = []', '\t\tself.test_suite = True', '', '\tdef run_tests(self):', '\t\timport pytest', '', '\t\terrcode = pytest.main(self.test_args)', '\t\tsys.exit(errcode)', '', 'setup(', '\tname=\'' + name + '\',', '\tversion=find_version(\'' + name + '\', \'__init__.py\'),', '\tdescription=\'Auto-generated description\',', '\tlong_description=long_description,', '\tkeywords=\'' + name + '\',', '\tauthor=\'@TheKevJames (auto-generated)\',', '\tauthor_email=\'KevinJames@thekev.in (auto-generated)\',', '\turl=\'' + disp_repo + '\',', '\tlicense=\'MIT License\',', '\tpackages=find_packages(exclude=[\'test\']),', '\tinclude_package_data=True,', '\tinstall_requires=requires,', '\ttests_require=[\'pytest\'],', '\tzip_safe=False,', '\tclassifiers=[', '\t\t\'Programming Language :: Python\',', '\t\t\'Development Status :: 1 - Planning\',', '\t\t\'Natural Language :: English\',', '\t\t\'Intended Audience :: Developers\',', '\t\t\'License :: OSI Approved :: MIT License\',', '\t],', '\ttest_suite=\'test\',', '\textras_require={', '\t\t\'testing\': [\'pytest\'],', '\t},', ')'])
	modify_file('setup.cfg', directory, ['[wheel]', 'universal = 1'])
	modify_file('MANIFEST.in', directory, ['include MANIFEST.in', 'include requirements.txt'])

	doc(readable_name, name, directory)
	meta(readable_name, directory)
	test(readable_name, name, directory)

	standard(directory)
	git(directory, repo, 'coverage run --source=\'' + name + '\' setup.py test')


def django(name, directory, repo):
	"""Run workflow for a Django project

	Install required packages, set up standard project structure, generate
	docs, enable testing, create app, hide secret key, set up database
	and migrations. If the user provided a repository, call git setup.
	"""
	app_name = name + '_app'
	readable_name = get_readable_name(name)

	# call(['source', '/usr/local/bin/virtualenvwrapper.sh'])
	# call(['mkvirtualenv', '-a', directory, '--no-site-packages', name])

	call(['pip', 'install', 'django'])
	call(['pip', 'install', 'south'])

	call(['django-admin.py', 'startproject', name, directory])

	os.remove(os.path.join(directory, 'manage.py'))
	modify_file('manage.py', directory, ['#!/usr/bin/env python', 'import os', 'import sys', '', 'if __name__ == "__main__":', '\tos.environ.setdefault("DJANGO_SETTINGS_MODULE", "conf.settings")', '', '\tfrom django.core.management import execute_from_command_line', '', '\texecute_from_command_line(sys.argv)'])
	os.chmod(os.path.join(directory, 'manage.py'), 0755)

	project_directory = os.path.join(directory, name)
	os.remove(os.path.join(project_directory, 'wsgi.py'))
	modify_file('wsgi.py', project_directory, ['import os', 'os.environ.setdefault("DJANGO_SETTINGS_MODULE", "conf.settings")', '', 'from django.core.wsgi import get_wsgi_application', 'application = get_wsgi_application()'])
	os.remove(os.path.join(project_directory, 'settings.py'))

	conf_directory = os.path.join(directory, 'conf')
	os.mkdir(conf_directory)
	modify_file('__init__.py', conf_directory)
	modify_file('settings.py', conf_directory, ['try:', '\tfrom conf.secret import *', 'except ImportError:', '\tSECRET_KEY = \'ThisKeyIsPubliclyViewableDoNotUseIt\'', '', '\tfrom os.path import dirname, join', '', '\tDATABASES = {', '\t\t\'default\': {', '\t\t\t\'ENGINE\': \'django.db.backends.sqlite3\',', '\t\t\t\'NAME\': join(dirname(__file__), \'../\', \'' + name + '\', \'db.sqlite3\'),', '\t\t}', '\t}', '', '', '', 'DEBUG = True', 'TEMPLATE_DEBUG = True', '', '', 'ALLOWED_HOSTS = [\'0.0.0.0\']', '', '', 'DEFAULT_APPS = (', '\t\'django.contrib.admin\',', '\t\'django.contrib.auth\',', '\t\'django.contrib.contenttypes\',', '\t\'django.contrib.sessions\',', '\t\'django.contrib.messages\',', '\t\'django.contrib.staticfiles\',', ')', 'THIRD_PARTY_APPS = (', '\t\'south\',', ')', 'LOCAL_APPS = ()', 'INSTALLED_APPS = DEFAULT_APPS + THIRD_PARTY_APPS + LOCAL_APPS', '', '', 'MIDDLEWARE_CLASSES = (', '\t\'django.contrib.sessions.middleware.SessionMiddleware\',', '\t\'django.middleware.common.CommonMiddleware\',', '\t\'django.middleware.csrf.CsrfViewMiddleware\',', '\t\'django.contrib.auth.middleware.AuthenticationMiddleware\',', '\t\'django.contrib.messages.middleware.MessageMiddleware\',', '\t\'django.middleware.clickjacking.XFrameOptionsMiddleware\',', ')', '', '', 'ROOT_URLCONF = \'' + name + '.urls\'', '', 'WSGI_APPLICATION = \'' + name + '.wsgi.application\'', '', '', 'LANGUAGE_CODE = \'en-us\'', '', 'TIME_ZONE = \'UTC\'', '', 'USE_I18N = True', 'USE_L10N = True', 'USE_TZ = True', '', 'STATIC_URL = \'/static/\''])
	secret_key = get_random_string(50, "abcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*(-_=+)")
	modify_file('secret.py', conf_directory, ['from os.path import dirname, join', '', 'SECRET_KEY = \'' + secret_key + '\'', '', 'DATABASES = {', '\t\'default\': {', '\t\t\'ENGINE\': \'django.db.backends.sqlite3\',', '\t\t\'NAME\': join(dirname(__file__), \'../\', \'' + name + '\', \'db.sqlite3\'),', '\t}', '}'])

	call(['python', 'manage.py', 'syncdb'], cwd=directory)

	call(['python', 'manage.py', 'startapp', app_name], cwd=directory)
	os.remove(os.path.join(conf_directory, 'settings.py'))
	modify_file('settings.py', conf_directory, ['try:', '\tfrom conf.secret import *', 'except ImportError:', '\tSECRET_KEY = \'ThisKeyIsPubliclyViewableDoNotUseIt\'', '', '\tfrom os.path import dirname, join', '', '\tDATABASES = {', '\t\t\'default\': {', '\t\t\t\'ENGINE\': \'django.db.backends.sqlite3\',', '\t\t\t\'NAME\': join(dirname(__file__), \'../\', \'' + name + '\', \'db.sqlite3\'),', '\t\t}', '\t}', '', '', '', 'DEBUG = True', 'TEMPLATE_DEBUG = True', '', '', 'ALLOWED_HOSTS = [\'0.0.0.0\']', '', '', 'DEFAULT_APPS = (', '\t\'django.contrib.admin\',', '\t\'django.contrib.auth\',', '\t\'django.contrib.contenttypes\',', '\t\'django.contrib.sessions\',', '\t\'django.contrib.messages\',', '\t\'django.contrib.staticfiles\',', ')', 'THIRD_PARTY_APPS = (', '\t\'south\',', ')', 'LOCAL_APPS = (', '\t\'' + app_name + '\',', ')', 'INSTALLED_APPS = DEFAULT_APPS + THIRD_PARTY_APPS + LOCAL_APPS', '', '', 'MIDDLEWARE_CLASSES = (', '\t\'django.contrib.sessions.middleware.SessionMiddleware\',', '\t\'django.middleware.common.CommonMiddleware\',', '\t\'django.middleware.csrf.CsrfViewMiddleware\',', '\t\'django.contrib.auth.middleware.AuthenticationMiddleware\',', '\t\'django.contrib.messages.middleware.MessageMiddleware\',', '\t\'django.middleware.clickjacking.XFrameOptionsMiddleware\',', ')', '', '', 'ROOT_URLCONF = \'' + name + '.urls\'', '', 'WSGI_APPLICATION = \'' + name + '.wsgi.application\'', '', '', 'LANGUAGE_CODE = \'en-us\'', '', 'TIME_ZONE = \'UTC\'', '', 'USE_I18N = True', 'USE_L10N = True', 'USE_TZ = True', '', 'STATIC_URL = \'/static/\''])

	call(['python', 'manage.py', 'schemamigration', app_name, '--initial'], cwd=directory)
	call(['python', 'manage.py', 'migrate', app_name], cwd=directory)

	static_directory = os.path.join(directory, 'static')
	os.mkdir(static_directory)
	css_dir = os.path.join(static_directory, 'css')
	os.mkdir(css_dir)
	modify_file('.gitkeep', css_dir)
	fonts_dir = os.path.join(static_directory, 'fonts')
	os.mkdir(fonts_dir)
	modify_file('.gitkeep', fonts_dir)
	images_dir = os.path.join(static_directory, 'images')
	os.mkdir(images_dir)
	modify_file('.gitkeep', images_dir)
	js_dir = os.path.join(static_directory, 'js')
	os.mkdir(js_dir)
	modify_file('.gitkeep', js_dir)
	templates_dir = os.path.join(static_directory, 'templates')
	os.mkdir(templates_dir)
	modify_file('.gitkeep', templates_dir)

	doc(readable_name, name, directory, additional=['# -- Options for WebApps ---------------------------------------------------', 'html_static_path = [\'../static/css\', \'../static/fonts\', \'../static/images\', \'../static/js\']', 'templates_path = [\'../static/templates\']'])
	meta(readable_name, directory)
	test(readable_name, name, directory)

	standard(directory, ignore=['conf/secret.py'])
	git(directory, repo, 'coverage run --source=\'' + app_name + '\' manage.py test')


def flask(name, directory, repo):
	app_name = name + '_app'
	readable_name = get_readable_name(name)

	# call(['source', '/usr/local/bin/virtualenvwrapper.sh'])
	# call(['mkvirtualenv', '-a', directory, '--no-site-packages', name])

	call(['pip', 'install', 'flask'])
	call(['pip', 'install', 'flask-script'])
	call(['pip', 'install', 'flask-migrate'])

	app_directory = os.path.join(directory, app_name)
	os.mkdir(app_directory)
	modify_file('__init__.py', app_directory, ['from flask import Blueprint, render_template', '', app_name + ' = Blueprint(\'' + app_name + '\', __name__, template_folder = \'templates\', static_folder = \'static\', static_url_path = \'/%s\' % __name__)', '', '@' + app_name + '.route(\'/\')', 'def index():', '\treturn render_template(\'index.html\')'])

	static_directory = os.path.join(app_directory, 'static')
	os.mkdir(static_directory)
	css_dir = os.path.join(static_directory, 'css')
	os.mkdir(css_dir)
	modify_file('.gitkeep', css_dir)
	fonts_dir = os.path.join(static_directory, 'fonts')
	os.mkdir(fonts_dir)
	modify_file('.gitkeep', fonts_dir)
	images_dir = os.path.join(static_directory, 'images')
	os.mkdir(images_dir)
	modify_file('.gitkeep', images_dir)
	js_dir = os.path.join(static_directory, 'js')
	os.mkdir(js_dir)
	modify_file('.gitkeep', js_dir)
	templates_dir = os.path.join(app_directory, 'templates')
	os.mkdir(templates_dir)
	modify_file('index.html', templates_dir, ['<!doctype html>', '<html>', '<head>', '\t<title>' + readable_name + '</title>', '</head>', '<body>', '\t<p>' + readable_name + ' was auto-generated by <a href=\'https://github.com/TheKevJames/almost-empty\'>AlmostEmpty</p>', '</body>', '</html>'])

	modify_file('manage.py', directory, ['#!/usr/bin/env python', 'import sys', 'import subprocess', '', 'from flask import Flask', 'from flask.ext.migrate import Migrate, MigrateCommand', 'from flask.ext.script import Manager', 'from flask.ext.sqlalchemy import SQLAlchemy', '', 'from ' + app_name + ' import ' + app_name, '', '', 'app = Flask(__name__)', 'app.register_blueprint(' + app_name + ')', '', 'db = SQLAlchemy(app)', 'migrate = Migrate(app, db)', '', 'manager = Manager(app)', 'manager.add_command(\'db\', MigrateCommand)', '', '@manager.command', 'def test():', '\t"""Runs the test suite"""', '\tsys.exit(subprocess.call(\'py.test test\', shell=True))', '', 'if __name__ == \'__main__\':', '\tmanager.run()'])
	os.chmod(os.path.join(directory, 'manage.py'), 0755)

	call(['python', 'manage.py', 'db', 'init'], cwd=directory)
	call(['python', 'manage.py', 'db', 'migrate'], cwd=directory)
	call(['python', 'manage.py', 'db', 'upgrade'], cwd=directory)

	doc(readable_name, app_name, directory, additional=['# -- Options for WebApps ---------------------------------------------------', 'html_static_path = [\'../' + app_name + 'static/css\', \'../' + app_name + 'static/fonts\', \'../' + app_name + 'static/images\', \'../' + app_name + 'static/js\']', 'templates_path = [\'../' + app_name + 'templates\']'])
	meta(readable_name, directory)
	test(readable_name, name, directory)

	standard(directory)
	git(directory, repo, 'coverage run --source=\'' + app_name + '\' manage.py test')
