import os
import tempfile
import unittest

from almost_empty.helper import (
    get_random_string,
    get_readable_name,
    modify_file,
    store_gitignore
)


class TestHelper(unittest.TestCase):
    def setUp(self):
        pass

    def test_random_string(self):
        length = 10

        first = get_random_string(length)
        second = get_random_string(length)

        self.assertTrue(len(first) == len(second) == length)
        self.assertTrue(first != second)

    def test_readable_name(self):
        self.assertTrue(get_readable_name('almost-empty') == 'AlmostEmpty')
        self.assertTrue(get_readable_name('the-kev_james') == 'TheKevJames')
        self.assertTrue(get_readable_name('-evil-') == 'Evil')
        self.assertTrue(
            get_readable_name('thIs-*(sTriNG)_is/=fd^up') == 'ThisStringIsFdUp'
        )

    def test_modify_file(self):
        temp_dir = tempfile.gettempdir()
        temp_name = get_random_string(20)
        modify_file(temp_name, temp_dir, ['Test'])

        self.assertTrue(os.path.isfile(os.path.join(temp_dir, temp_name)))

        tmp = open(os.path.join(temp_dir, temp_name), 'r').readlines()
        self.assertTrue(tmp == ['Test\n'])

        modify_file(
            temp_name,
            temp_dir,
            ['And another test', 'with a', '', 'blank line']
        )
        tmp = open(os.path.join(temp_dir, temp_name), 'r').readlines()
        self.assertTrue(
            tmp == [
                'Test\n',
                'And another test\n',
                'with a\n',
                '\n',
                'blank line\n'
            ]
        )

    def test_store_gitignore(self):
        temp_dir = tempfile.gettempdir()
        store_gitignore('Python', temp_dir)

        self.assertTrue(os.path.isfile(os.path.join(temp_dir, '.gitignore')))

    def tearDown(self):
        pass

if __name__ == '__main':
    unittest.main()
