"""Run workflow for Python-based projects"""
import os
from re import sub
from subprocess import (call, check_output)
import urllib

from .all import (git_init_and_hook, git_finalize, init_readme, meta)
from .helper import (
    get_dotfiles_url,
    get_random_string,
    get_readable_name,
    modify_file,
    store_gitignore
)


def doc(readable_name, name, directory, additional=None):
    """Set up documentation

    Enable auto-generation of Sphinx HTML docs.
    """
    call(['pip', 'install', 'sphinx'])

    call(['sphinx-apidoc', '-F', '-o', 'docs', name], cwd=directory)
    docs_directory = os.path.join(directory, 'docs')
    os.remove(os.path.join(docs_directory, 'conf.py'))
    modify_file(
        'conf.py',
        docs_directory,
        [
            '# -*- coding: utf-8 -*-',
            '# -- General configuration -------------------------------------',
            'extensions = [',
            '    \'sphinx.ext.autodoc\',',
            '    \'sphinx.ext.viewcode\',',
            ']',
            '',
            'master_doc = \'index\'',
            'source_suffix = \'.rst\'',
            'pygments_style = \'sphinx\'',
            '',
            'exclude_patterns = [\'_build\']',
            '',
            'project = u\'' + name + '\'',
            'copyright = u\'2014, @TheKevJames (auto-generated)\'',
            'show_authors = True',
            '',
            'version = \'0\'',
            'release = \'0\'',
            '',
            'add_module_names = False',
            '',
            '# -- Options for HTML output -----------------------------------',
            'html_theme = \'agogo\'',
            '',
            'htmlhelp_basename = \'' + readable_name + ' Docs\''
        ]
    )
    if additional:
        modify_file('conf.py', docs_directory, '')
        modify_file('conf.py', docs_directory, additional)


def git(directory, repo, app_name, test_runner):
    """Set up git repo, if provided

    Enable use of Shippable and requires.io. Provide useful git
    hooks.
    """
    if repo:
        modify_file(
            'shippable.yml',
            directory,
            [
                'language: python',
                '',
                'python:',
                '  - 2.7',
                '',
                'install:',
                '  - pip install -r requirements.txt',
                '',
                'before_script:',
                '  - mkdir -p shippable/codecoverage',
                '  - mkdir -p shippable/testresults',
                '',
                'script:',
                (
                    '  - nosetests --with-xunit '
                    '--xunit-file=shippable/testresults/nosetests.xml'
                ),
                (
                    '  - coverage run --branch --source=' + app_name + ''
                    ' ' + test_runner + ' test'
                ),
                '  - coverage xml -o shippable/codecoverage/coverage.xml'
            ]
        )

        git_init_and_hook(directory)

        git_user = sub(r'.*\.com/([\w]+)/.*\.git', r'\1', repo)
        git_name = sub(r'.*' + git_user + r'/([-\w]+)\.git', r'\1', repo)

        modify_file(
            'README.md',
            directory,
            [
                (
                    '[![Requirements Status](https://requires.io/'
                    'github/' + git_user + '/' + git_name + '/'
                    'requirements.svg?branch=master)](https://requires.io/'
                    'github/' + git_user + '/' + git_name + '/'
                    'requirements/?branch=master)'
                ),
                ''
            ]
        )

    init_readme(directory)

    if repo:
        git_finalize(directory, repo)


def test(readable_name, name, directory, test_class):
    """Set up test suite

    Enable usage of nosetests and coverage testing.
    """
    call(['pip', 'install', 'coverage'])
    call(['pip', 'install', 'nose'])

    test_directory = os.path.join(directory, 'test')
    os.mkdir(test_directory)
    modify_file('__init__.py', test_directory)
    modify_file(
        'test_' + name + '.py',
        test_directory,
        [
            'from ' + test_class + ' import TestCase',
            '',
            '',
            'class Test' + readable_name + '(TestCase):',
            '    def setUp(self):',
            '        pass',
            '',
            '    def test_something(self):',
            '        assert 1 == 1',
            '',
            '    def tearDown(self):',
            '        pass',
            '',
            'if __name__ == \'__main\':',
            '    unittest.main()'
        ]
    )


def standard(directory, ignore=None):
    """Set up standard features common to all Python-based projects

    Write .gitignore, pylintrc, pip requirements
    """
    store_gitignore('Python', directory)
    if ignore:
        modify_file('.gitignore', directory, ['# Project specific'])
        for line in ignore:
            modify_file('.gitignore', directory, [line])
    urllib.urlretrieve(
        get_dotfiles_url() + 'pylint',
        os.path.join(directory, 'pylintrc')
    )

    requirements = check_output(['pip', 'freeze'])
    requirements = filter(
        lambda req: 'almost-empty' not in req and not req.startswith('-e'),
        requirements.split()
    )
    common = {
        'Django',
        'Flask',
        'Jinja2',
        'Mako',
        'MarkupSafe',
        'SQLAlchemy',
        'Werkzeug',
        'alembic',
        'argparse',
        'itsdangerous',
        'wsgiref'
    }
    dev_requirements = ['-r common.txt', '']
    for req in list(requirements):
        dev = True
        for c in common:
            if c in req:
                dev = False
        if dev:
            dev_requirements.append(req)
            requirements.remove(req)
    prod_requirements = ['-r common.txt', '']

    req_directory = os.path.join(directory, 'requirements')
    os.mkdir(req_directory)
    modify_file('common.txt', req_directory, requirements)
    modify_file('dev.txt', req_directory, dev_requirements)
    modify_file('prod.txt', req_directory, prod_requirements)
    modify_file('requirements.txt', directory, ['-r requirements/dev.txt'])


####################
# Workflow Runners #
####################
def pure(name, directory, repo):
    """Run workflow for a pure Python project

    Install required packages, set up standard project structure, generate
    docs, enable testing, create setup.py file. If the user provided a
    repository, call git setup.
    """
    readable_name = get_readable_name(name)

    # call(['source', '/usr/local/bin/virtualenvwrapper.sh'])
    # call(['mkvirtualenv', '-a', directory, '--no-site-packages', name])

    project_directory = os.path.join(directory, name)
    os.mkdir(project_directory)
    modify_file('__init__.py', project_directory, ['__version__ = \'alpha\''])

    call(['pip', 'install', 'pyandoc'])
    call(['pip', 'install', 'wheel'])
    disp_repo = repo or 'thekev.in (auto-generated)'
    modify_file(
        'setup.py',
        directory,
        [
            '#!/usr/bin/env python',
            'import io',
            'import os',
            'import pandoc',
            'from pip.req import parse_requirements',
            'import re',
            'from setuptools import find_packages, setup',
            'from setuptools.command.test import test as TestCommand',
            'import sys',
            '',
            '',
            'if sys.argv[-1] == \'publish\':',
            '    os.system(\'python setup.py sdist bdist_wheel upload\')',
            '    sys.exit()',
            '',
            '',
            'def read(*filenames, **kwargs):',
            '    encoding = kwargs.get(\'encoding\', \'utf-8\')',
            '    sep = kwargs.get(\'sep\', \'\\n\')',
            '    buf = []',
            '    for filename in filenames:',
            '        with io.open(filename, encoding=encoding) as f:',
            '            buf.append(f.read())',
            '    return sep.join(buf)',
            '',
            'def find_version(*file_paths):',
            '    version_file = read(os.path.join(*file_paths))',
            '    version_match = re.search(',
            '        r"^__version__ = [\'\\"]([^\'\\"]*)[\'\\"]",',
            '        version_file,',
            '        re.M',
            '    )',
            '    if version_match:',
            '        return version_match.group(1)',
            '    raise RuntimeError(\'Unable to find version string.\')',
            '',
            'try:',
            '    pandoc.core.PANDOC_PATH = \'/usr/bin/pandoc\'',
            '',
            '    doc = pandoc.Document()',
            '    doc.markdown = open(\'README.md\').read()',
            '    with open(\'README.rst\', \'w+\') as rst:',
            '        rst.write(doc.rst)',
            '    long_description = read(\'README.rst\')',
            'except:',
            '    long_description = read(\'README.md\')',
            '',
            (
                'requires = [str(ir.req) for ir in '
                'parse_requirements(\'requirements/prod.txt\')]'
            ),
            '',
            'class PyTest(TestCommand):',
            '    def __init__(self):',
            '        self.test_args = []',
            '        self.test_suite = True',
            '',
            '        TestCommand.__init__()',
            '',
            '    def finalize_options(self):',
            '        TestCommand.finalize_options(self)',
            '',
            'setup(',
            '    name=\'' + name + '\',',
            '    version=find_version(\'' + name + '\', \'__init__.py\'),',
            '    description=\'Auto-generated description\',',
            '    long_description=long_description,',
            '    keywords=\'' + name + '\',',
            '    author=\'@TheKevJames (auto-generated)\',',
            '    author_email=\'KevinJames@thekev.in (auto-generated)\',',
            '    url=\'' + disp_repo + '\',', '    license=\'MIT License\',',
            '    packages=find_packages(exclude=[\'test\']),',
            '    include_package_data=True,', '    install_requires=requires,',
            '    tests_require=[\'nose\'],',
            '    zip_safe=False,',
            '    classifiers=[',
            '        \'Programming Language :: Python\',',
            '        \'Development Status :: 1 - Planning\',',
            '        \'Natural Language :: English\',',
            '        \'Intended Audience :: Developers\',',
            '        \'License :: OSI Approved :: MIT License\',',
            '    ],',
            '    test_suite=\'test\',',
            '    extras_require={',
            '        \'testing\': [\'nose\'],',
            '    },',
            ')'
        ]
    )
    modify_file('setup.cfg', directory, ['[wheel]', 'universal = 1'])
    modify_file(
        'MANIFEST.in',
        directory,
        ['include MANIFEST.in', 'include requirements.txt']
    )

    doc(readable_name, name, directory)
    meta(readable_name, directory)
    test(readable_name, name, directory, 'unittest')

    standard(directory)
    git(
        directory,
        repo,
        name,
        'setup.py'
    )


def django(name, directory, repo):
    """Run workflow for a Django project

    Install required packages, set up standard project structure, generate
    docs, enable testing, create app, hide secret key, set up database
    and migrations. If the user provided a repository, call git setup.
    """
    app_name = name + '_app'
    readable_name = get_readable_name(name)

    # call(['source', '/usr/local/bin/virtualenvwrapper.sh'])
    # call(['mkvirtualenv', '-a', directory, '--no-site-packages', name])

    call(['pip', 'install', 'django'])
    call(['pip', 'install', 'south'])

    call(['django-admin.py', 'startproject', name, directory])

    os.remove(os.path.join(directory, 'manage.py'))
    modify_file(
        'manage.py',
        directory,
        [
            '#!/usr/bin/env python',
            'import os',
            'import sys',
            '',
            'if __name__ == "__main__":',
            (
                '    os.environ.setdefault'
                '("DJANGO_SETTINGS_MODULE", "conf.settings")'
            ),
            '',
            '    from django.core.management import execute_from_command_line',
            '',
            '    execute_from_command_line(sys.argv)'
        ]
    )
    os.chmod(os.path.join(directory, 'manage.py'), 0755)

    project_directory = os.path.join(directory, name)
    os.remove(os.path.join(project_directory, 'wsgi.py'))
    modify_file(
        'wsgi.py',
        project_directory,
        [
            'import os',
            'os.environ.setdefault("DJANGO_SETTINGS_MODULE", "conf.settings")',
            '',
            'from django.core.wsgi import get_wsgi_application',
            'application = get_wsgi_application()'
        ]
    )
    os.remove(os.path.join(project_directory, 'settings.py'))

    conf_directory = os.path.join(directory, 'conf')
    os.mkdir(conf_directory)
    modify_file('__init__.py', conf_directory)
    modify_file(
        'settings.py',
        conf_directory,
        [
            'try:',
            '    from conf.secret import *',
            'except ImportError:',
            '    SECRET_KEY = \'ThisKeyIsPubliclyViewableDoNotUseIt\'',
            '',
            '    from os.path import dirname, join',
            '',
            '    DATABASES = {',
            '        \'default\': {',
            '            \'ENGINE\': \'django.db.backends.sqlite3\',',
            (
                '            \'NAME\': join('
                'dirname(__file__), \'../\', '
                '\'' + name + '\', '
                '\'db.sqlite3\'),'
            ),
            '        }',
            '    }',
            '',
            '',
            '',
            'DEBUG = True',
            'TEMPLATE_DEBUG = True',
            '',
            '',
            'ALLOWED_HOSTS = [\'0.0.0.0\']',
            '',
            '',
            'DEFAULT_APPS = (',
            '    \'django.contrib.admin\',',
            '    \'django.contrib.auth\',',
            '    \'django.contrib.contenttypes\',',
            '    \'django.contrib.sessions\',',
            '    \'django.contrib.messages\',',
            '    \'django.contrib.staticfiles\',',
            ')',
            'THIRD_PARTY_APPS = (',
            '    \'south\',',
            ')',
            'LOCAL_APPS = ()',
            'INSTALLED_APPS = DEFAULT_APPS + THIRD_PARTY_APPS + LOCAL_APPS',
            '',
            '',
            'MIDDLEWARE_CLASSES = (',
            '    \'django.contrib.sessions.middleware.SessionMiddleware\',',
            '    \'django.middleware.common.CommonMiddleware\',',
            '    \'django.middleware.csrf.CsrfViewMiddleware\',',
            '    \'django.contrib.auth.middleware.AuthenticationMiddleware\',',
            '    \'django.contrib.messages.middleware.MessageMiddleware\',',
            '    \'django.middleware.clickjacking.XFrameOptionsMiddleware\',',
            ')',
            '',
            '',
            'ROOT_URLCONF = \'' + name + '.urls\'',
            '',
            'WSGI_APPLICATION = \'' + name + '.wsgi.application\'',
            '',
            '',
            'LANGUAGE_CODE = \'en-us\'',
            '',
            'TIME_ZONE = \'UTC\'',
            '',
            'USE_I18N = True',
            'USE_L10N = True',
            'USE_TZ = True',
            '',
            'STATIC_URL = \'/static/\''
        ]
    )
    secret_key = get_random_string(
        50,
        "abcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*(-_=+)"
    )
    modify_file(
        'secret.py',
        conf_directory,
        [
            'from os.path import dirname, join',
            '',
            'SECRET_KEY = \'' + secret_key + '\'',
            '',
            'DATABASES = {',
            '    \'default\': {',
            '        \'ENGINE\': \'django.db.backends.sqlite3\',',
            (
                '        \'NAME\': join('
                'dirname(__file__), \'../\', '
                '\'' + name + '\', '
                '\'db.sqlite3\'),'
            ),
            '    }',
            '}'
        ]
    )

    call(['python', 'manage.py', 'syncdb'], cwd=directory)

    call(['python', 'manage.py', 'startapp', app_name], cwd=directory)
    os.remove(os.path.join(conf_directory, 'settings.py'))
    modify_file(
        'settings.py',
        conf_directory,
        [
            'try:',
            '    from conf.secret import *',
            'except ImportError:',
            '    SECRET_KEY = \'ThisKeyIsPubliclyViewableDoNotUseIt\'',
            '',
            '    from os.path import dirname, join',
            '',
            '    DATABASES = {',
            '        \'default\': {',
            '            \'ENGINE\': \'django.db.backends.sqlite3\',',
            (
                '            \'NAME\': join('
                'dirname(__file__), \'../\', '
                '\'' + name + '\', '
                '\'db.sqlite3\'),'
            ),
            '        }',
            '    }',
            '',
            '',
            '',
            'DEBUG = True',
            'TEMPLATE_DEBUG = True',
            '',
            '',
            'ALLOWED_HOSTS = [\'0.0.0.0\']',
            '',
            '',
            'DEFAULT_APPS = (',
            '    \'django.contrib.admin\',',
            '    \'django.contrib.auth\',',
            '    \'django.contrib.contenttypes\',',
            '    \'django.contrib.sessions\',',
            '    \'django.contrib.messages\',',
            '    \'django.contrib.staticfiles\',',
            ')',
            'THIRD_PARTY_APPS = (',
            '    \'south\',',
            ')',
            'LOCAL_APPS = (',
            '    \'' + app_name + '\',',
            ')',
            'INSTALLED_APPS = DEFAULT_APPS + THIRD_PARTY_APPS + LOCAL_APPS',
            '',
            '',
            'MIDDLEWARE_CLASSES = (',
            '    \'django.contrib.sessions.middleware.SessionMiddleware\',',
            '    \'django.middleware.common.CommonMiddleware\',',
            '    \'django.middleware.csrf.CsrfViewMiddleware\',',
            '    \'django.contrib.auth.middleware.AuthenticationMiddleware\',',
            '    \'django.contrib.messages.middleware.MessageMiddleware\',',
            '    \'django.middleware.clickjacking.XFrameOptionsMiddleware\',',
            ')',
            '',
            '',
            'ROOT_URLCONF = \'' + name + '.urls\'',
            '',
            'WSGI_APPLICATION = \'' + name + '.wsgi.application\'',
            '',
            '',
            'LANGUAGE_CODE = \'en-us\'',
            '',
            'TIME_ZONE = \'UTC\'',
            '',
            'USE_I18N = True',
            'USE_L10N = True',
            'USE_TZ = True',
            '',
            'STATIC_URL = \'/static/\''
        ]
    )

    call(
        ['python', 'manage.py', 'schemamigration', app_name, '--initial'],
        cwd=directory
    )
    call(['python', 'manage.py', 'migrate', app_name], cwd=directory)

    static_directory = os.path.join(directory, 'static')
    os.mkdir(static_directory)
    css_dir = os.path.join(static_directory, 'css')
    os.mkdir(css_dir)
    modify_file('.gitkeep', css_dir)
    fonts_dir = os.path.join(static_directory, 'fonts')
    os.mkdir(fonts_dir)
    modify_file('.gitkeep', fonts_dir)
    images_dir = os.path.join(static_directory, 'images')
    os.mkdir(images_dir)
    modify_file('.gitkeep', images_dir)
    js_dir = os.path.join(static_directory, 'js')
    os.mkdir(js_dir)
    modify_file('.gitkeep', js_dir)
    templates_dir = os.path.join(static_directory, 'templates')
    os.mkdir(templates_dir)
    modify_file('.gitkeep', templates_dir)

    doc(
        readable_name,
        name,
        directory,
        additional=[
            '# -- Options for WebApps ---------------------------------------',
            (
                'html_static_path = ['
                '\'../static/css\', '
                '\'../static/fonts\', '
                '\'../static/images\', '
                '\'../static/js\''
                ']'
            ),
            'templates_path = [\'../static/templates\']'
        ]
    )
    meta(readable_name, directory)
    test(readable_name, name, directory, 'django.test')

    standard(directory, ignore=['conf/secret.py'])
    git(
        directory,
        repo,
        app_name,
        'manage.py'
    )


def flask(name, directory, repo):
    app_name = name + '_app'
    readable_name = get_readable_name(name)

    # call(['source', '/usr/local/bin/virtualenvwrapper.sh'])
    # call(['mkvirtualenv', '-a', directory, '--no-site-packages', name])

    call(['pip', 'install', 'flask'])
    call(['pip', 'install', 'flask-script'])
    call(['pip', 'install', 'flask-migrate'])

    app_directory = os.path.join(directory, app_name)
    os.mkdir(app_directory)
    modify_file(
        '__init__.py',
        app_directory,
        [
            'from flask import Blueprint, render_template',
            '',
            (
                '' + app_name + ' = Blueprint('
                '\'' + app_name + '\', '
                '__name__, '
                'template_folder = \'templates\', '
                'static_folder = \'static\', '
                'static_url_path = \'/%s\' % __name__)'
            ),
            '',
            '@' + app_name + '.route(\'/\')',
            'def index():',
            '    return render_template(\'index.html\')'
        ]
    )

    static_directory = os.path.join(app_directory, 'static')
    os.mkdir(static_directory)
    css_dir = os.path.join(static_directory, 'css')
    os.mkdir(css_dir)
    modify_file('.gitkeep', css_dir)
    fonts_dir = os.path.join(static_directory, 'fonts')
    os.mkdir(fonts_dir)
    modify_file('.gitkeep', fonts_dir)
    images_dir = os.path.join(static_directory, 'images')
    os.mkdir(images_dir)
    modify_file('.gitkeep', images_dir)
    js_dir = os.path.join(static_directory, 'js')
    os.mkdir(js_dir)
    modify_file('.gitkeep', js_dir)
    templates_dir = os.path.join(app_directory, 'templates')
    os.mkdir(templates_dir)
    modify_file(
        'index.html',
        templates_dir,
        [
            '<!doctype html>',
            '<html>',
            '<head>',
            '\t<title>' + readable_name + '</title>',
            '</head>',
            '<body>',
            (
                '\t<p>' + readable_name + ' was auto-generated by '
                '<a href=\'https://github.com/TheKevJames/almost-empty\'>'
                'AlmostEmpty</p>'
            ),
            '</body>',
            '</html>'
        ]
    )

    modify_file(
        'manage.py',
        directory,
        [
            '#!/usr/bin/env python',
            'import sys',
            'import subprocess',
            '',
            'from flask import Flask',
            'from flask.ext.migrate import Migrate, MigrateCommand',
            'from flask.ext.script import Manager',
            'from flask.ext.sqlalchemy import SQLAlchemy',
            '',
            'from ' + app_name + ' import ' + app_name,
            '',
            '',
            'app = Flask(__name__)',
            'app.register_blueprint(' + app_name + ')',
            '',
            'db = SQLAlchemy(app)',
            'migrate = Migrate(app, db)',
            '',
            'manager = Manager(app)',
            'manager.add_command(\'db\', MigrateCommand)',
            '',
            '@manager.command',
            'def test():',
            '    """Runs the test suite"""',
            '    sys.exit(subprocess.call(\'nosetests\', shell=True))',
            '',
            'if __name__ == \'__main__\':',
            '    manager.run()'
        ]
    )
    os.chmod(os.path.join(directory, 'manage.py'), 0755)

    call(['python', 'manage.py', 'db', 'init'], cwd=directory)
    call(['python', 'manage.py', 'db', 'migrate'], cwd=directory)
    call(['python', 'manage.py', 'db', 'upgrade'], cwd=directory)

    doc(
        readable_name,
        app_name,
        directory,
        additional=[
            '# -- Options for WebApps ---------------------------------------',
            (
                'html_static_path = ['
                '\'../' + app_name + 'static/css\', '
                '\'../' + app_name + 'static/fonts\', '
                '\'../' + app_name + 'static/images\', '
                '\'../' + app_name + 'static/js\''
                ']'
            ),
            'templates_path = [\'../' + app_name + 'templates\']'
        ]
    )
    meta(readable_name, directory)
    test(readable_name, name, directory, 'unittest')

    standard(directory)
    git(
        directory,
        repo,
        app_name,
        'manage.py'
    )
