"""Run workflow for Scala-based projects"""
import os

from .all import (git_init_and_hook, git_finalize, init_readme, meta)
from .helper import (get_readable_name, modify_file, store_gitignore)


####################
# Workflow Runners #
####################
def pure(name, directory, repo):
    """Run workflow for a pure Python project"""
    name = name.replace('-', '_')
    readable_name = get_readable_name(name)

    modify_file(
        'build.sbt',
        directory,
        [
            'name := "' + readable_name + '"',
            '',
            'version := "alpha"',
            '',
            'scalaVersion := "2.10.4"',
            '',
            (
                'resolvers += "Typesafe repository" at '
                '"http://repo.typesafe.com/typesafe/releases/"'
            ),
            '',
            (
                'libraryDependencies += "org.scalatest"'
                ' %% "scalatest" % "1.9.1" % "test"'
            ),
            '',
            'libraryDependencies += "junit" % "junit" % "4.10" % "test"',
            '',
            'initialCommands in console := "import ' + name + '._"',
            '',
            'mainClass in (Compile, packageBin) := Some("' + name + '.Main")',
            '',
            'mainClass in (Compile, run) := Some("' + name + '.Main")',
            '',
            (
                'shellPrompt in ThisBuild := { '
                'state => Project.extract(state).currentRef.project'
                ' + "> " }'
            ),
            '',
            (
                'shellPrompt := { state => '
                'System.getProperty("user.name") + "> " }'
            ),
            '',
            'instrumentSettings'
        ]
    )

    meta(readable_name, directory)
    store_gitignore('Scala', directory)

    project_dir = os.path.join(directory, 'project')
    os.mkdir(project_dir)
    modify_file(
        'build.sbt',
        project_dir,
        [
            'resolvers += Classpaths.sbtPluginReleases',
            '',
            (
                'addSbtPlugin("org.scoverage" %% '
                '"sbt-scoverage" % "0.99.7.1")'
            )
        ]
    )

    src_dir = os.path.join(directory, 'src')
    os.mkdir(src_dir)

    main_dir = os.path.join(src_dir, 'main')
    os.mkdir(main_dir)

    test_dir = os.path.join(src_dir, 'test')
    os.mkdir(test_dir)

    for direc in (main_dir, test_dir):
        os.mkdir(os.path.join(direc, 'scala'))
        os.mkdir(os.path.join(direc, 'scala', name))

    modify_file(
        'Main.scala',
        os.path.join(main_dir, 'scala', name),
        [
            'package ' + name,
            'import ' + readable_name + '._',
            '',
            'object Main extends App {',
            '  println("Hello world!")',
            '}'
        ]
    )

    modify_file(
        readable_name + '.scala',
        os.path.join(main_dir, 'scala', name),
        [
            'package ' + name,
            '',
            'object ' + readable_name + ' {',
            '  ',
            '}'
        ]
    )

    modify_file(
        readable_name + 'Suite.scala',
        os.path.join(test_dir, 'scala', name),
        [
            'package ' + name,
            '',
            'import org.junit.runner.RunWith',
            'import org.scalatest.FunSuite',
            'import org.scalatest.junit.JUnitRunner',
            '',
            '@RunWith(classOf[JUnitRunner])',
            'class ' + readable_name + 'Suite extends FunSuite {',
            '  test("test name") {',
            '    assert(1 + 1 === 2)',
            '  }',
            '}'
        ]
    )

    init_readme(directory)

    if repo:
        modify_file(
            'shippable.yml',
            directory,
            [
                'language: scala',
                '',
                'scala:',
                '  - 2.10.4',
                '',
                'before_script:',
                '  - mkdir -p shippable/codecoverage',
                '  - mkdir -p shippable/testresults',
                '',
                'script:',
                '  - sbt clean scoverage:test',
                '',
                'after_script:',
                (
                    '  - mv target/scala-2.10/coverage-report/cobertura.xml '
                    '$PWD/shippable/codecoverage/'
                )
            ]
        )

        git_init_and_hook(directory)
        git_finalize(directory, repo)
