'''
    aml.node
    ~~~~~~~~

    :copyright: 2009 by Bjoern Schulz <bjoern.schulz.92@gmail.com>
    :license: MIT, see LICENSE for more details
'''

import string
from types import NoneType

class Node(object):
    
    def __init__(self, name, attributes=None, children=()):
        self.name = name
        self.attributes = attributes or {}
        self.children = []
        self.parent = self.first_child = self.last_child = None
        self.next_sibling = self.previous_sibling = None
        for child in children:
            self.append(child)
    
    def __repr__(self):
        return 'Node(%r, %r, (%s))' % (
            self.name,
            self.attributes,
            ', '.join(map(repr, self.children))
        )
    
    def __str__(self):
        def escape(s):
            for c in string.whitespace:
                s = s.replace(c, c.encode('string-escape'))
            return s.replace('"', '\\"')
        types = {
            NoneType: lambda v: '#n',
            bool: lambda v: '#t' if v else '#f',
            str: lambda v: '"%s"' % escape(v),
            unicode: lambda v: '"%s"' % escape(v).encode('utf-8'),
            int: lambda v: v,
            float: lambda v: v,
        }
        return '(%s%s%s%s)' % (
            self.name,
            ''.join(
                ' (%s %s)' % (
                    name, types[type(value)](value)
                ) for name, value in self.iter_attrs()
            ),
            ' ' if self.children else '',
            ' '.join(map(str, self.children))
        )
    
    def __int__(self):
        return self.__len__()
    
    def __nonzero__(self):
        return bool(self.children or self.attributes)
    
    def __contains__(self, node):
        return node in self.children
    
    def __len__(self):
        return len(self.children)
    
    def __iter__(self):
        return iter(self.children)
    
    def iter_attrs(self):
        for name in self.attributes:
            yield name, self.attributes[name]
    
    def __setitem__(self, name, value):
        self.attributes.__setitem__(name, value)
    
    def __getitem__(self, name):
        return self.attributes.__getitem__(name)
    
    def __delitem__(self, name):
        self.attributes.__delitem__(name)
    
    def has_attr(self, name):
        return name in self.attributes
    
    def get_attr(self, name, default=None):
        return self.__getitem__(name) if self.has_attr(name) else default
    
    def append(self, node):
        self.insert(len(self), node)
    
    def insert(self, index, node):
        node.parent = self
        self.children.insert(index, node)
        self.first_child = self.children[0]
        self.last_child = self.children[-1]
    
    def remove(self, node):
        self.children.remove(node)
        if self.children:
            self.first_child = self.children[0]
            self.last_child = self.children[-1]
        else:
            self.first_child = self.last_child = None
    
    def __eq__(self, other):
        return (
            self.name == other.name and
            self.attributes == other.attributes and
            self.children == other.children
            if isinstance(other, Node) else False
        )
    
    def __ne__(self, other):
        return not (self == other)