# Copyright 2014 Vivien Maisonneuve <v.maisonneuve@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Color screen output using ANSI escape sequences.
"""


import os
import re


__all__ = ['color', 'uncolor', 'colored', 'colorstrip', 'colorvalid']
__version__ = '0.1.1'


_ATTRIBUTES = {
    'clear':          '0',
    'reset':          '0',
    'bold':           '1',
    'dark':           '2',
    'faint':          '2',
    'italic':         '3',
    'underline':      '4',
    'underscore':     '4',
    'blink':          '5',
    'reverse':        '7',
    'concealed':      '8',

    'black':          '30',   'on_black':          '40',
    'red':            '31',   'on_red':            '41',
    'green':          '32',   'on_green':          '42',
    'yellow':         '33',   'on_yellow':         '43',
    'blue':           '34',   'on_blue':           '44',
    'magenta':        '35',   'on_magenta':        '45',
    'cyan':           '36',   'on_cyan':           '46',
    'white':          '37',   'on_white':          '47',

    'bright_black':   '90',   'on_bright_black':   '100',
    'bright_red':     '91',   'on_bright_red':     '101',
    'bright_green':   '92',   'on_bright_green':   '102',
    'bright_yellow':  '93',   'on_bright_yellow':  '103',
    'bright_blue':    '94',   'on_bright_blue':    '104',
    'bright_magenta': '95',   'on_bright_magenta': '105',
    'bright_cyan':    '96',   'on_bright_cyan':    '106',
    'bright_white':   '97',   'on_bright_white':   '107',
}

for _code in range(16):
    _ATTRIBUTES['ansi{}'.format(_code)] = '38;5;{}'.format(_code)
    _ATTRIBUTES['on_ansi{}'.format(_code)] = '48;5;{}'.format(_code)

for _r in range(6):
    for _g in range(6):
        for _b in range(6):
            _code = 16 + 6*6*_r + 6*_g + _b
            _attr = 'rgb{}{}{}'.format(_r, _g, _b)
            _ATTRIBUTES[_attr] = '38;5;{}'.format(_code)
            _ATTRIBUTES['on_{}'.format(_attr)] = '48;5;{}'.format(_code)

for _n in range(24):
    _code = _n + 232
    _ATTRIBUTES['grey{}'.format(_n)] = '38;5;{}'.format(_code)
    _ATTRIBUTES['on_grey{}'.format(_n)] = '48;5;{}'.format(_code)

_ATTRIBUTES_R = {}
for _attr in sorted(_ATTRIBUTES, reverse=True):
    _code = _ATTRIBUTES[_attr]
    _ATTRIBUTES_R[_code] = _attr

for _attr, _code in _ATTRIBUTES.items():
    _const = _attr.upper()
    globals()[_const] = '\033[{}m'.format(_code)
    __all__.append(_const)

def splitattrs(*attrs):
    for attr in attrs:
        yield from re.split('\s+', attr.lower())

def color(*attrs):
    """
    Return the escape code for a given set of color _attributes.
    """
    if os.getenv('ANSI_COLORS_DISABLED'):
        return ''
    codes = []
    for attr in splitattrs(*attrs):
        try:
            codes.append(_ATTRIBUTES[attr])
        except KeyError:
            raise ValueError('invalid attribute name: {!r}'.format(attr))
    if not codes:
        return ''
    codes = ';'.join(map(str, codes))
    codes = '\033[{}m'.format(codes)
    return codes

def uncolor(*escapes):
    """
    Return a list of named color _attributes for a given set of escape
    codes.  Escape sequences can be given with or without enclosing '\\033['
    and 'm'.  The empty escape sequence '' or '\\033[m' gives an empty list
    of attrs.
    """
    codes = []
    for escape in escapes:
        escape = re.sub('^\033\[', '', escape)
        escape = re.sub('m$', '', escape)
        if not re.match('^(?:\d+;)*\d*$', escape):
            raise ValueError('bad escape sequence: {!r}'.format(escape))
        for match in re.finditer('(0*[34]8;0*5;\d+|\d+)(?:;|$)', escape):
            codes.append(match.group(1))
    attrs = []
    for code in codes:
        code = re.sub('(^|;)0+(\d)', '\1\2', code)
        try:
            attr = _ATTRIBUTES_R[code]
        except KeyError:
            raise ValueError('no name for escape sequence {!r}'.format(code))
        attrs.append(attr)
    return attrs

def colored(string, *attrs, eachline=False):
    """
    Given a string and a set of _attributes, returns the string surrounded
    by escape codes to set those _attributes and then clear them at the end
    of the string.
    """
    if not string:
        return string
    codes = color(*attrs)
    if not codes:
        return string
    if eachline:
        lines = re.split('(\n|\r\n?)', string)
        for i in range(0, len(lines), 2):
            line = lines[i]
            if line:
                lines[i] = '{}{}\033[0m'.format(codes, line)
        return ''.join(lines)
    else:
        return '{}{}\033[0m'.format(codes, string)

def colorstrip(string):
    """
    Given a string, strip the ANSI color codes out of that string and return
    the result.  This removes only ANSI color codes, not movement codes and
    other escape sequences.
    """
    string = re.sub('\033\[[\d;]*m', '', string)
    return string

def colorvalid(*attrs):
    """
    Given a list of color _attributes (arguments for color, for instance),
    return True if they're all valid or False if any of them are invalid.
    """
    for attr in splitattrs(*attrs):
        if attr not in _ATTRIBUTES:
            return False
    return True
