try:
  import unittest2 as unittest
except ImportError:
  import unittest

import sys
from StringIO import StringIO

from mock import patch
from mock import MagicMock as Mock

import arklib.cli.ark_set_dns as ark_set_dns

class ark_set_dns_test(unittest.TestCase):

  @patch('arklib.cli.ark_set_dns.ark_route53')
  def test_set_dns_record(self, m_r53):
    r53 = Mock()
    m_r53.return_value = r53
    r53.find_hosted_zone_id.return_value = 'zone_id'
    r53.domain_exists.return_value = False
    _cred = Mock()
    _cred.aws_access_key = 'access'
    _cred.aws_secret_key = 'secret'

    saved_stdout = sys.stdout
    try:
      out = StringIO()
      sys.stdout = out
      # Invoke
      ark_set_dns.set_dns_record('a.domain.com', 'public.dns', _cred)
      output = out.getvalue().strip()
    finally:
      sys.stdout = saved_stdout

    # Asserts
    m_r53.assert_called_with('access', 'secret')
    r53.find_hosted_zone_id.assert_called_with("domain.com")
    r53.domain_exists.assert_called_with('a.domain.com', "CNAME", "zone_id")
    r53.create_cname_record.assert_called_with('a.domain.com', 'public.dns', 'zone_id')
    self.assertEquals(output, "Added a CNAME record: a.domain.com => public.dns")

  @patch('arklib.cli.ark_set_dns.ark_route53')
  def test_set_dns_record_zid_not_found(self, m_r53):
    r53 = Mock()
    m_r53.return_value = r53
    r53.find_hosted_zone_id.return_value = None
    _cred = Mock()
    _cred.aws_access_key = 'access'
    _cred.aws_secret_key = 'secret'

    saved_stdout = sys.stdout
    try:
      out = StringIO()
      sys.stdout = out
      # Invoke
      ark_set_dns.set_dns_record('a.domain.com', 'public.dns', _cred)
      output = out.getvalue().strip()
    finally:
      sys.stdout = saved_stdout

    # Asserts
    m_r53.assert_called_with('access', 'secret')
    r53.find_hosted_zone_id.assert_called_with("domain.com")
    self.assertEquals(output, "Could not find a zone for domain.com in route53")

  @patch('arklib.cli.ark_set_dns.ark_route53')
  def test_set_dns_record_domain_exists(self, m_r53):
    r53 = Mock()
    m_r53.return_value = r53
    r53.find_hosted_zone_id.return_value = 'zone_id'
    r53.domain_exists.return_value = True
    _cred = Mock()
    _cred.aws_access_key = 'access'
    _cred.aws_secret_key = 'secret'

    saved_stdout = sys.stdout
    try:
      out = StringIO()
      sys.stdout = out
      # Invoke
      ark_set_dns.set_dns_record('a.domain.com', 'public.dns', _cred)
      output = out.getvalue().strip()
    finally:
      sys.stdout = saved_stdout

    # Asserts
    m_r53.assert_called_with('access', 'secret')
    r53.find_hosted_zone_id.assert_called_with("domain.com")
    r53.domain_exists.assert_called_with('a.domain.com', "CNAME", "zone_id")
    self.assertEquals(output, "Domain a.domain.com exists")

  @patch('arklib.cli.ark_set_dns.set_dns_record')
  @patch('boto.utils.get_instance_metadata')
  @patch('arklib.cli.ark_set_dns.ark_ec2')
  @patch('arklib.cli.ark_set_dns.ark_cred')
  def test_main(self, m_cred, m_ec2, m_meta, m_set):
    cred = Mock()
    cred.aws_access_key = "key"
    cred.aws_secret_key = "secret"
    m_cred.return_value = cred
    ec2 = Mock()
    m_ec2.return_value = ec2
    metadata = {'public-hostname': 'ph', 'instance-id': 'id'}

    m_meta.return_value = metadata
    ec2.retrieve_tags.return_value = {'dns': 'new_dns'}

    # Invoke
    ark_set_dns.main()

    # Assertions
    m_ec2.assert_called_with("key", "secret", "us-east-1")
    m_meta.assert_called_with()
    ec2.retrieve_tags.assert_called_with('id')
    m_set.assert_called_with('new_dns', 'ph', cred)

  @patch('arklib.cli.ark_set_dns.ark_cred')
  def test_main_with_exception(self, m_cred):
    cred = Mock(side_effect = Exception)
    cred.__str__ = "mock"
    m_cred.return_value = cred

    with self.assertRaises(SystemExit) as cm:
      ark_set_dns.main()

    self.assertEquals(cm.exception.code, 1)

