try:
  import unittest2 as unittest
except ImportError:
  import unittest

from mock import patch
from mock import MagicMock as Mock

import sys
from StringIO import StringIO

import arklib.cli.ark_backup_dir as ark_backup_dir

class ark_backup_dir_test(unittest.TestCase):
  
  def test_generate_backup_filename(self):
    f = ark_backup_dir.generate_backup_filename("home")
    self.assertTrue(f.startswith("home"))
    self.assertTrue(f.endswith(".tar.gz"))

  @patch('arklib.cli.ark_backup_dir.ark_s3')
  def test_upload_backup_to_s3(self, m_s3):
    s3 = Mock()
    m_s3.return_value = s3

    _cred = Mock()
    _cred.aws_access_key = "key"
    _cred.aws_secret_key = "secret"

    s3.parse_bucket_key.return_value = ("s3_bucket", "s3_key")
    s3.trim_s3_scheme.return_value = "/bucket/key"

    # Invoke
    ark_backup_dir.upload_backup_to_s3("local", "s3:///bucket/key", _cred)

    # Asserts
    m_s3.assert_called_with("key", "secret")
    s3.trim_s3_scheme.assert_called_with("s3:///bucket/key")
    s3.parse_bucket_key.assert_called_with("/bucket/key")
    s3.upload_file.assert_called_with("s3_bucket", "s3_key", "local")

  def test_main_without_args(self):
    saved_stdout = sys.stdout
    try:
      out = StringIO()
      sys.stdout = out

      # Invoke
      with self.assertRaises(SystemExit) as cm:
        ark_backup_dir.run_main([])
      self.assertEquals(cm.exception.code, 1)

      output = out.getvalue().strip()
    finally:
      sys.stdout = saved_stdout

    self.assertEquals(output, "Syntax: ark-backup-dir [local-dir] [s3://backup-dir]")

  @patch('arklib.cli.ark_backup_dir.ark_cred')
  @patch('arklib.cli.ark_backup_dir.ark_utils')
  def test_main_with_exception(self, m_utils, m_cred):
    bad_mock = Mock(side_effect = Exception)
    bad_mock.__str__ = "str"
    m_cred.return_value = bad_mock

    with self.assertRaises(SystemExit) as cm:
      ark_backup_dir.run_main(['name', 'abce', 's3://bucket/path/puppet.tar.gz'])

    self.assertEquals(cm.exception.code, 1)

  @patch('arklib.cli.ark_backup_dir.ark_cred')
  @patch('arklib.cli.ark_backup_dir.upload_backup_to_s3')
  @patch('arklib.cli.ark_backup_dir.generate_backup_filename')
  @patch('arklib.cli.ark_backup_dir.ark_utils')
  @patch('arklib.cli.ark_backup_dir.backup_script')
  def test_main(self, m_script, m_utils, m_gen, m_upload, m_cred):
    _cred = Mock()
    _cred.aws_access_key = 'access'
    _cred.aws_secret_key = 'secret'
    m_cred.return_value = _cred

    m_utils.random_word.return_value = "rnd"

    m_gen.return_value = "local_backup"

    ark_backup_dir.run_main(['script_name', 'local', 's3://bucket/path/puppet.tar.gz'])

    m_cred.assert_called_with()
    m_gen.assert_called_with("local")
    m_utils.write_script.assert_called_with("/tmp/ark-dir-backup-rnd.sh", m_script)
    m_utils.exec_script.assert_called_with(["/tmp/ark-dir-backup-rnd.sh", "local", "/tmp/rnd-local_backup"])
    m_utils.delete_script.assert_called_with("/tmp/rnd-local_backup")

