try:
  import unittest2 as unittest
except ImportError:
  import unittest

from mock import patch
from mock import MagicMock as Mock

import sys
from StringIO import StringIO

import arklib.cli.ark_puppet_apply as ark_puppet_apply
import stat

class ark_puppet_apply_test(unittest.TestCase):

  @patch('arklib.cli.ark_puppet_apply.ark_s3')
  def test_download_manifest_from_s3(self, m_s3):
    _cred = Mock()
    _cred.aws_access_key = 'access'
    _cred.aws_secret_key = 'secret'

    s3 = Mock()
    m_s3.return_value = s3

    s3.trim_s3_scheme.return_value = 'trimmed_url'
    s3.parse_bucket_key.return_value = ('bucket', 'key')

    ark_puppet_apply.download_manifest_from_s3('url', _cred)

    # Asserts
    m_s3.assert_called_with('access', 'secret')
    s3.trim_s3_scheme.assert_called_with('url')
    s3.parse_bucket_key.assert_called_with('trimmed_url')
    s3.download_file.assert_called_with('bucket', 'key', ark_puppet_apply.local_puppet_path)

  @patch('__builtin__.open')
  @patch('os.stat')
  @patch('os.chmod')
  def test_write_script(self, m_chmod, m_stat, m_open):
    # Setup
    script = "script"
    ark_puppet_apply.apply_script = script
    m_file = Mock()
    m_open.return_value = m_file
    m_file.write
    m_st = Mock()
    m_stat.return_value = m_st
    m_st.st_mode = 10

    # Run
    ark_puppet_apply.write_script()

    # Asserts
    m_open.assert_called_with(ark_puppet_apply.script_path, 'w')
    m_file.write.assert_called_with(script)
    m_file.close.assert_called_with()
    m_stat.assert_called_with(ark_puppet_apply.script_path)
    m_chmod.assert_called_with(ark_puppet_apply.script_path, m_st.st_mode | stat.S_IEXEC)

  @patch('subprocess.call')
  def test_exec_script(self, sp_call):
    ark_puppet_apply.exec_script()
    sp_call.assert_called_with(ark_puppet_apply.script_path, shell=True)

  @patch('os.remove')
  def test_delete_script(self, os_remove):
    ark_puppet_apply.delete_script()
    os_remove.assert_called_with(ark_puppet_apply.script_path)

  @patch('arklib.cli.ark_puppet_apply.ark_cred')
  @patch('arklib.cli.ark_puppet_apply.download_manifest_from_s3')
  @patch('arklib.cli.ark_puppet_apply.delete_script')
  @patch('arklib.cli.ark_puppet_apply.exec_script')
  @patch('arklib.cli.ark_puppet_apply.write_script')
  def test_main(self, m_write, m_exec, m_delete, m_download, m_cred):
    _cred = Mock()
    _cred.aws_access_key = 'access'
    _cred.aws_secret_key = 'secret'
    m_cred.return_value = _cred

    ark_puppet_apply.run_main(['script_name', 's3://bucket/path/puppet.tar.gz'])

    m_cred.assert_called_with()
    m_download.assert_called_with('s3://bucket/path/puppet.tar.gz', _cred)
    m_write.assert_called_with()
    m_exec.assert_called_with()
    m_delete.assert_called_with()

  def test_main_without_args(self):
    saved_stdout = sys.stdout
    try:
      out = StringIO()
      sys.stdout = out

      # Invoke
      with self.assertRaises(SystemExit) as cm:
        ark_puppet_apply.run_main([])
      self.assertEquals(cm.exception.code, 1)

      output = out.getvalue().strip()
    finally:
      sys.stdout = saved_stdout

    self.assertEquals(output, "Syntax: ark-puppet-apply [s3 url to the <puppet manifest>.tar.gz]")

  @patch('arklib.cli.ark_puppet_apply.ark_cred')
  def test_main_with_exception(self, m_cred):
    _cred = Mock(side_effect = Exception)
    m_cred.return_value = _cred

    with self.assertRaises(SystemExit) as cm:
      ark_puppet_apply.run_main(['name', 's3://bucket/path/puppet.tar.gz'])

    self.assertEquals(cm.exception.code, 1)

